#!/usr/bin/env python
import os
import sys
import json
import logging
import textwrap
from predictions import generate_mut
from split import split_parameters_file
from aggregation import aggregate_result_file
import argparse
# logging config
logging.basicConfig(level=logging.WARNING, format='%(asctime)s,%(msecs)d %(levelname)-8s [%(filename)s:%(lineno)d] %(message)s', datefmt='%Y-%m-%d:%H:%M:%S',)

script_dir = os.path.dirname(os.path.realpath(__file__))

# _version = '0.1'

def generate_random_str(length):
    return ''.join(random.sample(string.digits+string.ascii_letters, length))


def eprint(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)


def read_peptides(fname):
    with open(fname, 'r') as r_file:
        peptides = r_file.readlines()
        peptides = [row.strip() for row in peptides if row.strip()]
        return peptides


def get_usage_info():
    # read in the example_commands.txt file and return it as a string
    f = open(os.path.join(script_dir, 'example_commands.txt'), 'r')
    lines = f.readlines()
    return "".join(lines)


def main():
    import select

    try:
        parser = argparse.ArgumentParser(
            prog='mutgen',
            description='mutated peptide generator',
            epilog='version: 0.8-beta',
            usage=textwrap.dedent(get_usage_info())
        )

        parser.add_argument("--split",
                            action="store_true",
                            dest="split_parameters_flag",
                            default=False,
                            help="flag to indicate the action we want to take with the standalone: split parameters into JSON files")

        parser.add_argument("--split-dir",
                            dest="split_parameters_dir",
                            default='',
                            help="the diretory for the JSON files that input parameters splitted into")

        parser.add_argument("--split-inputs-dir",
                            dest="split_inputs_dir",
                            default=None,
                            help="the diretory for the sequence and peptide files that input sequences splitted into")

        parser.add_argument("--aggregate",
                            action="store_true",
                            dest="aggregate_parameters_flag",
                            default=False,
                            help="flag to indicate the action to aggregate the results")

        parser.add_argument("--job-desc-file",
                            dest="job_desc_file",
                            default='',
                            help="the file path for the job description")

        parser.add_argument("--aggregate-input-dir",
                            dest="aggregate_input_dir",
                            default='',
                            help="the diretory for the JSON files which have input parameters")

        parser.add_argument("--aggregate-result-dir",
                            dest="aggregate_result_dir",
                            default='',
                            help="the path for the JSON file contains results need to be aggregated as well as the place we place the final result file")

        parser.add_argument("--batch",
                            action="store_true",
                            dest="batch_flag",
                            default=False,
                            help="flag to predict for all json file in input dir")

        parser.add_argument("--input-dir", "-i", dest="input_dir",
                            help="input dir for JSON files.", metavar="INPUT_DIR")

        parser.add_argument("--run-predictor",
                            action="store_true",
                            dest="predictor_flag",
                            default=True,
                            help="flag to indicate the action we want to take with the standalone: run predictor with atom unit")

        parser.add_argument("--assume-valid",
                            action="store_true",
                            dest="assume_valid_flag",
                            default=False,
                            help="flag to indicate skiping validation")

        parser.add_argument("-j", dest="json_filename",
                            help="FILE containing all parameters.", metavar="JSON_FILE")

        parser.add_argument("--output-format", "-f", dest="output_format", default="tsv",
                            help="prediction result output format.", metavar="OUTPUT_FORMAT")

        parser.add_argument("--output-directory",
                            dest="output_directory",
                            help="the directory for the mpg tool generated output files")

        options = parser.parse_args()

        if options.aggregate_parameters_flag:
            print(aggregate_result_file(options.job_desc_file, options.aggregate_input_dir, options.aggregate_result_dir))
            exit(0)

        if not (options.json_filename or options.filename_peptide or options.download_fasta_url):
            get_usage_info()
            exit(0)

        if options.split_parameters_flag:
            split_parameters_file(options.json_filename, options.split_parameters_dir, options.split_inputs_dir, options.assume_valid_flag)
            exit(0)


        commandline_input_prediction(options)

    except Exception as e:
        print(str(e), file=sys.stderr)
        exit(1)

def read_json_file(file_path):
    with open(file_path, 'r') as r_file:
        return json.load(r_file)

def eprint(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)

def truncate_file(output_path):
    with open(output_path, 'a+') as tsv_file:
        tsv_file.truncate(0)

def print_result(result):
    table_rows = result
    for row in table_rows:
        print( '\t'.join(map(str, row)))

def save_tsv(result, output_path):
    table_rows = result
    with open(output_path, 'a') as tsv_file:
        tsv_file.write( '\n'.join(['\t'.join(map(str, row)) for row in table_rows]))
        tsv_file.write( '\n')

def save_json(result, output_path):
    output_dir = os.path.dirname(output_path)
    if output_dir:
        os.makedirs(output_dir, exist_ok=True)
    with open(output_path, 'w') as w_file:
        json.dump(result, w_file, indent=2)


def commandline_input_prediction(options):
    """ This version takes a file containing an peptide sequences as input."""

    additional_result_info = {}
    warnings = []
    additional_result_info['warnings'] = warnings
    additional_result_info['results'] = []

    if options.json_filename:
        with open(options.json_filename, 'r') as r_file:
            try:
                input_data = json.load(r_file)
            except:
                eprint("This input file does not appear to be JSON format. Please check the help info with command: python3 src/run_mutgen.py -h")
                return
            
        if not options.output_directory:
            raise ValueError('The output-directory is required and cannot be None.')

        output_dir = options.output_directory
        output_json = os.path.join(output_dir, 'result.json')
        output_data = generate_mut(output_dir=output_dir, **input_data)
        # print out errors and exit
        if output_data.get("errors", None):
            eprint("Validation errors:")
            for error in output_data.get("errors"):
                eprint(f"- {error}")
            additional_result_info["errors"] = output_data.get("errors")
        if output_data.get("warnings", None):
            warnings.extend(output_data.get("warnings"))
        results = output_data.get("results")
        for i in range(len(results)):
            result = results[i]
            result['table_columns'] = ["mutgen."+ column_name for column_name in result['table_columns']]
            if "warnings" in result:
                additional_result_info['warnings'].extend(result.pop('warnings'))
            additional_result_info["results"].append(result)

        save_json(additional_result_info, output_json)


if __name__=='__main__':
    main()
