import sys
import os
import warnings
import argparse
import textwrap
from os.path import isfile, getsize
from ArgumentParserBuilder import NGArgumentParserBuilder
from database_functions import Database, SQLiteDatabase



class ValidatePath(argparse.Action):
    '''
    This will only get called when user sets the flag.
    It will check to see if the path points to a valid SQLite DB.
    '''
    def __call__(self, parser, namespace, values, option_string=None):
        print("Calling ValidatePath")
        
        if self.isSQLite3(values):
            # Set the result to an attribute in the namespace
            # >> setattr(namespace, self.dest, result)
            setattr(namespace, self.dest, values)
            return
        
        warnings.warn('Failed to connect to PepX SQLite Database. Please verify the path to the SQLite DB.', stacklevel=2)
        # sys.exit(0)

        # This is for sake of testing
        setattr(namespace, self.dest, values)

    
    def isSQLite3(self, filename):
        if not isfile(filename):
            return False
        
        # SQLite database file header is 100 bytes
        if getsize(filename)< 100: 
            return False

        with open(filename, 'rb') as fd:
            header = fd.read(100)
        
        return header[:16].decode('utf-8') == 'SQLite format 3\x00'


class PepXArgumentParser(NGArgumentParserBuilder):
    def __init__(self):
        super().__init__()

        # Program description
        self.description = "this is a test."
        self.name = "sample"
        self.version = '2.0'

        # Add arguments
        self.sequences()

        # Available tool options:
        # 'mhci', 'mhcii', 'mhcnp', ...
        # self.alleles('mhci')

        self._pepx_database = ''


        # Set PepX Database
        # dest -> db_path by default.
        self.database_path(
            single_dash_option='p',
            double_dash_option='database_path',
            help='Sets PepX Database.',
            action=ValidatePath,
        )

        self.set_db_path()

        # Set other arguments
        self.quantitation_level()
        self.datasource()
        self.dataset_id()
        # self.list_dataset()
        
    ###########################################################
    # Create Optional Argument specifically for PepX
    ###########################################################
    def set_db_path(self):
        db_path = ''
        
        # NOTE: Is this the best way to grab argument?
        args = sys.argv[1:]
        if ('-p' in args) or ('--database_path' in args):
            try:
                db_path_idx = args.index('-p')
            except:
                db_path_idx = args.index('--database_path')
            db_path = args[db_path_idx + 1]


        if db_path:
            self._pepx_database = SQLiteDatabase(path=db_path)
        else:
            warnings.warn("Database path was not set. By default, PepX-prod database will be used.", stacklevel=2)

            try :
                user = os.getenv('PEPX_DB_USER', '')
                password = os.getenv('PEPX_DB_PWD', '')
                self._pepx_database = Database(password=password, user=user)
                print("Successfully connected to the database.")
            except :
                print('Failed to connect PepX-prod database. Please check to make sure the username and password is set correctly.')
                sys.exit(0)


    def quantitation_level(self) -> None:
        self.parser.add_argument(
            '-q',
            '--quant_level',  
            dest = 'quant_level', 
            required = True,
            nargs = 1,
            type = str.lower,
            choices = ['gene', 'transcript'],
            help = 'Quantitation level.'
        )


    def datasource(self) -> None:
        available_datasources = []
        help_text = ''
        
        # Utilize parser.parse_known_args()
        if 'gene' in sys.argv:
            available_datasources = ['Abelin', 'CCLE', 'GTEX', 'HELA', 'TCGA']
            help_text = '''\
            available datasource when quantitation level is set to 'gene'
            : 'Abelin', 'CCLE', 'GTEX', 'HELA', 'TCGA'
            '''
        elif 'transcript' in sys.argv:
            available_datasources = ['CCLE', 'GTEX', 'HPA','TCGA']
            help_text = '''\
            available datasource when quantitation level is set to 'transcript'
            : 'CCLE', 'GTEX', 'HPA','TCGA'
            '''
        else:
            help_text =  '''\
            available datasource when quantitation level is set to 'gene'
            : 'Abelin', 'CCLE', 'GTEX', 'HELA', 'TCGA'\n
            available datasource when quantitation level is set to 'transcript'
            : 'CCLE', 'GTEX', 'HPA','TCGA'
            '''

        self.parser.add_argument(
            '-s',
            '--datasource', 
            dest = 'datasource', 
            required = True,
            nargs = 1,
            type = str,
            choices=available_datasources,
            help = textwrap.dedent(help_text))
        
    def dataset(self) -> None:
        pass

    # def dataset(self, args) :
    #     user_selected_dataset_id = args.dataset_id[0]
    #     valid_datasets = self._pepx_database.get_datasets(data_source=self._datasource, quantification=self._qlevel)
    #     valid_datasets = json.loads(valid_datasets)
    #     is_valid_dataset = False
    #     target_dataset = None

    #     for valid_dataset in valid_datasets :
    #         if user_selected_dataset_id == str(valid_dataset['dataset_id']):
    #             is_valid_dataset = True
    #             target_dataset = valid_dataset
    #             break
            
    #     if not is_valid_dataset :
    #         raise self.parser.error(textwrap.dedent('''
    #                             %s is not a valid dataset id. 
    #                             Please check the available dataset id using the --list_datasets/-l.
    #                              ''' %(user_selected_dataset_id)))
        
    #     self._dataset = target_dataset


        
    def dataset_id(self) -> None:
        self.parser.add_argument(
            '-d', 
            '--dataset_id',
            dest = 'dataset_id', 
            required = True,
            nargs = 1,
            type = str,
            help = textwrap.dedent('''
            use the following command to search for available dataset
            (NOTE: quantitation level and datasource name is required)
            : %(prog)s -q gene -s CCLE --list_datasets/-l
            '''))