# NXGArgumentParserBuilder

This is a builder class (Abstract Base Class) for CLI tools. This class will enforce certain properties or abstract methods to be implemented to properly create an Argument Parser class for other CLI tools.


## Create a class from the builder class
Let's create an example class that extends from this builder class.
```python
class ExampleArgumentParser(NXGArgumentParserBuilder):
    def __init__(self):
        super().__init__()
```

To fully create an Argument Parser class using `NXGArgumentParserBuilder`, a couple of elements must be implemented:

1. `description` : abstract property
2. `parse_nxg_arguments`: abstract method

Implement the `description` property in the example class:
```python
class ExampleArgumentParser(NXGArgumentParserBuilder):
    .
    .
    @property
    def description(self) -> str:
        ''' Write the description of the argument parser '''
        return 'this is an example...'
```
Implement the `parse_nxg_arguments` in the example class:
```python
class ExampleArgumentParser(NXGArgumentParserBuilder):
    .
    .
    def parse_nxg_arguments(self, args):
        super().parse_nxg_arguments(args)
        return self.parser.parse_args(args)
```

## Usage
After creating custom argument parser class,
```python
parser = ExampleArgumentParser()
parser.add_info()
parser.add_argument(
    '--quantitation_level', 
    '-q', 
    dest = 'quant_level', 
    required = True,
    nargs = 1,
    type = str,
    choices = ['gene', 'transcript'],
    default = 'gene',
    help = 'Quantitation level.'
    )
args = parser.parse_nxg_arguments(sys.argv[1:])
```

**Second line:** Add `description` and `formatter_class` to the parser.

**Third line:** The `add_argument()` will add argument to the parser. 

> __NOTE__ :
> 
> In addition, it will also <u>create attributes</u> based on the `dest` keyword. `NXGArgumentParser` enforces the `dest` keyword to be specified when adding an argument.

**Last line:** The `parse_nxg_arguments` method will parse all the arguments into a `Namespace`.

> __NOTE__ :
>
> In addition, it will check to make sure that all validation functions for each attributes have been created.
>
> For example, if `quant_level` is created using the `add_argument()`, this function will check to see if a validation function called `validate_quant_level()` is implemented.
>
> All validation methods will have `validate_` prefix to their function name.

If the validation method is not created in the example class when `parse_nxg_arguments()` is called, it will throw an error:
```bash
NotImplementedError: Please implment following validation methods:
        * validate_quant_level
```