import os
import sys
import json
import unittest
import pandas as pd

def find_root_dir(start_dir=None, anchor_files=None):
    """Find the app root directory by looking for known anchor files."""
    if start_dir is None:
        start_dir = os.getcwd()  # Default to the current working directory
    
    if anchor_files is None:
        anchor_files = ['LICENSE']
    
    # Normalize to absolute path
    current_dir = os.path.abspath(start_dir)
    
    # Traverse up the directory tree until an anchor file is found or root is reached
    while current_dir != os.path.dirname(current_dir):
        # Check for anchor files in the current directory
        for anchor in anchor_files:
            if os.path.isfile(os.path.join(current_dir, anchor)) or os.path.isdir(os.path.join(current_dir, anchor)):
                return current_dir  # Return the directory where the anchor file is found

        # Check for anchor files in child directories
        for root, dirs, files in os.walk(current_dir):
            for anchor in anchor_files:
                if anchor in dirs or anchor in files:
                    return root  # Return the child directory where the anchor file is found

        current_dir = os.path.dirname(current_dir)  # Move up one level
    
    # If no anchor file is found, return None or raise an error
    return None

# Add pepx-database-interface to the path
APP_ROOT_DIR = find_root_dir(anchor_files=['license-LJI.txt'])
PEPX_DB_INTF_PATH = f'{APP_ROOT_DIR}/lib/pepx-database-interface'
sys.path += [PEPX_DB_INTF_PATH]

from pepx_database_interface.database_functions import Database, SQLiteDatabase

pd.set_option('expand_frame_repr', False)
pd.set_option('display.max_columns', 999)


class TestDatabases(unittest.TestCase):
    pepx_pwd = os.getenv('PEPX_DB_PWD')
    pepx_user = os.getenv('PEPX_DB_USER')
    database = Database(password=pepx_pwd, user=pepx_user)
    sqlite_db_path = os.getenv('PEPX_SQLITE_DB', '')
    if not sqlite_db_path:
        raise ValueError('Please provide path to the PepX SQLite Database by setting environment variable PEPX_SQLITE_DB.')
    sqlite_db = SQLiteDatabase(path=sqlite_db_path)

    def test_get_data_sources(self):
        # Results from original Database
        ds = self.database.get_data_sources()
        ds_gene = self.database.get_data_sources('gene')
        ds_transcript = self.database.get_data_sources('transcript')

        # Results from SQLite Database
        sqlite_ds = self.sqlite_db.get_data_sources()
        sqlite_ds_gene = self.sqlite_db.get_data_sources('gene')
        sqlite_ds_transcript = self.sqlite_db.get_data_sources('transcript')

        self.assertListEqual(ds, sqlite_ds)
        self.assertListEqual(ds_gene, sqlite_ds_gene)
        self.assertListEqual(ds_transcript, sqlite_ds_transcript)

    def test_get_datasets(self):
        ''' Empty param test '''
        datasets = self.database.get_datasets()
        datasets = json.loads(datasets)
        sqlite_datasets = self.sqlite_db.get_datasets()
        sqlite_datasets = json.loads(sqlite_datasets)
        self.assertListEqual(datasets, sqlite_datasets)

        ''' Datasource pararm only '''
        datasets = self.database.get_datasets('Abelin', None)
        datasets = json.loads(datasets)
        sqlite_datasets = self.sqlite_db.get_datasets('Abelin', None)
        sqlite_datasets = json.loads(sqlite_datasets)
        self.assertListEqual(datasets, sqlite_datasets)

        ''' Quantitation level param only'''
        datasets = self.database.get_datasets(None, 'transcript')
        datasets = json.loads(datasets)
        sqlite_datasets = self.sqlite_db.get_datasets(None, 'transcript')
        sqlite_datasets = json.loads(sqlite_datasets)
        self.assertListEqual(datasets, sqlite_datasets)

        ''' Datasource with gene quantitation level '''
        datasets = self.database.get_datasets('HPA', 'gene')
        datasets = json.loads(datasets)
        sqlite_datasets = self.sqlite_db.get_datasets('HPA', 'gene')
        sqlite_datasets = json.loads(sqlite_datasets)
        self.assertListEqual(datasets, sqlite_datasets)

        ''' Datasource with transcript quantitation level '''
        datasets = self.database.get_datasets('TCGA', 'transcript')
        datasets = json.loads(datasets)
        sqlite_datasets = self.sqlite_db.get_datasets('TCGA', 'transcript')
        sqlite_datasets = json.loads(sqlite_datasets)
        self.assertListEqual(datasets, sqlite_datasets)



    def test_peptide_lookup(self):
        '''
        NOTE: As of 06/27/2023, there's a slight difference in numbers due to rounding issues.
              We will have to come back to this later.
        '''
        peptide_dictionary = self.database.peptide_lookup(['ALLVRYTKKVPQVS','APQVSTPTLVEAAR'], 1353, 'transcript')
        # print(peptide_dictionary)
        # print(type(peptide_dictionary))
        sqlite_peptide_dictionary = self.sqlite_db.peptide_lookup(['ALLVRYTKKVPQVS','APQVSTPTLVEAAR'], 1353, 'transcript')
        # print(sqlite_peptide_dictionary)
        # print(type(sqlite_peptide_dictionary))

        # print(peptide_dictionary.keys())
        # print(sqlite_peptide_dictionary.keys())

        # print(type(peptide_dictionary['collapsed']))
        
        # self.assertListEqual(list(peptide_dictionary['collapsed'].columns), list(sqlite_peptide_dictionary['collapsed'].columns))

        collapsed_df_1 = peptide_dictionary['collapsed']
        collapsed_df_2 = sqlite_peptide_dictionary['collapsed']

        # collapsed_df_1 = collapsed_df_1.drop(collapsed_df_1.index[0])
        # collapsed_df_2 = collapsed_df_2.drop(collapsed_df_2.index[0])

        # Lose dataframe header
        df_dict1 = dict.fromkeys(collapsed_df_1.columns, '')
        collapsed_df_1.rename(columns = df_dict1, inplace=True)
        df_dict2 = dict.fromkeys(collapsed_df_2.columns, '')
        collapsed_df_2.rename(columns = df_dict2, inplace=True)

        # print(collapsed_df_1.to_string())
        # print(collapsed_df_2.to_string())

        # print(peptide_dictionary['collapsed'].to_string())
        # print(sqlite_peptide_dictionary['collapsed'].to_string())

        # assert_frame_equal(collapsed_df_1, collapsed_df_2)
        # assert_frame_equal(peptide_dictionary['collapsed'], sqlite_peptide_dictionary['collapsed'])


if __name__=='__main__':
    unittest.main()