import os
import pickle
import pandas as pd
from copy import deepcopy
from pathlib import Path
DATA_DIR = str(Path(__file__).resolve().parents[0]) + "/data"


def read_pickle_file(file_name):
    with open(file_name, 'rb') as r_file:
        data = pickle.load(r_file)
    return data


class Allele_Validator:
    # ==============================================================================================================
    # Constructor
    # ==============================================================================================================
    def __init__(self):
        mro_pickle_file = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'pickles/tools_mapping_info.p')
        molecule_pickle_file = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'pickles/molecule_info.p')
        autocomp_pickle_file = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'pickles/autocomp_data_info.p')
        self.data = {"molecule_data": None, "mro_data": None, "autocomp_data": None}
        
        # Read mro file
        if mro_pickle_file:
            self.data["mro_data"] = read_pickle_file(mro_pickle_file)
        else :
            raise ValueError("Unable to locate the 'tools_mapping_info.p' file.")

        # Read molecule file
        if molecule_pickle_file:
            self.data["molecule_data"] = read_pickle_file(molecule_pickle_file)
            
            # Additional data required for allele autosuggestion.
            labels = self.data["molecule_data"]["IEDB Label"].squeeze().dropna().unique().tolist()
            self.data["molecule_data"]["Synonyms"] = self.data["molecule_data"]["Synonyms"].fillna("")
            
            synonyms = [_ for syn in self.data["molecule_data"]["Synonyms"].squeeze().dropna().unique().tolist() for _ in syn.split("|")]
            self.data["allele_names"] = labels + synonyms
            self.data["last_label"] = labels[-1]
        else :
            raise ValueError("Unable to locate the 'molecule_info.p' file.")

        # Read autocomplete datasource file
        if autocomp_pickle_file :
            self.data["autocomp_data"] = read_pickle_file(autocomp_pickle_file)


    # ==============================================================================================================
    # Helper Functions
    # ==============================================================================================================
    def is_single_data(self, allele):
        if isinstance(allele, str):
            return True
        elif isinstance(allele, list):
            return False
        else:
            raise TypeError(
                "Please provide string type for single allele, and list of strings for multiple alleles.")

    def compare_lengths(self, sample_lengths, original_lengths):
        valid_lengths = []
        invalid_lengths = []

        for length in sample_lengths :
            if length in original_lengths :
                valid_lengths.append(length)
            else :
                invalid_lengths.append(length)
        
        return valid_lengths, invalid_lengths


    # ==============================================================================================================
    # Allele-Valdiator Methods
    # ==============================================================================================================
    def validate_method(self, method, version=None, tools=None):
        '''This will validate the names of method and its version'''
        mro_df = self.data["mro_data"]
        method_name = ""
        default_version = ""

        mhci_method_df = pd.read_excel(r'{}/method-table.xlsx'.format(DATA_DIR), sheet_name="MHCI", engine='openpyxl')
        mhcii_method_df = pd.read_excel(r'{}/method-table.xlsx'.format(DATA_DIR), sheet_name="MHCII", engine='openpyxl')

        # if tools == "mhci":
        #     method_names = list(mhci_method_df["method"].unique())
        # else:
        #     method_names = list(mro_df["Tool"].unique())
        
        mhci_method_names = list(mhci_method_df["method"].unique())
        mhcii_method_names = list(mhcii_method_df["method"].unique())

        # Preprocess method and version parameters
        if method:
            method_name = method.strip().lower()
            # if version: method_name = method_name + "-" + version.strip()

        all_methods = mhci_method_names + mhcii_method_names
        if method_name not in all_methods:
            raise ValueError(
                "Provided method name and version ({}) is not valid.".format(method_name))
        
        if version and (method_name in mhci_method_names):
            default_method_row = mhci_method_df[(mhci_method_df['method']==method_name) & 
                                                (mhci_method_df['default_version']==True)]
            default_version = str(list(default_method_row['version'])[0])
            if version != default_version :
                raise ValueError(
                    "Please provide a valid version number for {}. Current default version is {}."
                    .format(method_name, default_version))

        if version and (method_name in mhcii_method_names):
            default_method_row = mhcii_method_df[(mhcii_method_df['method']==method_name) & 
                                                (mhcii_method_df['default_version']==True)]
            default_version = str(list(default_method_row['version'])[0])
            if version != default_version :
                raise ValueError(
                    "Please provide a valid version number for {}. Current default version is {}."
                    .format(method_name, default_version))
            
        return method_name

    def validate_tools_group(self, tools_group):
        mro_df = self.data["mro_data"]
        tools_group_name = tools_group

        # Validate tool's name
        tools_group_ls = list(mro_df["Tool Group"].unique())
        # TODO: This line is for only debugging purpose
        tools_group_ls = ["mhci", "mhcii"]

        if tools_group:
            tools_group_name = tools_group.lower()

        if tools_group_name not in tools_group_ls:
            raise ValueError(
                "Provided tools_group name ({}) is not a valid name".format(tools_group))

        return tools_group_name

    def filter_dataframe(self, df, method=None, version=None, tools_group=None):
        tools_group_name = ''
        method_name = ''

        # Validate tools_name
        if tools_group:
            tools_group_name = self.validate_tools_group(tools_group)

        # Validate method and version
        if method:
            method_name = self.validate_method(method, version)

        # Original Tools_MRO_mapping file has 'netmhcpan_el' and 'netmhcpan_ba' to both refer to 'netmhcpan'
        if 'netmhcpan' in method_name :
            method_name = 'netmhcpan'

        # Filter dataframe by tools and method
        if tools_group_name:
            df = df[df["Tool Group"] == tools_group_name]

        if method_name:
            df = df[df["Tool"] == method_name]

        if version :
            df = df[df["Tool Version"] == float(version)]

        return df.reset_index(drop=True)


    def validate_alleles(self, iedb_label, method=None, version=None, tools_group=None):
        '''======================================================================================

        Description :\n
          Given allele (IEDB label), it will list out valid dictionary (containing valid alleles/lengths)
          and invalid dictionary (containing invalid allele/lengths combination).

        Parameters :
          - iedb_label : list
            List of strings containing "IEDB Label" that suppose to match "IEDB Label"
            column from "mro-dev.tsv".
          - method : str
            Name of the method (ex. "netmhcpan" or "netmhcpan-4.1")
          - version : str
            Version of the tool (ex. "4.1")
          - tool_group : str
            Tools group name. (ex. "mhci", "mhcii", etc.)

        Return Value : dictionaries\n
          Valid and Invalid dictionaries

        ======================================================================================'''
        mro_df = self.data["mro_data"]
        header = list(mro_df.columns)
        validity = []

        # Basic paremeter check + filters dataframe according to the parameter
        mro_df = self.filter_dataframe(mro_df, method, version, tools_group)

        # Extract 'IEDB Label' as series
        iedb_label_idx = header.index("IEDB Label")
        mro_iedb_series = mro_df.iloc[:, iedb_label_idx]

        # If iedb_label is a single data (str)
        if self.is_single_data(iedb_label):
            if iedb_label in mro_iedb_series.values:
                return True
            else:
                return False

        # At this point, iedb_label is multiple values
        for each_allele in iedb_label:
            if each_allele in mro_iedb_series.values:
                validity.append(True)
            else:
                validity.append(False)
                
        return validity

    def validate_allele_lengths(self, iedb_label, lengths, method=None, version=None, tools_group=None):
        '''======================================================================================

        Description :\n
          Given allele (IEDB label), it will listout valid dictionary (containing valid alleles/lengths)
          and invalid dictionary (containing invalid allele/lengths combination).

        Parameters :
          - iedb_label : list
            List of strings containing "IEDB Label" that suppose to match "IEDB Label"
            column from "mro-dev.tsv".
          - lengths : [str]
            List of integers.
          - method : str
            Name of the method (ex. "netmhcpan" or "netmhcpan-4.1")
          - version : str
            Version of the tool (ex. "4.1")
          - tool_group : str
            Tools group name. (ex. "mhci", "mhcii", etc.)
          

        Return Value : dictionaries\n
          Valid and Invalid dictionaries

        ======================================================================================'''

        orig_mro_df = self.data["mro_data"]
        mro_df = deepcopy(orig_mro_df)
        header = list(mro_df.columns)
        valid_dict = {}
        invalid_dict = {}

        # Basic paremeter check + filters dataframe according to the parameter
        mro_df = self.filter_dataframe(mro_df, method, version, tools_group)

        # Extract 'IEDB Label' as series
        iedb_label_idx = header.index("IEDB Label")
        mro_iedb_series = mro_df.iloc[:, iedb_label_idx]

        # single data
        if self.is_single_data(iedb_label):
            if iedb_label in mro_iedb_series.values:
                # Get index of the allele
                target_idx = mro_df["IEDB Label"].loc[lambda x: x == iedb_label].index[0]

                # Get the correlated lengths
                target_lengths = mro_df["Lengths"][target_idx]
                target_lengths = [str(length) for length in target_lengths.split(",")]
                    
                valid_lengths, invalid_lengths = self.compare_lengths(lengths, target_lengths)
                
                if valid_lengths:
                    valid_dict[iedb_label] = valid_lengths
                if invalid_lengths:
                    invalid_dict[iedb_label] = invalid_lengths
                return valid_dict, invalid_dict
            else :
                invalid_dict[iedb_label] = lengths
                # raise ValueError("Please provide a correct IEDB Label.")

        # if iedb_label not in mro_iedb_series.values : return False
        for each_allele in iedb_label:
            if each_allele in mro_iedb_series.values:
                # Get index of the allele
                target_idx = mro_df["IEDB Label"].loc[lambda x: x == each_allele].index[0]

                # Get the correlated lengths
                target_lengths = mro_df["Lengths"][target_idx]
                target_lengths = [str(length) for length in target_lengths.split(",")]

                valid_lengths, invalid_lengths = self.compare_lengths(lengths, target_lengths)
                if valid_lengths:
                    valid_dict[each_allele] = valid_lengths
                if invalid_lengths:
                    invalid_dict[each_allele] = invalid_lengths
            else :
                invalid_dict[each_allele] = lengths
                # raise ValueError("Please provide a correct IEDB Label.")
                
        return valid_dict, invalid_dict


    def get_alleles(self, method=None, version=None, tools_group=None):
        '''======================================================================================
        
        Description :\n
          Given method or tools group, it will retrieve all available alleles (IEDB Label).

        Parameters :
          - tool_group : str
            Tools group name. (ex. "mhci", "mhcii", etc.)
          - method : str
            Name of the method (ex. "netmhcpan" or "netmhcpan-4.1")
          - version : str
            Version of the tool (ex. "4.1")
            
        Return Value : [str]\n
          List of allele names as strings.

        ======================================================================================'''
        mro_df = self.data["mro_data"]

        # Return all allele names if no parameters are given
        if not (method or tools_group) :
            alleles = list(mro_df["IEDB Label"].unique())
            return [_ for _ in alleles if str(_) != "nan"]

        # Basic paremeter check + filters dataframe according to the parameter
        mro_df = self.filter_dataframe(mro_df, method, version, tools_group)

        alleles = list(mro_df["IEDB Label"])
        alleles = [_ for _ in alleles if str(_) != "nan"]

        return alleles

    def convert_mroid_to_methodlabel(self, mro_ids, method, version=None, tools_group=None):
        '''======================================================================================
        
        Description :\n
          Given MRO ID and method, return the tools label.

        Parameters :
          mro_id : list of str
            List of MRO IDs from "mr-dev.tsv"
          method : str
            Name of the method (ex. "netmhcpan" or "netmhcpan-4.1")
          version : str
            Version of the tool (ex. "4.1")
          tool_group : str
            Tools group name. (ex. "mhci", "mhcii", etc.)
            
        Return Value : [str]\n
          List of strings (which are alleles ordered by the input) || list of None

        ======================================================================================'''
        # Make sure either mro_id or iedb_label is provided.
        if not mro_ids :
            raise ValueError("Please provide valid IEDB labels (ex. BoLA-1:02301).")

        mro_df = self.data["mro_data"]
        
        # Basic paremeter check + filters dataframe according to the parameter
        mro_df = self.filter_dataframe(mro_df, method, version, tools_group)

        ''' filter by mro_df by MRO ID / IEDB Label '''
        if self.is_single_data(mro_ids) :
            methodlabel = None
            idx = mro_df.index[mro_df["MRO ID"] == mro_ids].tolist()
            
            if len(idx) != 0 :
                methodlabel = mro_df.loc[idx[0]]["Tool Label"]

        else :
            methodlabel = []
            for each_idlabel in mro_ids :
                idx = mro_df.index[mro_df["MRO ID"] == each_idlabel].tolist()
                
                if len(idx) != 0 :
                    label = mro_df.loc[idx[0]]["Tool Label"]
                    methodlabel.append(label)
                else :
                    methodlabel.append(None)

        return methodlabel

    def convert_methodlabel_to_iedblabel(self, method_labels, method, version=None, tools_group=None):
        '''======================================================================================
        
        Description :\n
          Given tool label, it will convert it to IEDB label.

        Parameters :
          method_labels : str or [str]
            String or list of Tool Label from "mro-dev.tsv"
          method : str
            Name of the method (ex. "netmhcpan" or "netmhcpan-4.1")
          version : str
            Version of the tool (ex. "4.1")
          tool_group : str
            Tools group name. (ex. "mhci", "mhcii", etc.)
            
        Return Value : str or [str]\n
          Single tool label will return string of corresponding IEDB Label.
          Single or multiple tool labels will return list of IEDB Labels. All unmatched ones
          will return None.

        ======================================================================================'''

        # Make sure either mro_id or iedb_label is provided.
        if not method_labels :
            raise ValueError("Please provide valid IEDB labels (ex. BoLA-1:02301).")

        mro_df = self.data["mro_data"]
        
        # Basic paremeter check + filters dataframe according to the parameter
        mro_df = self.filter_dataframe(mro_df, method, version, tools_group)

        ''' filter by mro_df by MRO ID / IEDB Label '''
        if self.is_single_data(method_labels) :
            iedblabel = None
            idx = mro_df.index[mro_df["Tool Label"] == method_labels].tolist()
            
            if len(idx) != 0 :
                iedblabel = mro_df.loc[idx[0]]["IEDB Label"]

        else :
            iedblabel = []
            for each_idlabel in method_labels :
                idx = mro_df.index[mro_df["Tool Label"] == each_idlabel].tolist()
                
                if len(idx) != 0 :
                    label = mro_df.loc[idx[0]]["IEDB Label"]
                    iedblabel.append(label)
                else :
                    iedblabel.append(None)

        return iedblabel


    def convert_iedblabel_to_methodlabel(self, iedb_labels, method, version=None, tools_group=None):
        '''======================================================================================
        
        Description :\n
          Given IEDB Label and method, return the tool label.

        Parameters :
          iedb_label : str or [str]
            String or list of IEDB Label from "mro-dev.tsv"
          method : str
            Name of the method (ex. "netmhcpan" or "netmhcpan-4.1")
          version : str
            Version of the tool (ex. "4.1")
          tool_group : str
            Tools group name. (ex. "mhci", "mhcii", "mhcnp", etc.)
          
        Return Value : [str]\n
          List of strings (which are alleles ordered by the input) || list of None

        ======================================================================================'''
        # Make sure either mro_id or iedb_label is provided.
        if not iedb_labels :
            raise ValueError("Please provide valid IEDB labels (ex. BoLA-1:02301).")

        mro_df = self.data["mro_data"]
        
        # Basic paremeter check + filters dataframe according to the parameter
        mro_df = self.filter_dataframe(mro_df, method, version, tools_group)

        ''' filter by mro_df by MRO ID / IEDB Label '''
        if self.is_single_data(iedb_labels) :
            methodlabel = None
            idx = mro_df.index[mro_df["IEDB Label"] == iedb_labels].tolist()
            
            if len(idx) != 0 :
                methodlabel = mro_df.loc[idx[0]]["Tool Label"]

        else :
            methodlabel = []
            for each_idlabel in iedb_labels :
                idx = mro_df.index[mro_df["IEDB Label"] == each_idlabel].tolist()
                
                if len(idx) != 0 :
                    label = mro_df.loc[idx[0]]["Tool Label"]
                    methodlabel.append(label)
                else :
                    methodlabel.append(None)

        return methodlabel

    def convert_methodlabel_to_mroid(self, tools_label, method, version=None, tools_group=None):
        '''======================================================================================
          
          Description :\n
            Given the tool label, it will output MRO ID.

          Data Sheet :       
            "mro-dev.tsv"
          
          Parameters :
            - tool_label : str or [str]
              String or list of IEDB Label from "mro-dev.tsv"
            - method : str
              Name of the method (ex. "netmhcpan" or "netmhcpan-4.1")
            - version : str
              Version of the tool (ex. "4.1")

          Return Value : str or [str]\n
            Single tool label will return MRO ID as string.
            Single or multiple tool labels provided as list will return corresponding MRO IDs 
            as list of strings. All undetermined ones will return None.

        ======================================================================================'''
        # Make sure tools_label is provided
        if not tools_label :
            raise ValueError("Please provide valid tools labels (ex. BoLA-1:00901).")

        mro_df = self.data["mro_data"]
        
        # Basic paremeter check + filters dataframe according to the parameter
        mro_df = self.filter_dataframe(mro_df, method, version, tools_group)

        ''' filter by mro_df by MRO ID / IEDB Label '''
        if self.is_single_data(tools_label) :
            mro_ids = None
            idx = mro_df.index[mro_df["Tool Label"] == tools_label].tolist()
            
            if len(idx) != 0 :
                mro_ids = mro_df.loc[idx[0]]["MRO ID"]

        else :
            mro_ids = []
            for each_label in tools_label :
                idx = mro_df.index[mro_df["Tool Label"] == each_label].tolist()

                if len(idx) != 0 :
                    label = mro_df.loc[idx[0]]["MRO ID"]
                    mro_ids.append(label)
                else :
                    mro_ids.append(None)

        return mro_ids

    def convert_iedblabel_to_mroid(self, iedb_label, method, version=None, tools_group=None):
        '''======================================================================================
          
          Description :\n
            Given the IEDB label, it will output MRO ID.

          Data Sheet :       
            "mro-dev.tsv"
          
          Parameters :
            - iedb_label : str or [str]
              String or list of IEDB Label from "mro-dev.tsv"
            - method : str
              Name of the method (ex. "netmhcpan" or "netmhcpan-4.1")
            - version : str
              Version of the tool (ex. "4.1")

          Return Value : str or [str]\n
            Single IEDB label will return MRO ID as string.
            Single or multiple IEDB labels provided as list will return corresponding MRO IDs 
            as list of strings. All undetermined ones will return None.

        ======================================================================================'''

        # Make sure tools_label is provided
        if not iedb_label :
            raise ValueError("Please provide valid IEDB labels (ex. BoLA-1:00901).")

        mro_df = self.data["mro_data"]
        
        # Basic paremeter check + filters dataframe according to the parameter
        mro_df = self.filter_dataframe(mro_df, method, version, tools_group)

        ''' filter by mro_df by MRO ID / IEDB Label '''
        if self.is_single_data(iedb_label) :
            mro_ids = None
            idx = mro_df.index[mro_df["IEDB Label"] == iedb_label].tolist()
            
            if len(idx) != 0 :
                mro_ids = mro_df.loc[idx[0]]["MRO ID"]

        else :
            mro_ids = []
            for each_label in iedb_label :
                idx = mro_df.index[mro_df["IEDB Label"] == each_label].tolist()

                if len(idx) != 0 :
                    label = mro_df.loc[idx[0]]["MRO ID"]
                    mro_ids.append(label)
                else :
                    mro_ids.append(None)

        return mro_ids


    def convert_synonym_to_iedblabel(self, synonym, tools_group=None):
        '''======================================================================================
        
        Description :\n
          Given a synonym or list of synonyms, it will return the corresponding IEDB labels.
          If the synonym is already in IEDB label, it will simply return IEDB label.
          
          NOTE : Currently, there's no synonym, where synonym matches exactly the IEDB label.
          However, once tools label are added to the synonym, it will happen.

        Parameters :
          - synonym : str or [str] 
            Single synonym should be passed as type string.
            Else, list of strings containing synonyms.
          
        Return Value : str or [str]\n
          Single synonym will return the corresponding IEDB Label found in 'molecule-dev.tsv' as string.
          Single or multiple synonyms passed in as list will return corresponding IEDB Label as lists.
          Those that aren't able to be found, will be returned as None.

        ======================================================================================'''
        # Make sure tools_label is provided
        if not synonym :
            raise ValueError("Please provide valid synonym.")

        mol_df = self.data["molecule_data"]
        mol_header = list(mol_df.columns)
                
        # Basic paremeter check + filters dataframe according to the parameter
        mol_df = self.filter_dataframe(mol_df, tools_group=tools_group)

        ''' filter by mro_df by MRO ID / IEDB Label '''
        if self.is_single_data(synonym) :
            iedb_label = None

            # Check if there's a synonym and return it's IEDB label
            end_flag = 0
            for row in mol_df.itertuples(name=None):
                row_synonym = row[mol_header.index('Synonyms') + 1]
                
                if not row_synonym :
                    continue

                if synonym in row_synonym :
                    iedb_label = row[mol_header.index('IEDB Label') + 1]
                    end_flag = 1
                    break

            # If there's no synonym found, then try checking if the provided allele is already an IEDB Label
            if not end_flag :
              for row in mol_df.itertuples(name=None):
                  # print("synonym hasn't been found yet...")
                  row_iedb_label = row[mol_header.index('IEDB Label') + 1]

                  if synonym == row_iedb_label :
                      iedb_label = synonym

        else :
            iedb_label = []

            for each_synonym in synonym :
                corr_iedb_label = ''
                for row in mol_df.itertuples(name=None):
                    row_synonym = row[mol_header.index('Synonyms') + 1]
                    
                    if not row_synonym :
                        continue

                    if each_synonym in row_synonym :
                        corr_iedb_label = row[mol_header.index('IEDB Label') + 1]
                        break
                
                if corr_iedb_label :
                    iedb_label.append(corr_iedb_label)
                else :
                    iedb_label.append(None)

            for i in range(len(iedb_label)) :
                if iedb_label[i] != None : continue

                each_label = synonym[i]
                
                for row in mol_df.itertuples(name=None):
                    row_iedb_label = row[mol_header.index('IEDB Label') + 1]
                    
                    if each_label == row_iedb_label :
                        iedb_label[i] = each_label
                        break

        return iedb_label


    def get_available_lengths(self, iedb_label, method, version=None):
        '''======================================================================================
        
        Description :\n
          Given allele (IEDB label) and it's method, it will give out available lengths.

        Parameters :
          - iedb_label : str or [str] 
            Single allele (IEDB Label) should be passed as type string.
            Else, list of strings containing allele label (IEDB Label).
          - method : str
            Name of the method.

        Return Value : [str] or [[str]]\n
          Single allele will return list of available lengths for the given allele.
          For multiple alleles, it will return multiple lists, each corresponding to their 
          relevant alleles.

        ======================================================================================'''

        mro_df = self.data["mro_data"]

        # Basic paremeter check + filters dataframe according to the parameter
        mro_df = self.filter_dataframe(mro_df, method, version)

        if self.is_single_data(iedb_label) :
            try :            
                # Get index of the allele
                target_idx = mro_df["IEDB Label"].loc[lambda x: x==iedb_label].index[0]

                # Get the correlated lengths
                lengths = mro_df["Lengths"][target_idx]
                lengths = [str(length) for length in lengths.split(",")]
            except IndexError :
                lengths = None
        
        else :
            lengths = []
            for each_allele in iedb_label :
                try :
                    # Get index of the allele
                    target_idx = mro_df["IEDB Label"].loc[lambda x: x==each_allele].index[0]

                    # Get the correlated lengths
                    length = mro_df["Lengths"][target_idx]
                    length = [str(_) for _ in length.split(",")]
                except IndexError :
                    length = None

                lengths.append(length)
        
        return lengths

    def get_allele_info(self, iedb_label) :
        mol_df = self.data["molecule_data"]

        if self.is_single_data(iedb_label) :  
            allele_info = None

            # Get index of the allele
            target_idx = mol_df["IEDB Label"].loc[lambda x: x==iedb_label].index[0]
            target_row = mol_df.loc[[target_idx]]
            allele_info = target_row.set_index("IEDB Label").T.to_dict('dict')
        else :
            allele_info = {}
            
            for each_label in iedb_label :
                target_idx = mol_df["IEDB Label"].loc[lambda x: x==each_label].index[0]
                target_row = mol_df.loc[[target_idx]]
                one_allele_info = target_row.set_index("IEDB Label").T.to_dict('dict')

                for k, v in one_allele_info.items() :
                    allele_info[k] = v
                
        return allele_info
    
    def identify_label(self, allele, method) :
        # Use convert methods to deduce the allele's label
        converted_allele = self.convert_iedblabel_to_methodlabel(allele, method=method)
        if converted_allele:
          return "iedb_label"
        
        converted_allele = self.convert_methodlabel_to_iedblabel(allele, method=method)
        if converted_allele:
          return "tool_label"
        
        converted_allele = self.convert_synonym_to_iedblabel(allele)
        if converted_allele:
          return "synonym"
        
        return "Unknown"