from copy import deepcopy

# a map of column names to their display names, source, sort order, description, etc.

BASE_COLUMN_MAP = {
    # core
    "core.sequence_number": {
        "name": "sequence_number",
        "display_name": "seq #",
        "type": "int",
        "source": "core",
        "sort_order": 0,
        "row_sort_priority": 5,
        "default_order": "ascending",
        "description": "Index of the input sequence among all input sequences.",
        "hidden": False,
    },
    "core.sequence_name": {
        "name": "sequence_name",
        "display_name": "sequence name",
        "type": "text",
        "source": "core",
        "sort_order": 0,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Name of the input sequence as supplied by the user in FASTA, JSON, or named space-separated formats.  If no name is supplied, a serial number is assigned.",
        "hidden": False,
    },
    "core.sequence": {
        "name": "sequence",
        "display_name": "sequence",
        "type": "text",
        "source": "core",
        "sort_order": 0,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Amino acid sequence of the input sequence.",
        "hidden": False,
    },
    "core.peptide": {
        "name": "peptide",
        "display_name": "peptide",
        "type": "text",
        "source": "core",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Peptide sequence sequence",
        "hidden": False,
    },
    "core.length": {
        "name": "length",
        "display_name": "peptide length",
        "type": "int",
        "source": "core",
        "sort_order": 4,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Peptide sequence length",
        "hidden": False,
    },
    "core.start": {
        "name": "start",
        "display_name": "start",
        "type": "int",
        "source": "core",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Peptide sequence start within the context of the input sequence",
        "hidden": False,
    },
    "core.end": {
        "name": "end",
        "display_name": "end",
        "type": "int",
        "source": "core",
        "sort_order": 2,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Peptide sequence end within the context of the input sequence",
        "hidden": False,
    },
    "core.allele": {
        "name": "allele",
        "display_name": "allele",
        "type": "text",
        "source": "core",
        "sort_order": 5,
        "row_sort_priority": None,
        "default_order": None,
        "description": "MHC allele used in the prediction",
        "hidden": False,
    },
    "core.peptide_index": {
        "name": "peptide_index",
        "display_name": "peptide index",
        "type": "int",
        "source": "core",
        "sort_order": 6,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Serial number of the peptide among all peptides",
        "hidden": True,
    },
    # binding - overall
    # these parameters will be from source 'binding', not tied to a specific method
    # there will be 1 instance of these columns regardless of how many methods
    "binding.median_percentile": {
        "name": "median_percentile",
        "display_name": "median binding percentile",
        "type": "float",
        "source": "binding/elution",
        "sort_order": 2,
        "row_sort_priority": 0,
        "default_order": "ascending",
        "number_of_digits": None,
        "description": "The median percentile rank of binding/elution predictions was calculated by using the adjusted percentiles when available, otherwise the standard percentiles",
        "hidden": False,
    },
    "binding.overall_median_percentile": {
        "name": "overall_median_percentile",
        "display_name": "overall median binding percentile",
        "type": "float",
        "source": "binding/elution",
        "sort_order": 1,
        "row_sort_priority": 3,
        "default_order": "ascending",
        "number_of_digits": None,
        "description": "The overall median percentile rank for each peptide across methods and alleles was calculated by using the adjusted percentiles when available, otherwise the standard percentiles",
        "hidden": False,
    },
    "binding.global_median_percentile": {
        "name": "global_median_percentile",
        "display_name": "global median binding percentile",
        "type": "float",
        "source": "binding/elution",
        "sort_order": 1,
        "row_sort_priority": 3,
        "default_order": "ascending",
        "number_of_digits": None,
        "description": "The median of the median binding percentiles by allele for each peptide",
        "hidden": False,
    },
    # binding - method-specific
    # this set of columns (or a subset thereof) will be created for each binding
    # method that is selected
    "binding.score": {
        "name": "score",
        "display_name": "score",
        "type": "float",
        "source": "binding/elution",
        "sort_order": 0,
        "row_sort_priority": 3,
        "default_order": "descending",
        "number_of_digits": 4,
        "description": "binding/elution prediction score which indicates binding affinity",
        "hidden": False,
    },
    "binding.ic50": {
        "name": "ic50",
        "display_name": "IC50",
        "type": "float",
        "source": "binding/elution",
        "sort_order": 1,
        "row_sort_priority": 3,
        "default_order": "ascending",
        "number_of_digits": 2,
        "description": "Measured in (nM). Lower number indicates higher affinity.",
        "hidden": False,
    },
    "binding.percentile": {
        "name": "percentile",
        "display_name": "percentile",
        "type": "float",
        "source": "binding/elution",
        "sort_order": 1,
        "row_sort_priority": 3,
        "default_order": "ascending",
        "number_of_digits": None,
        "description": "The percentile rank generated by comparing the peptide's IC50/score against those of a set of random peptides from SWISSPROT database",
        "hidden": False,
    },
    "binding.adjusted_percentile": {
        "name": "adjusted_percentile",
        "display_name": "adjusted_percentile",
        "type": "float",
        "source": "binding/elution",
        "sort_order": 1,
        "row_sort_priority": 3,
        "default_order": "ascending",
        "number_of_digits": None,
        "description": "The percentile rank adjusted based on the frequency of peptide lengths",
        "hidden": False,
    },
    "binding.core": {
        "name": "core",
        "display_name": "core",
        "type": "text",
        "source": "binding/elution",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Always 9 amino acids long sequence. It's a construction used for sequence alignment and identification of binding anchors.",
        "hidden": False,
    },
    "binding.icore": {
        "name": "icore",
        "display_name": "icore",
        "type": "text",
        "source": "binding/elution",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Substring of peptide that encompasses all residues between P1 and P-omega of the MHC.",
        "hidden": False,
    },
    # consensus
    "binding.consensus_percentile": {
        "name": "percentile",
        "display_name": "consensus percentile",
        "type": "float",
        "source": "binding/elution.consensus",
        "sort_order": 2,
        "row_sort_priority": 1,
        "default_order": "ascending",
        "number_of_digits": None,
        "description": "The median percentile rank of the binding predictions 'ann', 'smm', and 'comblib_sidney2008'",
        "hidden": False,
    },
    # make the key different for mhcii
    "binding.consensus_percentile_rank": {
        "name": "percentile",
        "display_name": "consensus percentile",
        "type": "float",
        "source": "binding/elution.consensus",
        "sort_order": 2,
        "row_sort_priority": 1,
        "default_order": "ascending",
        "number_of_digits": None,
        "description": "The Consensus approach considers a combination of any three of the four methods, if available, where Sturniolo as a final choice.",
        "hidden": False,
    },
    "binding.consensus_adjusted_percentile": {
        "name": "adjusted_percentile",
        "display_name": "consensus adjusted percentile",
        "type": "float",
        "source": "binding/elution.consensus",
        "sort_order": 2,
        "row_sort_priority": 1,
        "default_order": "ascending",
        "number_of_digits": None,
        "description": "The Consensus approach considers a combination of any three of the four methods, if available, where Sturniolo as a final choice.",
        "hidden": False,
    },
    # mhcii are with mhci

    # mhciinp
    # these 4 columns are from the mhciinp method only
    #binding.mhciinp.n_motif 	binding.mhciinp.c_motif 	binding.mhciinp.cleavage_probability_score 	binding.mhciinp.cleavage_probability_percentile_rank 
    "processing.mhciinp.n_motif": {
        "name": "n_motif",
        "display_name": "MHCII-NP N motif",
        "type": "text",
        "source": "processing.mhciinp",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "the N-terminus considered for calculating the cleavage probability score",
        "hidden": False,
    },
    "processing.mhciinp.c_motif": {
        "name": "c_motif",
        "display_name": "MHCII-NP C motif",
        "type": "text",
        "source": "processing.mhciinp",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "the C-terminus considered for calculating the cleavage probability score",
        "hidden": False,
    },
    "processing.mhciinp.cleavage_probability_score": {
        "name": "cleavage_probability_score",
        "display_name": "MHCII-NP Cleavage probability score",
        "type": "float",
        "source": "processing.mhciinp",
        "sort_order": 0,
        "row_sort_priority": 3,
        "default_order": "descending",
        "number_of_digits": 4,
        "description": "The cleavage probability score indicates the probability of the peptide being an MHC II ligand based on three components: (1) the peptide length (2) the N-terminus cleavage motif and (3) the C-terminus cleavage motif. The higher the numerical value of cleavage probability score, the better is the chance of the peptide being a ligand.",
        "hidden": False,
    },
    "processing.mhciinp.cleavage_probability_percentile_rank": {
        "name": "cleavage_probability_percentile_rank",
        "display_name": "MHCII-NP Cleavage probability percentile rank",
        "type": "float",
        "source": "processing.mhciinp",
        "sort_order": 1,
        "row_sort_priority": 3,
        "default_order": "ascending",
        "number_of_digits": None,
        "description": "The predicted cleavage probability score in terms of percentile ranks per the corresponding source protein. The lower the numerical value of the percentile score, the better is the chance of the peptide being a ligand.",
        "hidden": False,
    },

    # cd4episcore
    "immunogenicity.cd4episcore.core": {
        "name": "cd4episcore_core",
        "display_name": "CD4episcore core",
        "type": "text",
        "source": "immunogenicity.cd4episcore",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "The 9amino acid peptide core used by immunogenicity Neural network model",
        "hidden": True,
    },
    "immunogenicity.cd4episcore.score": {
        "name": "score",
        "display_name": "immunogenicity score",
        "type": "float",
        "source": "immunogenicity.cd4episcore",
        "sort_order": 0,
        "row_sort_priority": 2,
        "default_order": "descending",
        "number_of_digits": 5,
        "description": "'Immunogenicity Score' ranges from 0 to 100, with low values identifying more immunogenic peptides and high values non-immunogenic peptides.",
        "hidden": False,
    },
    "immunogenicity.cd4episcore.combined_score": {
        "name": "combined_score",
        "display_name": "immunogenicity Combined Score",
        "type": "float",
        "source": "immunogenicity.cd4episcore",
        "sort_order": 0,
        "row_sort_priority": 1,
        "default_order": "descending",
        "number_of_digits": 5,
        "description": "the Combined Score is given by combining both HLA binding and the immunogenicity prediction which presumably incorporate the capacity of being recognized by TCR. And low value identifying high capacity of being recognized by TCR.",
        "hidden": False,
    },
    # immunogenicity
    "immunogenicity.score": {
        "name": "score",
        "display_name": "immunogenicity score",
        "type": "float",
        "source": "immunogenicity",
        "sort_order": 0,
        "row_sort_priority": 2,
        "default_order": "descending",
        "number_of_digits": 5,
        "description": "Scores greater than 0 indecate a higher probability of being immunogenic than non-immunogenic, while scores less than 0 indicate the opposite.  The greater the distance from 0, the higher the certainty.",
        "hidden": False,
    },
    # icerfire
    "icerfire.peptide_a_core": {
        "name": "peptide_a_core",
        "display_name": "icerfire peptide_a_core",
        "type": "text",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Always 9 amino acids long sequence. It's a construction used for sequence alignment and identification of binding anchors.",
        "hidden": True,
    },
    "icerfire.peptide_b_core": {
        "name": "peptide_b_core",
        "display_name": "icerfire peptide_b_core",
        "type": "text",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Always 9 amino acids long sequence. It's a construction used for sequence alignment and identification of binding anchors.",
        "hidden": True,
    },
    "icerfire.icore_start_pos": {
        "name": "icore_start_pos",
        "display_name": "icerfire icore_start_pos",
        "type": "int",
        "source": "icerfire",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "The starting position of the identified ICORE (the submer with the highest MHC binding potential) within the full neopeptide.",
        "hidden": False,
    },
    "icerfire.peptide_a_icore": {
        "name": "peptide_a_icore",
        "display_name": "icerfire peptide_a_icore",
        "type": "text",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Substring of peptide that encompasses all residues between P1 and P-omega of the MHC.",
        "hidden": True,
    },
    "icerfire.peptide_b_icore": {
        "name": "peptide_b_icore",
        "display_name": "icerfire peptide_b_icore",
        "type": "text",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Substring of peptide that encompasses all residues between P1 and P-omega of the MHC.",
        "hidden": True,
    },
    "icerfire.peptide_a_el_rank": {
        "name": "icerfire.peptide_a_el_rank",
        "display_name": "icerfire peptide_a_el_rank",
        "type": "float",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 2,
        "description": "The predicted eluted ligand rank (percentile) of the corresponding wild-type (peptide A) ICORE for the MHC, used to assess how different the mutant is from the self-peptide in terms of binding potential.",
        "hidden": True,
    },
    "icerfire.peptide_b_el_rank": {
        "name": "icerfire.peptide_b_el_rank",
        "display_name": "icerfire peptide_b_el_rank",
        "type": "float",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 2,
        "description": "The predicted eluted ligand rank (percentile) of the mutated (peptide B) ICORE for the MHC, based on the NetMHCpan EL score. Lower values indicate stronger binding.",
        "hidden": True,
    },
    "icerfire.icore_similarity_score": {
        "name": "icore_similarity_score",
        "display_name": "icerfire icore_similarity_score",
        "type": "float",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 4,
        "description": "The self-similarity score, representing how similar the mutated ICORE is to its wild-type counterpart. This helps assess self-tolerance, as peptides highly similar to self are less likely to trigger an immune response.",
        "hidden": True,
    },
    "icerfire.icore_blsm_mut_score": {
        "name": "icore_blsm_mut_score",
        "display_name": "icerfire icore_blsm_mut_score",
        "type": "float",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 4,
        "description": "The BLOSUM substitution matrix score, which quantifies the impact of amino acid mutations in the ICORE compared to the wild-type sequence. A higher score suggests conservative mutations, while a lower score suggests radical changes.",
        "hidden": True,
    },
    "icerfire.total_gene_tpm": {
        "name": "total_gene_tpm",
        "display_name": "icerfire total_gene_tpm",
        "type": "float",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 4,
        "description": "The expression level of the gene (in transcripts per million, TPM) encoding the neo-epitope, capturing the abundance of the antigen. Highly expressed genes may lead to more frequent presentation of the neo-epitope.",
        "hidden": False,
    },
    "icerfire.prediction": {
        "name": "prediction",
        "display_name": "icerfire prediction",
        "type": "float",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 4,
        "description": "The final immunogenicity prediction from the ICERFIRE model, indicating whether the neo-epitope is likely to trigger an immune response.",
        "hidden": False,
    },
    "icerfire.%rank": {
        "name": "percentile_rank",
        "display_name": "icerfire percentile rank",
        "type": "float",
        "source": "icerfire",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 2,
        "description": "The percentile rank of the ICERFIRE prediction. A lower percentile rank indicates a higher probability of presentation.",
        "hidden": False,
    },
    # netchop
    "processing.netchop.prediction_score": {
        "name": "prediction_score",
        "display_name": "NetChop Predictions score",
        "type": "float",
        "source": "processing.netchop",
        "sort_order": 2,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 6,
        "description": "NetChop prediction score",
        "hidden": False,
    },
    "core.amino_acid": {
        "name": "amino_acid",
        "display_name": "amino acid",
        "type": "text",
        "source": "core",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "amino acid residue",
        "hidden": False,
    },
    "core.position": {
        "name": "position",
        "display_name": "position",
        "type": "int",
        "source": "core",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Position of the amino acid residue within the input sequence",
        "hidden": False,
    },
    # netctl
    "processing.netctl.supertype": {
        "name": "supertype",
        "display_name": "NetCTL Supertype",
        "type": "text",
        "source": "processing.netctl",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Supertype of MHC allele",
        "hidden": True,
    },
    "processing.netctl.c_terminal_cleavage_affinity": {
        "name": "c_terminal_cleavage_affinity",
        "display_name": "NetCTL c terminal cleavage affinity",
        "type": "float",
        "source": "processing.netctl",
        "sort_order": 0,
        "row_sort_priority": None,
        "default_order": None,
        "default_order": None,
        "number_of_digits": 4,
        "description": " Predicted proteasomal cleavage score",
        "hidden": False,
    },
    "processing.netctl.tap_transport_efficiency": {
        "name": "tap_transport_efficiency",
        "display_name": "NetCTL TAP transport efficiency",
        "type": "float",
        "source": "processing.netctl",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 4,
        "description": "Predicted TAP transport efficiency",
        "hidden": False,
    },
    "processing.netctl.predictions_score": {
        "name": "predictions_score",
        "display_name": "NetCTL Prediction score",
        "type": "float",
        "source": "processing.netctl",
        "sort_order": 2,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 4,
        "description": "Overall prediction score",
        "hidden": False,
    },
    "processing.netctl.predicted_mhc_binding_affinity": {
        "name": "predicted_mhc_binding_affinity",
        "display_name": "NetCTL Predicted MHC binding affinity",
        "type": "float",
        "source": "processing.netctl",
        "sort_order": 4,
        "row_sort_priority": 2,
        "default_order": "descending",
        "number_of_digits": 4,
        "description": "The value is given as 1 - log50k(aff), where log50k is the logarithm with base 50.000, and aff is the affinity in nM units",
        "hidden": False,
    },
    "processing.netctl.rescale_binding_affinity": {
        "name": "rescale_binding_affinity",
        "display_name": "NetCTL Rescaled binding affinity",
        "type": "float",
        "source": "processing.netctl",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 4,
        "description": "The predicted binding affinity is normalized by the 1st percentile score",
        "hidden": False,
    },
    # netctlpan
    "processing.netctlpan.cleavage_prediction_score": {
        "name": "cleavage_prediction_score",
        "display_name": "NetCTLpan cleavage prediction score",
        "type": "float",
        "source": "processing.netctlpan",
        "sort_order": 0,
        "row_sort_priority": None,
        "default_order": None,
        "default_order": None,
        "number_of_digits": 5,
        "description": "Predicted proteasomal cleavage score",
        "hidden": False,
    },
    "processing.netctlpan.tap_prediction_score": {
        "name": "tap_prediction_score",
        "display_name": "NetCTLpan tap score",
        "type": "float",
        "source": "processing.netctlpan",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 5,
        "description": "The TAP score estimates an effective -log(IC50) values for the binding to TAP of a peptide or its N-terminal prolonged precursors.",
        "hidden": False,
    },
    "processing.netctlpan.mhc_prediction": {
        "name": "mhc_prediction",
        "display_name": "NetCTLpan MHC score",
        "type": "float",
        "source": "processing.netctlpan",
        "sort_order": 2,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 5,
        "description": "The MHC binding prediction is identical to the Class-I. And the output is -log(IC50) values.",
        "hidden": False,
    },
    "processing.netctlpan.combined_prediction_score": {
        "name": "combined_prediction_score",
        "display_name": "NetCTLpan combined score",
        "type": "float",
        "source": "processing.netctlpan",
        "sort_order": 4,
        "row_sort_priority": 2,
        "default_order": "descending",
        "number_of_digits": 5,
        "description": "This score combines the proteasomal cleavage, TAP transport and MHC binding predictions.",
        "hidden": False,
    },
    "processing.netctlpan.percentile_rank": {
        "name": "percentile_rank",
        "display_name": "NetCTLpan percentile rank",
        "type": "float",
        "source": "processing.netctlpan",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 2,
        "description": "Percentile Rank of prediction score to a set of 1000 random natural 9mer peptides",
        "hidden": False,
    },
    # mhcnp
    "processing.mhcnp.score": {
        "name": "score",
        "display_name": "MHC-NP Prob score",
        "type": "float",
        "source": "processing.mhcnp",
        "sort_order": 0,
        "row_sort_priority": 2,
        "default_order": "descending",
        "number_of_digits": 5,
        "description": "higher the score, more likely to be naturally processed",
        "hidden": False,
    },
    # basic_processing
    "processing.basic_processing.proteasome_score": {
        "name": "proteasome_score",
        "display_name": "proteasome score",
        "type": "float",
        "source": "processing.basic_processing",
        "sort_order": 0,
        "row_sort_priority": None,
        "default_order": None,
        "default_order": None,
        "number_of_digits": 2,
        "description": "The scores can be interpreted as logarithms of the total amount of cleavage site usage liberating the peptide C-terminus.",
        "hidden": False,
    },
    "processing.basic_processing.tap_score": {
        "name": "tap_score",
        "display_name": "tap score",
        "type": "float",
        "source": "processing.basic_processing",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 2,
        "description": "The TAP score estimates an effective -log(IC50) values for the binding to TAP of a peptide or its N-terminal prolonged precursors.",
        "hidden": False,
    },
    "processing.basic_processing.mhc_score": {
        "name": "mhc_score",
        "display_name": "mhc score",
        "type": "float",
        "source": "processing.basic_processing",
        "sort_order": 2,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 2,
        "description": "The -log(IC50) using the selected MHC binding method.",
        "hidden": False,
    },
    "processing.basic_processing.processing_score": {
        "name": "processing_score",
        "display_name": "processing score",
        "type": "float",
        "source": "processing.basic_processing",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "number_of_digits": 2,
        "description": "This score combines the proteasomal cleavage and TAP transport predictions.",
        "hidden": False,
    },
    "processing.basic_processing.total_score": {
        "name": "total_score",
        "display_name": "processing total score",
        "type": "float",
        "source": "processing.basic_processing",
        "sort_order": 4,
        "row_sort_priority": 2,
        "default_order": "descending",
        "number_of_digits": 2,
        "description": "Sum of the MHC and Processing scores",
        "hidden": False,
    },
    # phbr
    "phbr.#a":    { "name": "#a", "display_name":"Number of HLA-A alleles", "type": "int", "source": "phbr", "sort_order": 3, "row_sort_priority": None, "default_order": None, "description": "Number of HLA-A alleles.", "hidden": True, "display_length": 5 },
    "phbr.#b":    { "name": "#b", "display_name":"Number of HLA-B alleles", "type": "int", "source": "phbr", "sort_order": 3, "row_sort_priority": None, "default_order": None, "description": "Number of HLA-B alleles.", "hidden": True, "display_length": 5 },
    "phbr.#c":    { "name": "#c", "display_name":"Number of HLA-C alleles", "type": "int", "source": "phbr", "sort_order": 3, "row_sort_priority": None, "default_order": None, "description": "Number of HLA-C alleles.", "hidden": True, "display_length": 5 },
    "phbr.phbr":  { "name": "phbr", "display_name":"PHBR score", "type": "float", "source": "phbr", "sort_order": 1, "row_sort_priority": 0, "default_order": 'descending', "number_of_digits": 4, "description": "Harmonic mean of the lowest (best) ranked peptides across all class I or class II input alleles.  Peptides that do not overlap the mutation position(s) are removed from consideration.", "hidden": False },
    "phbr.phbr-i":  { "name": "phbr-i", "display_name":"PHBR-I", "type": "float", "source": "phbr", "sort_order": 1, "row_sort_priority": 0, "default_order": 'descending', "number_of_digits": 4, "description": "Harmonic mean of the lowest (best) ranked peptides across all class I  input alleles.  Peptides that do not overlap the mutation position(s) are removed from consideration.", "hidden": False },
    "phbr.phbr-ii":  { "name": "phbr-ii", "display_name":"PHBR-II", "type": "float", "source": "phbr", "sort_order": 1, "row_sort_priority": 0, "default_order": 'descending', "number_of_digits": 4, "description": "Harmonic mean of the lowest (best) ranked peptides across all class II input alleles.  Peptides that do not overlap the mutation position(s) are removed from consideration.", "hidden": False },
    "phbr.phbr-mhci":  { "name": "phbr-i", "display_name":"PHBR-I", "type": "float", "source": "phbr", "sort_order": 1, "row_sort_priority": 0, "default_order": 'descending', "number_of_digits": 4, "description": "Harmonic mean of the lowest (best) ranked peptides across all class I  input alleles.  Peptides that do not overlap the mutation position(s) are removed from consideration.", "hidden": False },
    "phbr.phbr-mhcii":  { "name": "phbr-ii", "display_name":"PHBR-II", "type": "float", "source": "phbr", "sort_order": 1, "row_sort_priority": 0, "default_order": 'descending', "number_of_digits": 4, "description": "Harmonic mean of the lowest (best) ranked peptides across all class II input alleles.  Peptides that do not overlap the mutation position(s) are removed from consideration.", "hidden": False },

    # pepmatch
    "pepmatch.peptide": {
        "name": "peptide",
        "display_name": "Input Sequence",
        "type": "text",
        "source": "pepmatch",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Input/query peptide sequence",
        "hidden": False,
    },
    "pepmatch.matched_sequence": {
        "name": "matched_sequence",
        "display_name": "Matched Sequence",
        "type": "text",
        "source": "pepmatch",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Peptide sequence from the selected proteome that matches the input sequence with the currently selected parameter set",
        "hidden": False,
    },
    "pepmatch.taxon_id": {
        "name": "taxon_id",
        "display_name": "Taxon ID",
        "type": "int",
        "source": "pepmatch",
        "sort_order": 4,
        "row_sort_priority": None,
        "default_order": None,
        "description": "NCBI Taxonomy ID of the proteome database",
        "hidden": True,
    },
    "pepmatch.species": {
        "name": "species",
        "display_name": "Species",
        "type": "text",
        "source": "pepmatch",
        "sort_order": 4,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Species name of the proteome database",
        "hidden": True,
    },
    "pepmatch.gene": {
        "name": "gene",
        "display_name": "Gene",
        "type": "text",
        "source": "pepmatch",
        "sort_order": 4,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Gene symbol corresponding to the matched protein in the proteome",
        "hidden": False,
    },
    "pepmatch.protein_id": {
        "name": "protein_id",
        "display_name": "Protein ID",
        "type": "text",
        "source": "pepmatch",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Unique identifier/accession of the protein containing the matched peptide",
        "hidden": False,
    },
    "pepmatch.protein_name": {
        "name": "protein_name",
        "display_name": "Protein Name",
        "type": "text",
        "source": "pepmatch",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Name of the protein containing the matched peptide",
        "hidden": False,
        "display_length": 30,
    },
    "pepmatch.mismatches": {
        "name": "mismatches",
        "display_name": "Mismatches",
        "type": "int",
        "source": "pepmatch",
        "sort_order": 1,
        "row_sort_priority": 0,
        "default_order": "ascending",
        "description": "The number of substitutions between the input peptide and the matched peptide",
        "hidden": False,
    },
    "pepmatch.mutated_positions": {
        "name": "mutated_positions",
        "display_name": "Mutated Positions",
        "type": "text",
        "source": "pepmatch",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "The positions of the mismatches between the input and matched peptides",
        "hidden": False,
    },
    "pepmatch.index_start": {
        "name": "index_start",
        "display_name": "Index start",
        "type": "int",
        "source": "pepmatch",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "The start position of the matched peptide within the full-length protein sequence",
        "hidden": True,
    },
    "pepmatch.index_end": {
        "name": "index_end",
        "display_name": "Index end",
        "type": "int",
        "source": "pepmatch",
        "sort_order": 2,
        "row_sort_priority": None,
        "default_order": None,
        "description": "The end position of the matched peptide within the full-length protein sequence",
        "hidden": True,
    },
    "pepmatch.protein_existence_level": {
        "name": "protein_existence_level",
        "display_name": "Protein Existence Level",
        "type": "int",
        "source": "pepmatch",
        "sort_order": 5,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Evidence level that the protein exists",
        "hidden": True,
    },
    # mutgen
    # verbosity_level=0
    "mutgen.variant_type": {
        "name": "variant_type",
        "display_name": "variant_type",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "snp/complex/ins/del",
        "hidden": False,
    },
    "mutgen.chr": {
        "name": "chr",
        "display_name": "chr",
        "type": "text",
        "source": "mutgen",
        "sort_order": 4,
        "row_sort_priority": None,
        "default_order": None,
        "description": "chromosome",
        "hidden": False,
    },
    "mutgen.position": {
        "name": "position",
        "display_name": "position",
        "type": "int",
        "source": "mutgen",
        "sort_order": 4,
        "row_sort_priority": None,
        "default_order": None,
        "description": "chromosomal position of mutation",
        "hidden": False,
    },
    "mutgen.ref_seq": {
        "name": "ref_seq",
        "display_name": "reference nucleotide",
        "type": "text",
        "source": "mutgen",
        "sort_order": 4,
        "row_sort_priority": None,
        "default_order": None,
        "description": "reference nucleotide",
        "hidden": False,
    },
    "mutgen.alt_seq": {
        "name": "alt_seq",
        "display_name": "mutated nucleotide",
        "type": "text",
        "source": "mutgen",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "mutated nucleotide",
        "hidden": False,
    },
    "mutgen.effect": {
        "name": "effect",
        "display_name": "mutation effect",
        "type": "text",
        "source": "mutgen",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "predicted mutation effect (e.g., missense_variant, frameshift_variant, inframe_insertion, inframe_deletion, etc.)",
        "hidden": False,
        "display_length": 30,
    },
    "mutgen.gene_name": {
        "name": "gene_name",
        "display_name": "gene name",
        "type": "text",
        "source": "mutgen",
        "sort_order": 1,
        "row_sort_priority": 0,
        "default_order": "ascending",
        "description": "HGNC gene symbol",
        "hidden": False,
    },
    "mutgen.gene_id": {
        "name": "gene_id",
        "display_name": "Ensembl gene accession",
        "type": "text",
        "source": "mutgen",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Ensembl gene identifier",
        "hidden": False,
    },
    "mutgen.protein_ref": {
        "name": "protein_ref",
        "display_name": "reference aa",
        "type": "text",
        "source": "mutgen",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "reference amino acid",
        "hidden": False,
    },
    "mutgen.protein_alt": {
        "name": "protein_alt",
        "display_name": "mutated aa",
        "type": "text",
        "source": "mutgen",
        "sort_order": 2,
        "row_sort_priority": None,
        "default_order": None,
        "description": "mutated amino acid",
        "hidden": False,
    },
    "mutgen.protein_position": {
        "name": "protein_position",
        "display_name": "protein position",
        "type": "text",
        "source": "mutgen",
        "sort_order": 5,
        "row_sort_priority": None,
        "default_order": None,
        "description": "mutation position in protein",
        "hidden": False,
    },
    "mutgen.peptides": {
        "name": "peptides",
        "display_name": "peptide pairs",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "list of reference-mutant peptide pairs derived from this SNP, along with peptide mutation position.  Corresopnding peptides will be found in the peptide output tables.",
        "hidden": False,
    },
    "mutgen.peptide_warning": {
        "name": "peptide_warning",
        "display_name": "peptide warnings",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "warnings from peptide generation for each variant.  This will include all warnings for successfully generated peptides, as well as warnings for variants where peptides could not be generated",
        "hidden": False,
    },
    "mutgen.reference_peptide": {
        "name": "reference_peptide",
        "display_name": "reference peptide",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "reference peptide",
        "hidden": False,
    },
    "mutgen.mutant_peptide": {
        "name": "mutant_peptide",
        "display_name": "mutated peptide",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "mutated peptide",
        "hidden": False,
    },
    "mutgen.mutation_position_in_peptide": {
        "name": "mutation_position_in_peptide",
        "display_name": "peptide mutation position",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "peptide mutation position",
        "hidden": False,
    },
    "mutgen.transcript_id": {
        "name": "transcript_id",
        "display_name": "Ensembl transcript accession",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Ensembl transcript identifier",
        "hidden": False,
    },
    # verbosity_level=1
    "mutgen.peptide_pair_id": {
        "name": "peptide_pair_id",
        "display_name": "peptide pair id",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "A serial number for peptide pairs in the peptide-output table.",
        "hidden": True,
    },
    "mutgen.hgvs_ref_allele": {
        "name": "hgvs_ref_allele",
        "display_name": "transcript reference allele",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "reference allele (nucleotide) decoded from hgvs_dna",
        "hidden": True,
    },
    "mutgen.hgvs_tum_allele": {
        "name": "hgvs_tum_allele",
        "display_name": "transcript mutant allele",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "tumor allele (nucleotide) decoded from hgvs_dna",
        "hidden": True,
    },
    "mutgen.variant_id": {
        "name": "variant_id",
        "display_name": "variant id",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "an internal unique identifier assigned to each variant",
        "hidden": True,
    },
    "mutgen.impact": {
        "name": "impact",
        "display_name": "mutation impact",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "SNPEff-predicted variant impact LOW/MODERATE/HIGH.",
        "hidden": True,
    },
    "mutgen.transcript_biotype": {
        "name": "transcript_biotype",
        "display_name": "transcript biotype",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "a classification of the transcript type.  These will include protein_coding, the different IG_ and TR_ types, as well as nonsense_mediated_decay, non_stop_decay, pseudogene, etc.",
        "hidden": True,
    },
    "mutgen.hgvs_dna": {
        "name": "hgvs_dna",
        "display_name": "transcript mutation code",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "mutation in hgvs format (nucleotide level) with coordinates based on the transcript",
        "hidden": True,
    },
    "mutgen.hgvs_protein": {
        "name": "hgvs_protein",
        "display_name": "protein mutation code",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "mutation in hgvs format (amino acid level) with coordinates based on the protein",
        "hidden": True,
    },
    "mutgen.cdna_position": {
        "name": "cdna_position",
        "display_name": "cdna position",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "mutation position in cdna",
        "hidden": True,
    },
    "mutgen.cds_position": {
        "name": "cds_position",
        "display_name": "cds position",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "mutation position in cds",
        "hidden": True,
    },
    "mutgen.strand": {
        "name": "strand",
        "display_name": "strand",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "transcript strand: 1 for sense and -1 for anti-sense strand",
        "hidden": True,
    },
    "mutgen.warning": {
        "name": "warning",
        "display_name": "warnings",
        "type": "text",
        "source": "mutgen",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "any warnings associated with peptide generation for each reference-mutant peptide pair",
        "hidden": True,
    },
    # cluster
    "cluster.cluster_number": {
        "name": "cluster_number",
        "display_name": "Cluster.Sub-Cluster Number",
        "type": "text",
        "source": "cluster",
        "sort_order": 1,
        "row_sort_priority": 1,
        "default_order": "ascending",
        "description": "Serial number of the cluster (parental cluster) and subcluster, separated by a period.",
        "hidden": False,
    },
    "cluster.peptide_number": {
        "name": "peptide_number",
        "display_name": "Peptide Number",
        "type": "text",
        "source": "cluster",
        "sort_order": 2,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Serial number of the peptide within the cluster, starting with 1 at the N terminus of the cluster.",
        "hidden": False,
    },
    "cluster.alignment": {
        "name": "alignment",
        "display_name": "Alignment",
        "type": "text",
        "source": "cluster",
        "sort_order": 3,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Alignment of the peptide to the cluster",
        "hidden": False,
    },
    "cluster.position": {
        "name": "position",
        "display_name": "Position",
        "type": "text",
        "source": "cluster",
        "sort_order": 4,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Position of the peptide within the alignment/consensus of the cluster",
        "hidden": False,
    },
    "cluster.sequence_number": {
        "name": "sequence_number",
        "display_name": "Input seq id",
        "type": "text",
        "source": "cluster",
        "sort_order": 5,
        "row_sort_priority": None,
        "default_order": "ascending",
        "description": "Index of the peptide among the input sequences",
        "hidden": False,
    },
    "cluster.peptide": {
        "name": "peptide",
        "display_name": "Peptide",
        "type": "text",
        "source": "cluster",
        "sort_order": 6,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Peptide sequence.",
        "hidden": False,
    },
    "cluster.cluster_consensus": {
        "name": "cluster_consensus",
        "display_name": "Cluster Consensus",
        "type": "text",
        "source": "cluster",
        "sort_order": 7,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Consensus sequence of the cluster, built from the alignment.",
        "hidden": False,
    },
    "cluster.clique_number": {
        "name": "clique_number",
        "display_name": "Cliques Number",
        "type": "int",
        "source": "cluster",
        "sort_order": 1,
        "row_sort_priority": None,
        "default_order": None,
        "description": "Serial number of the clique",
        "hidden": False,
    },
    # allele_distances_table
    "allele_distances.input_allele": {
        "name": "input_allele",
        "display_name": "Input Allele",
        "type": "text",
        "source": "allele_distances",
        "sort_order": 5,
        "row_sort_priority": None,
        "default_order": None,
        "description": "the predicted allele",
        "hidden": False,
    },
    "allele_distances.closest_allele": {
        "name": "closest_allele",
        "display_name": "Closest Allele",
        "type": "text",
        "source": "allele_distances",
        "sort_order": 5,
        "row_sort_priority": None,
        "default_order": None,
        "description": " its nearest neighbor in the training set",
        "hidden": False,
    },
    "allele_distances.allele_distances": {
        "name": "allele_distances",
        "display_name": "Distance",
        "type": "int",
        "source": "allele_distances",
        "sort_order": 6,
        "row_sort_priority": None,
        "default_order": None,
        "description": " Alleles with lower distances to the training set will have more accurate predictions. A distance of 0 indicates a perfect match between alleles and values at or below 0.1 is considered acceptable for generating accurate predictions.",
        "hidden": False,
    },
}

# define the binding methods
# TODO: this should be pulled from some central source
binding_methods = [
    "ann",
    "comblib_sidney2008",
    "consensus",
    "netmhcpan_ba",
    "netmhcpan_el",
    "smm",
    "smmpmbec",
    "mhcnp",
    "mhcflurry",
]
# add mhcii binding methods
binding_methods += [
    "comblib",
    "consensus",
    "netmhciipan",
    "netmhciipan_ba",
    "netmhciipan_el",
    "netmhciipan_ba-4.3",
    "netmhciipan_el-4.3",
    "netmhciipan_ba-4.2",
    "netmhciipan_el-4.2",
    "netmhciipan_ba-4.1",
    "netmhciipan_el-4.1",
    "nn_align",
    "smm_align",
    "tepitope",
    "sturniolo",
]

# define the binding summary columns that do not need to be copied into each method
# mhciinp has its unique column names so those are added here too
binding_summary_fields = [
    "binding.consensus_percentile", "binding.consensus_percentile_rank", "binding.consensus_adjusted_percentile",
    "processing.mhciinp.n_motif", "processing.mhciinp.c_motif","processing.mhciinp.cleavage_probability_score", "processing.mhciinp.cleavage_probability_percentile_rank",
]

peptide_variant_comparison_suffixes = ["peptideA", "peptideB", "peptideA_minusB", "peptideB_minusA"]

# add additional data for each binding method

# this is the final column map that will be used
# we copy fields into it since we can't change the
# size of the hash while we iterate over it
COLUMN_MAP = dict()

for k, v in BASE_COLUMN_MAP.items():
    if k.startswith("binding.") and k not in binding_summary_fields:
        k_parts = k.split(".")
        for m in binding_methods:
            new_k_parts = k_parts.copy()
            new_k_parts.insert(1, m)
            new_k = ".".join(new_k_parts)
            # we make a copy of the base column map items
            # as we don't want to operate on the originals
            v_copy = deepcopy(v)
            v_copy["display_name"] = m + " " + v["display_name"]
            # add mehtod name to column name
            v_copy["name"] = f'{m}_{v_copy["name"]}'
            if m in ["comblib_sidney2008",  "comblib", ] and "score" in k_parts:
                v_copy["default_order"] = "ascending"
                v_copy["number_of_digits"] = 5
            # do not display method specific median_percentile
            elif "median_percentile" in k_parts:
                v_copy["hidden"] = True
                v_copy["row_sort_priority"] = None
            # update description for mhcnp and mhcflurry
            elif m == "mhcnp":
                if "score" in k_parts:
                    v_copy["description"] = "Ranging from 0 to 1 with larger values indicating a higher likelihood of being naturally processed and binding to the given allele."
                elif "percentile" in k_parts:
                    v_copy["description"] = "The percentile rank (ranging from 0 to 100) generated by comparing the peptide's score against those of a set of random peptides from SWISSPROT database.  Lower indicates a higher likelihood."
            elif m == "mhcflurry":
                if "ic50" in k_parts:
                    v_copy["description"] = "Predicted binding affinity (measured in nM) of the peptide for the given allele.  Lower indicates stronger predicted binding."
                elif "percentile" in k_parts:
                    v_copy["description"] = "The percentile rank (ranging from 0 to 100) generated by comparing the peptide's predicted IC50 against those of a set of random peptides from SWISSPROT database.  Lower indicates stronger predicted binding."
            COLUMN_MAP[new_k] = v_copy
        # add binding.overall_median_percentile and binding.median_percentile
        if 'percentile' in k:
            COLUMN_MAP[k] = v
    else:
        COLUMN_MAP[k] = v

# adding peptide_variant_comparison_suffixes
for k, v in COLUMN_MAP.copy().items():
    if (
        k.startswith("core.")
        or k.startswith("binding.")
        or k.startswith("immunogenicity.")
    ):
        for suffix in peptide_variant_comparison_suffixes:
            new_k = f"{k}-{suffix.lower()}"
            v_copy = deepcopy(v)
            v_copy["name"] = f"{v_copy['name']}-{suffix.lower()}"
            # modify display name with peptide_variant_comparison preffixes
            v_copy["display_name"] = (
                suffix.replace("peptideA_minusB", "difference (peptideB - peptideA)").replace("peptideB_minusA", "difference (peptideB - peptideA)")
                + " "
                + v["display_name"]
            )
            v_copy["display_name"] = (
                v_copy["display_name"]
                .replace("peptideA peptide", "peptideA")
                .replace("peptideB peptide", "peptideB")
            )
            if suffix == "peptideA_minusB":
                v_copy["number_of_digits"] = 4
                if "percentile" in k:
                    v_copy["number_of_digits"] = 2
            if "percentile" in k or "length" in k:
                v_copy["hidden"] = True
            # all new columns will not be used for soring result
            v_copy["row_sort_priority"] = None
            COLUMN_MAP[new_k] = v_copy


def get_column_info(column_name):
    """Given a column name, return its info"""

    lower_name = column_name.lower()

    if lower_name in COLUMN_MAP:
        return COLUMN_MAP[lower_name]
    # adding Percentile Columns for each allele and method
    elif lower_name.startswith('binding.') and '.percentile.' in lower_name:
        print("Percentile Column for allele and method was found: " + lower_name)
        return {
            "name": lower_name,
            "display_name": column_name,
            "type": "float",
            "source": "binding/elution",
            "sort_order": 3,
            "row_sort_priority": None,
            "default_order": "ascending",
            "number_of_digits": None,
            "description": "The percentile rank generated by comparing the peptide's IC50/score against those of a set of random peptides from SWISSPROT database",
            "hidden": True,
        }
    # handle columns like "binding.nn_align.adjusted_percentile.HLA-DRB1*01:01"
    elif lower_name.startswith('binding.') and '.adjusted_percentile.' in lower_name:
        print("Adjusted Percentile Column for allele and method was found: " + lower_name)
        return {
            "name": lower_name,
            "display_name": column_name,
            "type": "float",
            "source": "binding/elution",
            "sort_order": 3,
            "row_sort_priority": None,
            "default_order": "ascending",
            "number_of_digits": None,
            "description": "The adjusted percentile rank generated by comparing the peptide's IC50/score against those of a set of random peptides from SWISSPROT database",
            "hidden": True,
        }
    # handle columns like "binding.median_percentile.HLA-DRB1*01:03"
    elif lower_name.startswith('binding.median_percentile.'):
        print("Median Percentile Column for allele was found: " + lower_name)
        return {
            "name": lower_name,
            "display_name": column_name,
            "type": "float",
            "source": "binding/elution",
            "sort_order": 2,
            "row_sort_priority": None,
            "default_order": "ascending",
            "number_of_digits": None,
            "description": "The median percentile rank of binding/elution predictions by alleles was calculated by using the adjusted percentiles when available, otherwise the standard percentiles",
            "hidden": True,
        }
    else:
        # TODO: throw a warning
        print("Warning - Column name not found: " + lower_name)
        return {
            "name": lower_name,
            "display_name": column_name,
            "type": "text",
            "source": "unknown",
            "sort_order": None,
            "row_sort_priority": None,
            "default_order": None,
            "description": "N/A",
            "hidden": False,
        }
