from collections import namedtuple, defaultdict
from curses.ascii import isdigit
import logging
import os
import re
from subprocess import Popen, PIPE
from tempfile import NamedTemporaryFile
import pkg_resources
from iedbtools_utilities.sequence_io import SequenceOutput

logger = logging.getLogger(__name__)

_executable_filename = \
    pkg_resources.resource_filename('netmhcpan_4_1_executable', 'netMHCpan')
EXECUTABLE_PATH = os.path.join(os.path.dirname(__file__), _executable_filename)

PredictionInput = namedtuple('PredictionInput', ['sequence', 'allele', 'binding_length'])

# moved from allele-info
def is_user_defined_allele(allele):
    '''Returns True if *allele* is a user-defined sequence, False if it is anallele name.
    '''
    if '-' in allele or '*' in allele or '1' in allele:
        return False
    return True


def predict_many_peptides_file(peptide_file, allele_list, el=False, with_distance_info=False):
    """ 
        Performs a prediction for every sequence / allele-length combination.
        Returns a dictionary { peptide: [score, score, ...], ... }
        with one entry per prediction made.
        from netmhcpan_4_1_executable import predict_many_peptides_file
        predict_many_peptides_file('test.pep',['HLA-A*0101'])
        predict_many_peptides_file('test.pep',['HLA-A*0101'],el=True,with_distance_info=True)
    """
    
    to_delete = []
    
    try:
        all_scores = {}
        distances = {}
        scores_by_alleles_and_peptides = {}
        for allele in allele_list:
            peptide_filepath = peptide_file
            
            if is_user_defined_allele(allele):
                # need to be updated too
                # A user-defined allele must be passed via a fasta file.
                t = NamedTemporaryFile(delete=False, mode='w')
                fasta = SequenceOutput.to_fasta(allele)
                t.write(fasta)
                t.close()

                user_defined_allele_filepath = sequence_list_to_fasta_file([allele])

                if not el :
                    cmd = [
                        EXECUTABLE_PATH, '-p', peptide_filepath,
                        '-hlaseq', user_defined_allele_filepath,
                        '-BA'
                    ]
                else :
                    cmd = [
                        EXECUTABLE_PATH, '-p', peptide_filepath,
                        '-hlaseq', user_defined_allele_filepath, 
                    ] 
            else:
                # An allele name can be passed as a command-line argument.
                # netmhcpan uses allele names with the '*' stripped
                stripped_allele_name = strip_alllele_name(allele)
                if not el :
                    cmd = [
                        EXECUTABLE_PATH, '-p', peptide_filepath,
                        '-a', stripped_allele_name,
                        '-BA'
                    ]
                else :
                    cmd = [
                        EXECUTABLE_PATH, '-p', peptide_filepath,
                        '-a', stripped_allele_name,
                    ]


            logger.info('Executing: "%s"', ' '.join(cmd))
            #print('Executing: "%s"', ' '.join(cmd))
            process = Popen(cmd, stdout=PIPE)
            stdoutdata, stderrdata_ignored = process.communicate()
            stdoutdata = stdoutdata.decode()
            logger.debug('Raw output:\n%s', stdoutdata)
            scores_by_peptides = get_peptides_and_scores_from_output(stdoutdata, el)
            for peptide, score in scores_by_peptides.items():
                scores_by_alleles_and_peptides[(allele,peptide)] = score

            distance = get_distance(stdoutdata)
            distances.update(distance)
    except:
        raise

    if with_distance_info:
        return scores_by_alleles_and_peptides, distances
    else:
        return scores_by_alleles_and_peptides



def predict_many_peptides(sequence_list, allele_length_2tuple_list, delete_tempfiles=False, el=False, with_distance_info=False, peptide_file=None):
    """ 
        Performs a prediction for every sequence / allele-length combination.
        Returns a dictionary { (sequence, allele, length): [score, score, ...], ... }
        with one entry per prediction made.
    """
    
    to_delete = []
    
    try:
        all_scores = {}
        distances = {}
        for allele, binding_length in allele_length_2tuple_list:
            peptide_filepath = sequence_list_to_peptide_file(sequence_list,binding_length)
            peptide_number_list = [len(seq)-int(binding_length)+1 for seq in sequence_list]
            to_delete.append(peptide_filepath)
            #with open(peptide_filepath,'r') as r_f:
            #    print('peptide file content: %s' % r_f.read())
            # The call made to netmhcpan is different for a user-defined allele (hlaseq)
            #    vs an allele name.
            if is_user_defined_allele(allele):
                # need to be updated too
                # A user-defined allele must be passed via a fasta file.
                t = NamedTemporaryFile(delete=False, mode='w')
                fasta = SequenceOutput.to_fasta(allele)
                t.write(fasta)
                t.close()

                user_defined_allele_filepath = sequence_list_to_fasta_file([allele])

                if not el :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-hlaseq', user_defined_allele_filepath,
                        '-l', str(binding_length),
                        '-BA'
                    ]
                else :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-hlaseq', user_defined_allele_filepath,
                        '-l', str(binding_length)
                    ] 
            else:
                # An allele name can be passed as a command-line argument.
                # netmhcpan uses allele names with the '*' stripped
                stripped_allele_name = strip_alllele_name(allele)
                if not el :
                    cmd = [
                        EXECUTABLE_PATH, '-p', peptide_filepath,
                        '-a', stripped_allele_name,
                        '-l', str(binding_length),
                        '-BA'
                    ]
                else :
                    cmd = [
                        EXECUTABLE_PATH, '-p', peptide_filepath,
                        '-a', stripped_allele_name,
                        '-l', str(binding_length)
                    ]
            if not el : 
                get_scores_from_output = get_allele_name_scores_from_netmhcpan_output
            else :
                get_scores_from_output = get_allele_name_scores_from_netmhcpan_output_EL

            logger.info('Executing: "%s"', ' '.join(cmd))
            #print('Executing: "%s"', ' '.join(cmd))
            process = Popen(cmd, stdout=PIPE)
            stdoutdata, stderrdata_ignored = process.communicate()
            stdoutdata = stdoutdata.decode()
            logger.debug('Raw output:\n%s', stdoutdata)
            scores_by_sequence_idx = get_scores_from_output(stdoutdata, peptide_number_list)
            distance = get_distance(stdoutdata)
            distances.update(distance)
            # Wrap the scores up nicely to return to the caller
            for seqidx, scores in sorted(scores_by_sequence_idx.items()):
                sequence = sequence_list[seqidx]
                prediction_input = PredictionInput(sequence, allele, binding_length)
                all_scores[prediction_input] = scores
    except:
        raise
    finally:
        if delete_tempfiles:
            for filepath in to_delete:
                os.unlink(filepath)
    if with_distance_info:
        return all_scores, distances
    else:
        return all_scores


def predict_many(sequence_list, allele_length_2tuple_list, delete_tempfiles=False, el=False, with_distance_info=False):
    """ 
        Performs a prediction for every sequence / allele-length combination.
        Returns a dictionary { (sequence, allele, length): [score, score, ...], ... }
        with one entry per prediction made.
    """
    sequence_filepath = sequence_list_to_fasta_file(sequence_list)
    to_delete = [sequence_filepath]

    try:
        all_scores = {}
        distances = {}
        for allele, binding_length in allele_length_2tuple_list:
            # The call made to netmhcpan is different for a user-defined allele (hlaseq)
            #    vs an allele name.
            if is_user_defined_allele(allele):
                # A user-defined allele must be passed via a fasta file.
                t = NamedTemporaryFile(delete=False, mode='w')
                fasta = SequenceOutput.to_fasta(allele)
                t.write(fasta)
                t.close()

                user_defined_allele_filepath = sequence_list_to_fasta_file([allele])

                if not el :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-hlaseq', user_defined_allele_filepath,
                        '-l', str(binding_length),
                        '-BA'
                    ]
                else :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-hlaseq', user_defined_allele_filepath,
                        '-l', str(binding_length)
                    ] 
            else:
                # An allele name can be passed as a command-line argument.
                # netmhcpan uses allele names with the '*' stripped
                stripped_allele_name = strip_alllele_name(allele)
                if not el :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-a', stripped_allele_name,
                        '-l', str(binding_length),
                        '-BA', 
                    ]
                else :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-a', stripped_allele_name,
                        '-l', str(binding_length),
                    ]
            if not el : 
                get_scores_from_output = get_allele_name_scores_from_netmhcpan_output
            else :
                get_scores_from_output = get_allele_name_scores_from_netmhcpan_output_EL

            logger.info('Executing: "%s"', ' '.join(cmd))
            process = Popen(cmd, stdout=PIPE)
            stdoutdata, stderrdata_ignored = process.communicate()
            stdoutdata = stdoutdata.decode()
            logger.debug('Raw output:\n%s', stdoutdata)
            scores_by_sequence_idx = get_scores_from_output(stdoutdata)
            distance = get_distance(stdoutdata)
            distances.update(distance)
            # Wrap the scores up nicely to return to the caller
            for seqidx, scores in sorted(scores_by_sequence_idx.items()):
                sequence = sequence_list[seqidx]
                prediction_input = PredictionInput(sequence, allele, binding_length)
                all_scores[prediction_input] = scores
    except:
        raise
    finally:
        if delete_tempfiles:
            for filepath in to_delete:
                os.unlink(filepath)
    if with_distance_info:
        return all_scores, distances
    else:
        return all_scores

def predict_many_into_one_sequence(sequence_list, allele_length_2tuple_list, delete_tempfiles=False, el=False, with_distance_info=False):
    """ 
        Performs a prediction for every sequence / allele-length combination.
        Returns a dictionary { (sequence, allele, length): [score, score, ...], ... }
        with one entry per prediction made.
    """
    sequence_filepath = sequence_list_to_one_seq_fasta_file(sequence_list)
    seq_len_list = [len(seq) for seq in sequence_list]
    to_delete = [sequence_filepath]

    try:
        all_scores = {}
        distances = {}
        for allele, binding_length in allele_length_2tuple_list:
            # The call made to netmhcpan is different for a user-defined allele (hlaseq)
            #    vs an allele name.
            if is_user_defined_allele(allele):
                # A user-defined allele must be passed via a fasta file.
                t = NamedTemporaryFile(delete=False, mode='w')
                fasta = SequenceOutput.to_fasta(allele)
                t.write(fasta)
                t.close()

                user_defined_allele_filepath = sequence_list_to_fasta_file([allele])

                if not el :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-hlaseq', user_defined_allele_filepath,
                        '-l', str(binding_length),
                        '-BA'
                    ]
                else :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-hlaseq', user_defined_allele_filepath,
                        '-l', str(binding_length)
                    ] 
            else:
                # An allele name can be passed as a command-line argument.
                # netmhcpan uses allele names with the '*' stripped
                stripped_allele_name = strip_alllele_name(allele)
                if not el :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-a', stripped_allele_name,
                        '-l', str(binding_length),
                        '-BA', 
                    ]
                else :
                    cmd = [
                        EXECUTABLE_PATH, '-f', sequence_filepath,
                        '-a', stripped_allele_name,
                        '-l', str(binding_length),
                    ]
            if not el : 
                get_scores_from_output = get_allele_name_scores_from_netmhcpan_output
            else :
                get_scores_from_output = get_allele_name_scores_from_netmhcpan_output_EL

            logger.info('Executing: "%s"', ' '.join(cmd))
            process = Popen(cmd, stdout=PIPE)
            stdoutdata, stderrdata_ignored = process.communicate()
            stdoutdata = stdoutdata.decode()
            logger.debug('Raw output:\n%s', stdoutdata)
            scores_by_sequence_idx = get_scores_from_output(stdoutdata, seq_len_list, binding_length)
            distance = get_distance(stdoutdata)
            distances.update(distance)
            # Wrap the scores up nicely to return to the caller
            for seqidx, scores in sorted(scores_by_sequence_idx.items()):
                print('seqidx:%s' % seqidx)
                sequence = sequence_list[seqidx]
                prediction_input = PredictionInput(sequence, allele, binding_length)
                all_scores[prediction_input] = scores
    except:
        raise
    finally:
        if delete_tempfiles:
            for filepath in to_delete:
                os.unlink(filepath)
    if with_distance_info:
        return all_scores, distances
    else:
        return all_scores

def strip_alllele_name(allele):
    if any(prefix in allele for prefix in ['SLA', 'Mamu', 'BoLA']):
        stripped_allele_name = allele.replace('*', ':')
    else:
        stripped_allele_name = allele.replace('*', '')
    return stripped_allele_name


def predict_from_peptide_file(allele_name, binding_length, peptide_file_path):
    """ @brief Performs a prediction with a file containing a different peptide on each line as input.
        This is a special mode of netMHCpan that is used for efficient calculation of score distributions 
        used in generating percentiles from raw scores.
    """
    
    cmd = [
        EXECUTABLE_PATH, '-a', allele_name, '-l', str(binding_length), '-p', peptide_file_path
    ]
    
    process = Popen(cmd, stdout=PIPE)
    stdoutdata, stderrdata = process.communicate()
    stdoutdata = stdoutdata.decode()
    output = stdoutdata.splitlines()
    
    # list_res is list of tuples of prediction scores without the comment lines and column headers
    list_res = [tuple(res.split()) for res in output if not res.startswith('#') and res.split() and res.split()[0].isdigit()]
    
    scores = []
    for score in list_res:
        scores.append(score[12])
    return scores


def sequence_list_to_fasta_file(sequence_list):
    """ | *brief*: Writes the sequences in *sequence_list* as fasta sequences to a file and returns
        |    the filepath.
    """
    t = NamedTemporaryFile(delete=False, mode='w')
    for i, sequence in enumerate(sequence_list):
        t.write('>seq-{}\n'.format(i))
        t.write(sequence)
        t.write('\n\n')
    t.close()
    return t.name


def sequence_list_to_one_seq_fasta_file(sequence_list):
    """ | *brief*: Writes the sequences in *sequence_list* as fasta sequences to a file and returns
        |    the filepath.
    """
    t = NamedTemporaryFile(delete=False, mode='w')
    t.write('>seq\n')
    t.write('\n'.join(sequence_list))
    t.close()
    return t.name

def sequence_list_to_peptide_file(sequence_list, length):
    """ | *brief*: Writes the sequences in *sequence_list* as fasta sequences to a file and returns
        |    the filepath.
    """
    t = NamedTemporaryFile(delete=False, mode='w')
    length = int(length)
    peptides = []
    for sequence in sequence_list:
         
        while len(sequence) >= length:

            peptides.append(sequence[:length])

            sequence = sequence[1:]
    t.write('\n'.join(peptides))
    t.close()
    return t.name


def get_allele_name_scores_from_netmhcpan_output(netmhcpan_output, seq_len_list=None, length='9'):
    """ | *brief*: Parses the string *netmhcpan_output* for scores from an allele name
        |    prediction request.
    """
    # Output lines of interest when allele name is used look like this:
    #---------------------------------------------------------------------------------------------------------------------------
    # Pos         MHC        Peptide      Core Of Gp Gl Ip Il        Icore        Identity  Score_EL %Rank_EL Score_BA %Rank_BA  Aff(nM) BindLevel
    #---------------------------------------------------------------------------------------------------------------------------
    #   1  SLA-6:0105      TMDKSELVQ TMDKSELVQ  0  0  0  0  0    TMDKSELVQ 143B_BOVIN_P293 0.0050150   34.875 0.027888   61.292 36976.44
    #   2  SLA-6:0105      MDKSELVQK MDKSELVQK  0  0  0  0  0    MDKSELVQK 143B_BOVIN_P293 0.0110540   24.847 0.042580   41.352 31541.90
    #   3  SLA-6:0105      DKSELVQKA DKSELVQKA  0  0  0  0  0    DKSELVQKA 143B_BOVIN_P293 0.0170930   19.788 0.029120   59.174 36486.81
    #   4  SLA-6:0105      KSELVQKAK KSELVQKAK  0  0  0  0  0    KSELVQKAK 143B_BOVIN_P293 0.0011810   54.667 0.017205   82.890 41507.27
    #   5  SLA-6:0105      SELVQKAKL SELVQKAKL  0  0  0  0  0    SELVQKAKL 143B_BOVIN_P293 0.3265630    0.348 0.164852    2.876  8401.08 <= SB
    
    # We can get the sequence index and score from the 11th & 13th items.
    # netmhcpan_output is list of list of prediction scores without the comment lines and column headers
    netmhcpan_output = [line.split() for line in netmhcpan_output.split('\n') if not line.startswith('#') and line.split() and line.split()[0].isdigit()]
    length = int(length)
    scores = defaultdict(list)

    if seq_len_list:
        index_list = []
        sequence_idx = 0
        for seq_len in seq_len_list:
            for i in range(seq_len-length+1):
                index_list.append(sequence_idx)
                sequence_idx += 1
            for j in range(length-1):
                sequence_idx += 1

        for i in index_list:
            row = netmhcpan_output[i]
            
            for j in range(len(seq_len_list)):
                seq_len = seq_len_list[j]
                if i < seq_len:
                    sequence_idx = j
                else:
                    i -= seq_len
            #print(row)
            idx = 10
            core = row[3]
            icore = row[9]
            ic50_score = float(row[idx+1])  
            scores[sequence_idx].append(ic50_score)
            scores[sequence_idx].append(core)
            scores[sequence_idx].append(icore)

    else:
        for row in netmhcpan_output:
            idx = 10
            core = row[3]
            icore = row[9]
            sequence_idx = int(row[idx].split('-')[1])
            ic50_score = float(row[idx+5])  
            scores[sequence_idx].append(ic50_score)
            scores[sequence_idx].append(core)
            scores[sequence_idx].append(icore)

    return scores



def get_allele_name_scores_from_netmhcpan_output_EL(netmhcpan_output, seq_len_list=None, length='9'):
    """ | *brief*: Parses the string *netmhcpan_output* for scores from an allele name
        |    prediction request.
    """
    #---------------------------------------------------------------------------------------------------------------------------
    # Pos         MHC        Peptide      Core Of Gp Gl Ip Il        Icore        Identity  Score_EL %Rank_EL BindLevel
    #---------------------------------------------------------------------------------------------------------------------------
    #   1  SLA-6:0105      TMDKSELVQ TMDKSELVQ  0  0  0  0  0    TMDKSELVQ 143B_BOVIN_P293 0.0050150   34.875
    #   2  SLA-6:0105      MDKSELVQK MDKSELVQK  0  0  0  0  0    MDKSELVQK 143B_BOVIN_P293 0.0110540   24.847
    #   3  SLA-6:0105      DKSELVQKA DKSELVQKA  0  0  0  0  0    DKSELVQKA 143B_BOVIN_P293 0.0170930   19.788
    #   4  SLA-6:0105      KSELVQKAK KSELVQKAK  0  0  0  0  0    KSELVQKAK 143B_BOVIN_P293 0.0011810   54.667
    #   5  SLA-6:0105      SELVQKAKL SELVQKAKL  0  0  0  0  0    SELVQKAKL 143B_BOVIN_P293 0.3265630    0.348 <= SB
    #   6  SLA-6:0105      ELVQKAKLA ELVQKAKLA  0  0  0  0  0    ELVQKAKLA 143B_BOVIN_P293 0.0095020   26.706
        # -----------------------------------------------------------------------------------

    # netmhcpan_output is list of list of prediction scores without the comment lines and column headers
    netmhcpan_output = [line.split() for line in netmhcpan_output.split('\n') if not line.startswith('#') and line.split() and line.split()[0].isdigit()]
    length = int(length)
    scores = defaultdict(list)

    if seq_len_list:
        index_list = []
        sequence_idx = 0
        for seq_len in seq_len_list:
            for i in range(seq_len-length+1):
                index_list.append(sequence_idx)
                sequence_idx += 1
            for j in range(length-1):
                sequence_idx += 1

        for i in index_list:
            row = netmhcpan_output[i]
            
            for j in range(len(seq_len_list)):
                seq_len = seq_len_list[j]
                if i < seq_len:
                    sequence_idx = j
                else:
                    i -= seq_len
            #print(row)
            idx = 10
            core = row[3]
            icore = row[9]
            ic50_score = float(row[idx+1])  
            scores[sequence_idx].append(ic50_score)
            scores[sequence_idx].append(core)
            scores[sequence_idx].append(icore)
    else:
        for row in netmhcpan_output:
            idx = 10
            core = row[3]
            icore = row[9]
            sequence_idx = int(row[idx].split('-')[1])
            ic50_score = float(row[idx+1])  
            scores[sequence_idx].append(ic50_score)
            scores[sequence_idx].append(core)
            scores[sequence_idx].append(icore)

    return scores

def get_peptides_and_scores_from_output(netmhcpan_output, el=False):
    """ | *brief*: Parses the string *netmhcpan_output* for scores from an allele name
        |    prediction request.

# NetMHCpan version 4.1b

# Tmpdir made /mnt/beegfs/jyan/pbstmp.3825179.herman.hpc.lji.org/netMHCpanIFh5PE
# Input is in PEPTIDE format

# Make EL predictions

HLA-A0101 : Distance to training data  0.000 (using nearest neighbor HLA-A01:01)

# Rank Threshold for Strong binding peptides   0.500
# Rank Threshold for Weak binding peptides   2.000
---------------------------------------------------------------------------------------------------------------------------
 Pos         MHC        Peptide      Core Of Gp Gl Ip Il        Icore        Identity  Score_EL %Rank_EL BindLevel
---------------------------------------------------------------------------------------------------------------------------
   1  HLA-A*0101     TMDKSELVQK TMDKSELQK  0  7  1  0  0   TMDKSELVQK         PEPLIST 0.0271010    2.853
   1  HLA-A*0101     EILNSPEKAC EILNSPEKC  0  8  1  0  0   EILNSPEKAC         PEPLIST 0.0001070   56.400
   1  HLA-A*0101      KMKGDYFRY KMKGDYFRY  0  0  0  0  0    KMKGDYFRY         PEPLIST 0.0930380    1.347 <= WB
---------------------------------------------------------------------------------------------------------------------------
    """

    # netmhcpan_output is list of list of prediction scores without the comment lines and column headers
    netmhcpan_output = [line.split() for line in netmhcpan_output.split('\n') if not line.startswith('#') and line.split() and line.split()[0].isdigit()]
    
    scores = defaultdict(list)

    for row in netmhcpan_output:
        
        #print(row)
        peptide = row[2]
        core = row[3]
        icore = row[9]
        if el:
            ic50_score = float(row[11])  
        else:
            ic50_score = float(row[15])  
        scores[peptide].append(ic50_score)
        scores[peptide].append(core)
        scores[peptide].append(icore)

    return scores


def get_peptides_and_scores_from_output(netmhcpan_output, el=False):
    """ | *brief*: Parses the string *netmhcpan_output* for scores from an allele name
        |    prediction request.

# NetMHCpan version 4.1b

# Tmpdir made /mnt/beegfs/jyan/pbstmp.3825179.herman.hpc.lji.org/netMHCpanIFh5PE
# Input is in PEPTIDE format

# Make EL predictions

HLA-A0101 : Distance to training data  0.000 (using nearest neighbor HLA-A01:01)

# Rank Threshold for Strong binding peptides   0.500
# Rank Threshold for Weak binding peptides   2.000
---------------------------------------------------------------------------------------------------------------------------
 Pos         MHC        Peptide      Core Of Gp Gl Ip Il        Icore        Identity  Score_EL %Rank_EL BindLevel
---------------------------------------------------------------------------------------------------------------------------
   1  HLA-A*0101     TMDKSELVQK TMDKSELQK  0  7  1  0  0   TMDKSELVQK         PEPLIST 0.0271010    2.853
   1  HLA-A*0101     EILNSPEKAC EILNSPEKC  0  8  1  0  0   EILNSPEKAC         PEPLIST 0.0001070   56.400
   1  HLA-A*0101      KMKGDYFRY KMKGDYFRY  0  0  0  0  0    KMKGDYFRY         PEPLIST 0.0930380    1.347 <= WB
---------------------------------------------------------------------------------------------------------------------------
    """

    # netmhcpan_output is list of list of prediction scores without the comment lines and column headers
    netmhcpan_output = [line.split() for line in netmhcpan_output.split('\n') if not line.startswith('#') and line.split() and line.split()[0].isdigit()]
    
    scores = defaultdict(list)

    for row in netmhcpan_output:
        
        #print(row)
        peptide = row[2]
        core = row[3]
        icore = row[9]
        if el:
            ic50_score = float(row[11])  
        else:
            ic50_score = float(row[15])  
        scores[peptide].append(ic50_score)
        scores[peptide].append(core)
        scores[peptide].append(icore)

    return scores



def get_distance(netmhcpan_output):
    #  get these information from the output: input allele, closest allele, distance
    # output format: HLA-A24:138 : Distance to training data  0.050 (using nearest neighbor HLA-A24:03)

    distance_rows = [row.strip() for row in netmhcpan_output.split('\n') if ': Distance to training data' in row]
    if not distance_rows:
        logger.warning('did not find allele distance info from netmhcpan_output')

    distances_dict = {}
    for row in distance_rows:
        m=re.match(r'^([\-\:\w]+) : Distance to training data  ([\d\.]+) \(using nearest neighbor ([\-\:\w]+)\)', row)
        if m:
            input_allele, distance, closest_allele = m.groups()
            distances_dict[input_allele] = ( closest_allele, distance, )
    return distances_dict



# Setting binding length options same as those available for web tool 
allowed_binding_lengths = (8, 9, 10, 11, 12, 13, 14)
    
# Retreived using option -listMHC on the executable
allowed_allele_names = [
    'HLA-A01:01',
    'HLA-A01:02',
    'HLA-A01:03',
    'HLA-A01:06',
    'HLA-A01:07',
    'HLA-A01:08',
    'HLA-A01:09',
    'HLA-A01:10',
    'HLA-A01:12',
    'HLA-A01:13',
    'HLA-A01:14',
    'HLA-A01:17',
    'HLA-A01:19',
    'HLA-A01:20',
    'HLA-A01:21',
    'HLA-A01:23',
    'HLA-A01:24',
    'HLA-A01:25',
    'HLA-A01:26',
    'HLA-A01:28',
    'HLA-A01:29',
    'HLA-A01:30',
    'HLA-A01:32',
    'HLA-A01:33',
    'HLA-A01:35',
    'HLA-A01:36',
    'HLA-A01:37',
    'HLA-A01:38',
    'HLA-A01:39',
    'HLA-A01:40',
    'HLA-A01:41',
    'HLA-A01:42',
    'HLA-A01:43',
    'HLA-A01:44',
    'HLA-A01:45',
    'HLA-A01:46',
    'HLA-A01:47',
    'HLA-A01:48',
    'HLA-A01:49',
    'HLA-A01:50',
    'HLA-A01:51',
    'HLA-A01:54',
    'HLA-A01:55',
    'HLA-A01:58',
    'HLA-A01:59',
    'HLA-A01:60',
    'HLA-A01:61',
    'HLA-A01:62',
    'HLA-A01:63',
    'HLA-A01:64',
    'HLA-A01:65',
    'HLA-A01:66',
    'HLA-A02:01',
    'HLA-A02:02',
    'HLA-A02:03',
    'HLA-A02:04',
    'HLA-A02:05',
    'HLA-A02:06',
    'HLA-A02:07',
    'HLA-A02:08',
    'HLA-A02:09',
    'HLA-A02:10',
    'HLA-A02:11',
    'HLA-A02:12',
    'HLA-A02:13',
    'HLA-A02:14',
    'HLA-A02:16',
    'HLA-A02:17',
    'HLA-A02:18',
    'HLA-A02:19',
    'HLA-A02:20',
    'HLA-A02:21',
    'HLA-A02:22',
    'HLA-A02:24',
    'HLA-A02:25',
    'HLA-A02:26',
    'HLA-A02:27',
    'HLA-A02:28',
    'HLA-A02:29',
    'HLA-A02:30',
    'HLA-A02:31',
    'HLA-A02:33',
    'HLA-A02:34',
    'HLA-A02:35',
    'HLA-A02:36',
    'HLA-A02:37',
    'HLA-A02:38',
    'HLA-A02:39',
    'HLA-A02:40',
    'HLA-A02:41',
    'HLA-A02:42',
    'HLA-A02:44',
    'HLA-A02:45',
    'HLA-A02:46',
    'HLA-A02:47',
    'HLA-A02:48',
    'HLA-A02:49',
    'HLA-A02:50',
    'HLA-A02:51',
    'HLA-A02:52',
    'HLA-A02:54',
    'HLA-A02:55',
    'HLA-A02:56',
    'HLA-A02:57',
    'HLA-A02:58',
    'HLA-A02:59',
    'HLA-A02:60',
    'HLA-A02:61',
    'HLA-A02:62',
    'HLA-A02:63',
    'HLA-A02:64',
    'HLA-A02:65',
    'HLA-A02:66',
    'HLA-A02:67',
    'HLA-A02:68',
    'HLA-A02:69',
    'HLA-A02:70',
    'HLA-A02:71',
    'HLA-A02:72',
    'HLA-A02:73',
    'HLA-A02:74',
    'HLA-A02:75',
    'HLA-A02:76',
    'HLA-A02:77',
    'HLA-A02:78',
    'HLA-A02:79',
    'HLA-A02:80',
    'HLA-A02:81',
    'HLA-A02:84',
    'HLA-A02:85',
    'HLA-A02:86',
    'HLA-A02:87',
    'HLA-A02:89',
    'HLA-A02:90',
    'HLA-A02:91',
    'HLA-A02:92',
    'HLA-A02:93',
    'HLA-A02:95',
    'HLA-A02:96',
    'HLA-A02:97',
    'HLA-A02:99',
    'HLA-A02:101',
    'HLA-A02:102',
    'HLA-A02:103',
    'HLA-A02:104',
    'HLA-A02:105',
    'HLA-A02:106',
    'HLA-A02:107',
    'HLA-A02:108',
    'HLA-A02:109',
    'HLA-A02:110',
    'HLA-A02:111',
    'HLA-A02:112',
    'HLA-A02:114',
    'HLA-A02:115',
    'HLA-A02:116',
    'HLA-A02:117',
    'HLA-A02:118',
    'HLA-A02:119',
    'HLA-A02:120',
    'HLA-A02:121',
    'HLA-A02:122',
    'HLA-A02:123',
    'HLA-A02:124',
    'HLA-A02:126',
    'HLA-A02:127',
    'HLA-A02:128',
    'HLA-A02:129',
    'HLA-A02:130',
    'HLA-A02:131',
    'HLA-A02:132',
    'HLA-A02:133',
    'HLA-A02:134',
    'HLA-A02:135',
    'HLA-A02:136',
    'HLA-A02:137',
    'HLA-A02:138',
    'HLA-A02:139',
    'HLA-A02:140',
    'HLA-A02:141',
    'HLA-A02:142',
    'HLA-A02:143',
    'HLA-A02:144',
    'HLA-A02:145',
    'HLA-A02:146',
    'HLA-A02:147',
    'HLA-A02:148',
    'HLA-A02:149',
    'HLA-A02:150',
    'HLA-A02:151',
    'HLA-A02:152',
    'HLA-A02:153',
    'HLA-A02:154',
    'HLA-A02:155',
    'HLA-A02:156',
    'HLA-A02:157',
    'HLA-A02:158',
    'HLA-A02:159',
    'HLA-A02:160',
    'HLA-A02:161',
    'HLA-A02:162',
    'HLA-A02:163',
    'HLA-A02:164',
    'HLA-A02:165',
    'HLA-A02:166',
    'HLA-A02:167',
    'HLA-A02:168',
    'HLA-A02:169',
    'HLA-A02:170',
    'HLA-A02:171',
    'HLA-A02:172',
    'HLA-A02:173',
    'HLA-A02:174',
    'HLA-A02:175',
    'HLA-A02:176',
    'HLA-A02:177',
    'HLA-A02:178',
    'HLA-A02:179',
    'HLA-A02:180',
    'HLA-A02:181',
    'HLA-A02:182',
    'HLA-A02:183',
    'HLA-A02:184',
    'HLA-A02:185',
    'HLA-A02:186',
    'HLA-A02:187',
    'HLA-A02:188',
    'HLA-A02:189',
    'HLA-A02:190',
    'HLA-A02:191',
    'HLA-A02:192',
    'HLA-A02:193',
    'HLA-A02:194',
    'HLA-A02:195',
    'HLA-A02:196',
    'HLA-A02:197',
    'HLA-A02:198',
    'HLA-A02:199',
    'HLA-A02:200',
    'HLA-A02:201',
    'HLA-A02:202',
    'HLA-A02:203',
    'HLA-A02:204',
    'HLA-A02:205',
    'HLA-A02:206',
    'HLA-A02:207',
    'HLA-A02:208',
    'HLA-A02:209',
    'HLA-A02:210',
    'HLA-A02:211',
    'HLA-A02:212',
    'HLA-A02:213',
    'HLA-A02:214',
    'HLA-A02:215',
    'HLA-A02:216',
    'HLA-A02:217',
    'HLA-A02:218',
    'HLA-A02:219',
    'HLA-A02:220',
    'HLA-A02:221',
    'HLA-A02:224',
    'HLA-A02:228',
    'HLA-A02:229',
    'HLA-A02:230',
    'HLA-A02:231',
    'HLA-A02:232',
    'HLA-A02:233',
    'HLA-A02:234',
    'HLA-A02:235',
    'HLA-A02:236',
    'HLA-A02:237',
    'HLA-A02:238',
    'HLA-A02:239',
    'HLA-A02:240',
    'HLA-A02:241',
    'HLA-A02:242',
    'HLA-A02:243',
    'HLA-A02:244',
    'HLA-A02:245',
    'HLA-A02:246',
    'HLA-A02:247',
    'HLA-A02:248',
    'HLA-A02:249',
    'HLA-A02:251',
    'HLA-A02:252',
    'HLA-A02:253',
    'HLA-A02:254',
    'HLA-A02:255',
    'HLA-A02:256',
    'HLA-A02:257',
    'HLA-A02:258',
    'HLA-A02:259',
    'HLA-A02:260',
    'HLA-A02:261',
    'HLA-A02:262',
    'HLA-A02:263',
    'HLA-A02:264',
    'HLA-A02:265',
    'HLA-A02:266',
    'HLA-A03:01',
    'HLA-A03:02',
    'HLA-A03:04',
    'HLA-A03:05',
    'HLA-A03:06',
    'HLA-A03:07',
    'HLA-A03:08',
    'HLA-A03:09',
    'HLA-A03:10',
    'HLA-A03:12',
    'HLA-A03:13',
    'HLA-A03:14',
    'HLA-A03:15',
    'HLA-A03:16',
    'HLA-A03:17',
    'HLA-A03:18',
    'HLA-A03:19',
    'HLA-A03:20',
    'HLA-A03:22',
    'HLA-A03:23',
    'HLA-A03:24',
    'HLA-A03:25',
    'HLA-A03:26',
    'HLA-A03:27',
    'HLA-A03:28',
    'HLA-A03:29',
    'HLA-A03:30',
    'HLA-A03:31',
    'HLA-A03:32',
    'HLA-A03:33',
    'HLA-A03:34',
    'HLA-A03:35',
    'HLA-A03:37',
    'HLA-A03:38',
    'HLA-A03:39',
    'HLA-A03:40',
    'HLA-A03:41',
    'HLA-A03:42',
    'HLA-A03:43',
    'HLA-A03:44',
    'HLA-A03:45',
    'HLA-A03:46',
    'HLA-A03:47',
    'HLA-A03:48',
    'HLA-A03:49',
    'HLA-A03:50',
    'HLA-A03:51',
    'HLA-A03:52',
    'HLA-A03:53',
    'HLA-A03:54',
    'HLA-A03:55',
    'HLA-A03:56',
    'HLA-A03:57',
    'HLA-A03:58',
    'HLA-A03:59',
    'HLA-A03:60',
    'HLA-A03:61',
    'HLA-A03:62',
    'HLA-A03:63',
    'HLA-A03:64',
    'HLA-A03:65',
    'HLA-A03:66',
    'HLA-A03:67',
    'HLA-A03:70',
    'HLA-A03:71',
    'HLA-A03:72',
    'HLA-A03:73',
    'HLA-A03:74',
    'HLA-A03:75',
    'HLA-A03:76',
    'HLA-A03:77',
    'HLA-A03:78',
    'HLA-A03:79',
    'HLA-A03:80',
    'HLA-A03:81',
    'HLA-A03:82',
    'HLA-A11:01',
    'HLA-A11:02',
    'HLA-A11:03',
    'HLA-A11:04',
    'HLA-A11:05',
    'HLA-A11:06',
    'HLA-A11:07',
    'HLA-A11:08',
    'HLA-A11:09',
    'HLA-A11:10',
    'HLA-A11:11',
    'HLA-A11:12',
    'HLA-A11:13',
    'HLA-A11:14',
    'HLA-A11:15',
    'HLA-A11:16',
    'HLA-A11:17',
    'HLA-A11:18',
    'HLA-A11:19',
    'HLA-A11:20',
    'HLA-A11:22',
    'HLA-A11:23',
    'HLA-A11:24',
    'HLA-A11:25',
    'HLA-A11:26',
    'HLA-A11:27',
    'HLA-A11:29',
    'HLA-A11:30',
    'HLA-A11:31',
    'HLA-A11:32',
    'HLA-A11:33',
    'HLA-A11:34',
    'HLA-A11:35',
    'HLA-A11:36',
    'HLA-A11:37',
    'HLA-A11:38',
    'HLA-A11:39',
    'HLA-A11:40',
    'HLA-A11:41',
    'HLA-A11:42',
    'HLA-A11:43',
    'HLA-A11:44',
    'HLA-A11:45',
    'HLA-A11:46',
    'HLA-A11:47',
    'HLA-A11:48',
    'HLA-A11:49',
    'HLA-A11:51',
    'HLA-A11:53',
    'HLA-A11:54',
    'HLA-A11:55',
    'HLA-A11:56',
    'HLA-A11:57',
    'HLA-A11:58',
    'HLA-A11:59',
    'HLA-A11:60',
    'HLA-A11:61',
    'HLA-A11:62',
    'HLA-A11:63',
    'HLA-A11:64',
    'HLA-A23:01',
    'HLA-A23:02',
    'HLA-A23:03',
    'HLA-A23:04',
    'HLA-A23:05',
    'HLA-A23:06',
    'HLA-A23:09',
    'HLA-A23:10',
    'HLA-A23:12',
    'HLA-A23:13',
    'HLA-A23:14',
    'HLA-A23:15',
    'HLA-A23:16',
    'HLA-A23:17',
    'HLA-A23:18',
    'HLA-A23:20',
    'HLA-A23:21',
    'HLA-A23:22',
    'HLA-A23:23',
    'HLA-A23:24',
    'HLA-A23:25',
    'HLA-A23:26',
    'HLA-A24:02',
    'HLA-A24:03',
    'HLA-A24:04',
    'HLA-A24:05',
    'HLA-A24:06',
    'HLA-A24:07',
    'HLA-A24:08',
    'HLA-A24:10',
    'HLA-A24:13',
    'HLA-A24:14',
    'HLA-A24:15',
    'HLA-A24:17',
    'HLA-A24:18',
    'HLA-A24:19',
    'HLA-A24:20',
    'HLA-A24:21',
    'HLA-A24:22',
    'HLA-A24:23',
    'HLA-A24:24',
    'HLA-A24:25',
    'HLA-A24:26',
    'HLA-A24:27',
    'HLA-A24:28',
    'HLA-A24:29',
    'HLA-A24:30',
    'HLA-A24:31',
    'HLA-A24:32',
    'HLA-A24:33',
    'HLA-A24:34',
    'HLA-A24:35',
    'HLA-A24:37',
    'HLA-A24:38',
    'HLA-A24:39',
    'HLA-A24:41',
    'HLA-A24:42',
    'HLA-A24:43',
    'HLA-A24:44',
    'HLA-A24:46',
    'HLA-A24:47',
    'HLA-A24:49',
    'HLA-A24:50',
    'HLA-A24:51',
    'HLA-A24:52',
    'HLA-A24:53',
    'HLA-A24:54',
    'HLA-A24:55',
    'HLA-A24:56',
    'HLA-A24:57',
    'HLA-A24:58',
    'HLA-A24:59',
    'HLA-A24:61',
    'HLA-A24:62',
    'HLA-A24:63',
    'HLA-A24:64',
    'HLA-A24:66',
    'HLA-A24:67',
    'HLA-A24:68',
    'HLA-A24:69',
    'HLA-A24:70',
    'HLA-A24:71',
    'HLA-A24:72',
    'HLA-A24:73',
    'HLA-A24:74',
    'HLA-A24:75',
    'HLA-A24:76',
    'HLA-A24:77',
    'HLA-A24:78',
    'HLA-A24:79',
    'HLA-A24:80',
    'HLA-A24:81',
    'HLA-A24:82',
    'HLA-A24:85',
    'HLA-A24:87',
    'HLA-A24:88',
    'HLA-A24:89',
    'HLA-A24:91',
    'HLA-A24:92',
    'HLA-A24:93',
    'HLA-A24:94',
    'HLA-A24:95',
    'HLA-A24:96',
    'HLA-A24:97',
    'HLA-A24:98',
    'HLA-A24:99',
    'HLA-A24:100',
    'HLA-A24:101',
    'HLA-A24:102',
    'HLA-A24:103',
    'HLA-A24:104',
    'HLA-A24:105',
    'HLA-A24:106',
    'HLA-A24:107',
    'HLA-A24:108',
    'HLA-A24:109',
    'HLA-A24:110',
    'HLA-A24:111',
    'HLA-A24:112',
    'HLA-A24:113',
    'HLA-A24:114',
    'HLA-A24:115',
    'HLA-A24:116',
    'HLA-A24:117',
    'HLA-A24:118',
    'HLA-A24:119',
    'HLA-A24:120',
    'HLA-A24:121',
    'HLA-A24:122',
    'HLA-A24:123',
    'HLA-A24:124',
    'HLA-A24:125',
    'HLA-A24:126',
    'HLA-A24:127',
    'HLA-A24:128',
    'HLA-A24:129',
    'HLA-A24:130',
    'HLA-A24:131',
    'HLA-A24:133',
    'HLA-A24:134',
    'HLA-A24:135',
    'HLA-A24:136',
    'HLA-A24:137',
    'HLA-A24:138',
    'HLA-A24:139',
    'HLA-A24:140',
    'HLA-A24:141',
    'HLA-A24:142',
    'HLA-A24:143',
    'HLA-A24:144',
    'HLA-A25:01',
    'HLA-A25:02',
    'HLA-A25:03',
    'HLA-A25:04',
    'HLA-A25:05',
    'HLA-A25:06',
    'HLA-A25:07',
    'HLA-A25:08',
    'HLA-A25:09',
    'HLA-A25:10',
    'HLA-A25:11',
    'HLA-A25:13',
    'HLA-A26:01',
    'HLA-A26:02',
    'HLA-A26:03',
    'HLA-A26:04',
    'HLA-A26:05',
    'HLA-A26:06',
    'HLA-A26:07',
    'HLA-A26:08',
    'HLA-A26:09',
    'HLA-A26:10',
    'HLA-A26:12',
    'HLA-A26:13',
    'HLA-A26:14',
    'HLA-A26:15',
    'HLA-A26:16',
    'HLA-A26:17',
    'HLA-A26:18',
    'HLA-A26:19',
    'HLA-A26:20',
    'HLA-A26:21',
    'HLA-A26:22',
    'HLA-A26:23',
    'HLA-A26:24',
    'HLA-A26:26',
    'HLA-A26:27',
    'HLA-A26:28',
    'HLA-A26:29',
    'HLA-A26:30',
    'HLA-A26:31',
    'HLA-A26:32',
    'HLA-A26:33',
    'HLA-A26:34',
    'HLA-A26:35',
    'HLA-A26:36',
    'HLA-A26:37',
    'HLA-A26:38',
    'HLA-A26:39',
    'HLA-A26:40',
    'HLA-A26:41',
    'HLA-A26:42',
    'HLA-A26:43',
    'HLA-A26:45',
    'HLA-A26:46',
    'HLA-A26:47',
    'HLA-A26:48',
    'HLA-A26:49',
    'HLA-A26:50',
    'HLA-A29:01',
    'HLA-A29:02',
    'HLA-A29:03',
    'HLA-A29:04',
    'HLA-A29:05',
    'HLA-A29:06',
    'HLA-A29:07',
    'HLA-A29:09',
    'HLA-A29:10',
    'HLA-A29:11',
    'HLA-A29:12',
    'HLA-A29:13',
    'HLA-A29:14',
    'HLA-A29:15',
    'HLA-A29:16',
    'HLA-A29:17',
    'HLA-A29:18',
    'HLA-A29:19',
    'HLA-A29:20',
    'HLA-A29:21',
    'HLA-A29:22',
    'HLA-A30:01',
    'HLA-A30:02',
    'HLA-A30:03',
    'HLA-A30:04',
    'HLA-A30:06',
    'HLA-A30:07',
    'HLA-A30:08',
    'HLA-A30:09',
    'HLA-A30:10',
    'HLA-A30:11',
    'HLA-A30:12',
    'HLA-A30:13',
    'HLA-A30:15',
    'HLA-A30:16',
    'HLA-A30:17',
    'HLA-A30:18',
    'HLA-A30:19',
    'HLA-A30:20',
    'HLA-A30:22',
    'HLA-A30:23',
    'HLA-A30:24',
    'HLA-A30:25',
    'HLA-A30:26',
    'HLA-A30:28',
    'HLA-A30:29',
    'HLA-A30:30',
    'HLA-A30:31',
    'HLA-A30:32',
    'HLA-A30:33',
    'HLA-A30:34',
    'HLA-A30:35',
    'HLA-A30:36',
    'HLA-A30:37',
    'HLA-A30:38',
    'HLA-A30:39',
    'HLA-A30:40',
    'HLA-A30:41',
    'HLA-A31:01',
    'HLA-A31:02',
    'HLA-A31:03',
    'HLA-A31:04',
    'HLA-A31:05',
    'HLA-A31:06',
    'HLA-A31:07',
    'HLA-A31:08',
    'HLA-A31:09',
    'HLA-A31:10',
    'HLA-A31:11',
    'HLA-A31:12',
    'HLA-A31:13',
    'HLA-A31:15',
    'HLA-A31:16',
    'HLA-A31:17',
    'HLA-A31:18',
    'HLA-A31:19',
    'HLA-A31:20',
    'HLA-A31:21',
    'HLA-A31:22',
    'HLA-A31:23',
    'HLA-A31:24',
    'HLA-A31:25',
    'HLA-A31:26',
    'HLA-A31:27',
    'HLA-A31:28',
    'HLA-A31:29',
    'HLA-A31:30',
    'HLA-A31:31',
    'HLA-A31:32',
    'HLA-A31:33',
    'HLA-A31:34',
    'HLA-A31:35',
    'HLA-A31:36',
    'HLA-A31:37',
    'HLA-A32:01',
    'HLA-A32:02',
    'HLA-A32:03',
    'HLA-A32:04',
    'HLA-A32:05',
    'HLA-A32:06',
    'HLA-A32:07',
    'HLA-A32:08',
    'HLA-A32:09',
    'HLA-A32:10',
    'HLA-A32:12',
    'HLA-A32:13',
    'HLA-A32:14',
    'HLA-A32:15',
    'HLA-A32:16',
    'HLA-A32:17',
    'HLA-A32:18',
    'HLA-A32:20',
    'HLA-A32:21',
    'HLA-A32:22',
    'HLA-A32:23',
    'HLA-A32:24',
    'HLA-A32:25',
    'HLA-A33:01',
    'HLA-A33:03',
    'HLA-A33:04',
    'HLA-A33:05',
    'HLA-A33:06',
    'HLA-A33:07',
    'HLA-A33:08',
    'HLA-A33:09',
    'HLA-A33:10',
    'HLA-A33:11',
    'HLA-A33:12',
    'HLA-A33:13',
    'HLA-A33:14',
    'HLA-A33:15',
    'HLA-A33:16',
    'HLA-A33:17',
    'HLA-A33:18',
    'HLA-A33:19',
    'HLA-A33:20',
    'HLA-A33:21',
    'HLA-A33:22',
    'HLA-A33:23',
    'HLA-A33:24',
    'HLA-A33:25',
    'HLA-A33:26',
    'HLA-A33:27',
    'HLA-A33:28',
    'HLA-A33:29',
    'HLA-A33:30',
    'HLA-A33:31',
    'HLA-A34:01',
    'HLA-A34:02',
    'HLA-A34:03',
    'HLA-A34:04',
    'HLA-A34:05',
    'HLA-A34:06',
    'HLA-A34:07',
    'HLA-A34:08',
    'HLA-A36:01',
    'HLA-A36:02',
    'HLA-A36:03',
    'HLA-A36:04',
    'HLA-A36:05',
    'HLA-A43:01',
    'HLA-A66:01',
    'HLA-A66:02',
    'HLA-A66:03',
    'HLA-A66:04',
    'HLA-A66:05',
    'HLA-A66:06',
    'HLA-A66:07',
    'HLA-A66:08',
    'HLA-A66:09',
    'HLA-A66:10',
    'HLA-A66:11',
    'HLA-A66:12',
    'HLA-A66:13',
    'HLA-A66:14',
    'HLA-A66:15',
    'HLA-A68:01',
    'HLA-A68:02',
    'HLA-A68:03',
    'HLA-A68:04',
    'HLA-A68:05',
    'HLA-A68:06',
    'HLA-A68:07',
    'HLA-A68:08',
    'HLA-A68:09',
    'HLA-A68:10',
    'HLA-A68:12',
    'HLA-A68:13',
    'HLA-A68:14',
    'HLA-A68:15',
    'HLA-A68:16',
    'HLA-A68:17',
    'HLA-A68:19',
    'HLA-A68:20',
    'HLA-A68:21',
    'HLA-A68:22',
    'HLA-A68:23',
    'HLA-A68:24',
    'HLA-A68:25',
    'HLA-A68:26',
    'HLA-A68:27',
    'HLA-A68:28',
    'HLA-A68:29',
    'HLA-A68:30',
    'HLA-A68:31',
    'HLA-A68:32',
    'HLA-A68:33',
    'HLA-A68:34',
    'HLA-A68:35',
    'HLA-A68:36',
    'HLA-A68:37',
    'HLA-A68:38',
    'HLA-A68:39',
    'HLA-A68:40',
    'HLA-A68:41',
    'HLA-A68:42',
    'HLA-A68:43',
    'HLA-A68:44',
    'HLA-A68:45',
    'HLA-A68:46',
    'HLA-A68:47',
    'HLA-A68:48',
    'HLA-A68:50',
    'HLA-A68:51',
    'HLA-A68:52',
    'HLA-A68:53',
    'HLA-A68:54',
    'HLA-A69:01',
    'HLA-A74:01',
    'HLA-A74:02',
    'HLA-A74:03',
    'HLA-A74:04',
    'HLA-A74:05',
    'HLA-A74:06',
    'HLA-A74:07',
    'HLA-A74:08',
    'HLA-A74:09',
    'HLA-A74:10',
    'HLA-A74:11',
    'HLA-A74:13',
    'HLA-A80:01',
    'HLA-A80:02',
    'HLA-B07:02',
    'HLA-B07:03',
    'HLA-B07:04',
    'HLA-B07:05',
    'HLA-B07:06',
    'HLA-B07:07',
    'HLA-B07:08',
    'HLA-B07:09',
    'HLA-B07:10',
    'HLA-B07:11',
    'HLA-B07:12',
    'HLA-B07:13',
    'HLA-B07:14',
    'HLA-B07:15',
    'HLA-B07:16',
    'HLA-B07:17',
    'HLA-B07:18',
    'HLA-B07:19',
    'HLA-B07:20',
    'HLA-B07:21',
    'HLA-B07:22',
    'HLA-B07:23',
    'HLA-B07:24',
    'HLA-B07:25',
    'HLA-B07:26',
    'HLA-B07:27',
    'HLA-B07:28',
    'HLA-B07:29',
    'HLA-B07:30',
    'HLA-B07:31',
    'HLA-B07:32',
    'HLA-B07:33',
    'HLA-B07:34',
    'HLA-B07:35',
    'HLA-B07:36',
    'HLA-B07:37',
    'HLA-B07:38',
    'HLA-B07:39',
    'HLA-B07:40',
    'HLA-B07:41',
    'HLA-B07:42',
    'HLA-B07:43',
    'HLA-B07:44',
    'HLA-B07:45',
    'HLA-B07:46',
    'HLA-B07:47',
    'HLA-B07:48',
    'HLA-B07:50',
    'HLA-B07:51',
    'HLA-B07:52',
    'HLA-B07:53',
    'HLA-B07:54',
    'HLA-B07:55',
    'HLA-B07:56',
    'HLA-B07:57',
    'HLA-B07:58',
    'HLA-B07:59',
    'HLA-B07:60',
    'HLA-B07:61',
    'HLA-B07:62',
    'HLA-B07:63',
    'HLA-B07:64',
    'HLA-B07:65',
    'HLA-B07:66',
    'HLA-B07:68',
    'HLA-B07:69',
    'HLA-B07:70',
    'HLA-B07:71',
    'HLA-B07:72',
    'HLA-B07:73',
    'HLA-B07:74',
    'HLA-B07:75',
    'HLA-B07:76',
    'HLA-B07:77',
    'HLA-B07:78',
    'HLA-B07:79',
    'HLA-B07:80',
    'HLA-B07:81',
    'HLA-B07:82',
    'HLA-B07:83',
    'HLA-B07:84',
    'HLA-B07:85',
    'HLA-B07:86',
    'HLA-B07:87',
    'HLA-B07:88',
    'HLA-B07:89',
    'HLA-B07:90',
    'HLA-B07:91',
    'HLA-B07:92',
    'HLA-B07:93',
    'HLA-B07:94',
    'HLA-B07:95',
    'HLA-B07:96',
    'HLA-B07:97',
    'HLA-B07:98',
    'HLA-B07:99',
    'HLA-B07:100',
    'HLA-B07:101',
    'HLA-B07:102',
    'HLA-B07:103',
    'HLA-B07:104',
    'HLA-B07:105',
    'HLA-B07:106',
    'HLA-B07:107',
    'HLA-B07:108',
    'HLA-B07:109',
    'HLA-B07:110',
    'HLA-B07:112',
    'HLA-B07:113',
    'HLA-B07:114',
    'HLA-B07:115',
    'HLA-B08:01',
    'HLA-B08:02',
    'HLA-B08:03',
    'HLA-B08:04',
    'HLA-B08:05',
    'HLA-B08:07',
    'HLA-B08:09',
    'HLA-B08:10',
    'HLA-B08:11',
    'HLA-B08:12',
    'HLA-B08:13',
    'HLA-B08:14',
    'HLA-B08:15',
    'HLA-B08:16',
    'HLA-B08:17',
    'HLA-B08:18',
    'HLA-B08:20',
    'HLA-B08:21',
    'HLA-B08:22',
    'HLA-B08:23',
    'HLA-B08:24',
    'HLA-B08:25',
    'HLA-B08:26',
    'HLA-B08:27',
    'HLA-B08:28',
    'HLA-B08:29',
    'HLA-B08:31',
    'HLA-B08:32',
    'HLA-B08:33',
    'HLA-B08:34',
    'HLA-B08:35',
    'HLA-B08:36',
    'HLA-B08:37',
    'HLA-B08:38',
    'HLA-B08:39',
    'HLA-B08:40',
    'HLA-B08:41',
    'HLA-B08:42',
    'HLA-B08:43',
    'HLA-B08:44',
    'HLA-B08:45',
    'HLA-B08:46',
    'HLA-B08:47',
    'HLA-B08:48',
    'HLA-B08:49',
    'HLA-B08:50',
    'HLA-B08:51',
    'HLA-B08:52',
    'HLA-B08:53',
    'HLA-B08:54',
    'HLA-B08:55',
    'HLA-B08:56',
    'HLA-B08:57',
    'HLA-B08:58',
    'HLA-B08:59',
    'HLA-B08:60',
    'HLA-B08:61',
    'HLA-B08:62',
    'HLA-B13:01',
    'HLA-B13:02',
    'HLA-B13:03',
    'HLA-B13:04',
    'HLA-B13:06',
    'HLA-B13:09',
    'HLA-B13:10',
    'HLA-B13:11',
    'HLA-B13:12',
    'HLA-B13:13',
    'HLA-B13:14',
    'HLA-B13:15',
    'HLA-B13:16',
    'HLA-B13:17',
    'HLA-B13:18',
    'HLA-B13:19',
    'HLA-B13:20',
    'HLA-B13:21',
    'HLA-B13:22',
    'HLA-B13:23',
    'HLA-B13:25',
    'HLA-B13:26',
    'HLA-B13:27',
    'HLA-B13:28',
    'HLA-B13:29',
    'HLA-B13:30',
    'HLA-B13:31',
    'HLA-B13:32',
    'HLA-B13:33',
    'HLA-B13:34',
    'HLA-B13:35',
    'HLA-B13:36',
    'HLA-B13:37',
    'HLA-B13:38',
    'HLA-B13:39',
    'HLA-B14:01',
    'HLA-B14:02',
    'HLA-B14:03',
    'HLA-B14:04',
    'HLA-B14:05',
    'HLA-B14:06',
    'HLA-B14:08',
    'HLA-B14:09',
    'HLA-B14:10',
    'HLA-B14:11',
    'HLA-B14:12',
    'HLA-B14:13',
    'HLA-B14:14',
    'HLA-B14:15',
    'HLA-B14:16',
    'HLA-B14:17',
    'HLA-B14:18',
    'HLA-B15:01',
    'HLA-B15:02',
    'HLA-B15:03',
    'HLA-B15:04',
    'HLA-B15:05',
    'HLA-B15:06',
    'HLA-B15:07',
    'HLA-B15:08',
    'HLA-B15:09',
    'HLA-B15:10',
    'HLA-B15:11',
    'HLA-B15:12',
    'HLA-B15:13',
    'HLA-B15:14',
    'HLA-B15:15',
    'HLA-B15:16',
    'HLA-B15:17',
    'HLA-B15:18',
    'HLA-B15:19',
    'HLA-B15:20',
    'HLA-B15:21',
    'HLA-B15:23',
    'HLA-B15:24',
    'HLA-B15:25',
    'HLA-B15:27',
    'HLA-B15:28',
    'HLA-B15:29',
    'HLA-B15:30',
    'HLA-B15:31',
    'HLA-B15:32',
    'HLA-B15:33',
    'HLA-B15:34',
    'HLA-B15:35',
    'HLA-B15:36',
    'HLA-B15:37',
    'HLA-B15:38',
    'HLA-B15:39',
    'HLA-B15:40',
    'HLA-B15:42',
    'HLA-B15:43',
    'HLA-B15:44',
    'HLA-B15:45',
    'HLA-B15:46',
    'HLA-B15:47',
    'HLA-B15:48',
    'HLA-B15:49',
    'HLA-B15:50',
    'HLA-B15:51',
    'HLA-B15:52',
    'HLA-B15:53',
    'HLA-B15:54',
    'HLA-B15:55',
    'HLA-B15:56',
    'HLA-B15:57',
    'HLA-B15:58',
    'HLA-B15:60',
    'HLA-B15:61',
    'HLA-B15:62',
    'HLA-B15:63',
    'HLA-B15:64',
    'HLA-B15:65',
    'HLA-B15:66',
    'HLA-B15:67',
    'HLA-B15:68',
    'HLA-B15:69',
    'HLA-B15:70',
    'HLA-B15:71',
    'HLA-B15:72',
    'HLA-B15:73',
    'HLA-B15:74',
    'HLA-B15:75',
    'HLA-B15:76',
    'HLA-B15:77',
    'HLA-B15:78',
    'HLA-B15:80',
    'HLA-B15:81',
    'HLA-B15:82',
    'HLA-B15:83',
    'HLA-B15:84',
    'HLA-B15:85',
    'HLA-B15:86',
    'HLA-B15:87',
    'HLA-B15:88',
    'HLA-B15:89',
    'HLA-B15:90',
    'HLA-B15:91',
    'HLA-B15:92',
    'HLA-B15:93',
    'HLA-B15:95',
    'HLA-B15:96',
    'HLA-B15:97',
    'HLA-B15:98',
    'HLA-B15:99',
    'HLA-B15:101',
    'HLA-B15:102',
    'HLA-B15:103',
    'HLA-B15:104',
    'HLA-B15:105',
    'HLA-B15:106',
    'HLA-B15:107',
    'HLA-B15:108',
    'HLA-B15:109',
    'HLA-B15:110',
    'HLA-B15:112',
    'HLA-B15:113',
    'HLA-B15:114',
    'HLA-B15:115',
    'HLA-B15:116',
    'HLA-B15:117',
    'HLA-B15:118',
    'HLA-B15:119',
    'HLA-B15:120',
    'HLA-B15:121',
    'HLA-B15:122',
    'HLA-B15:123',
    'HLA-B15:124',
    'HLA-B15:125',
    'HLA-B15:126',
    'HLA-B15:127',
    'HLA-B15:128',
    'HLA-B15:129',
    'HLA-B15:131',
    'HLA-B15:132',
    'HLA-B15:133',
    'HLA-B15:134',
    'HLA-B15:135',
    'HLA-B15:136',
    'HLA-B15:137',
    'HLA-B15:138',
    'HLA-B15:139',
    'HLA-B15:140',
    'HLA-B15:141',
    'HLA-B15:142',
    'HLA-B15:143',
    'HLA-B15:144',
    'HLA-B15:145',
    'HLA-B15:146',
    'HLA-B15:147',
    'HLA-B15:148',
    'HLA-B15:150',
    'HLA-B15:151',
    'HLA-B15:152',
    'HLA-B15:153',
    'HLA-B15:154',
    'HLA-B15:155',
    'HLA-B15:156',
    'HLA-B15:157',
    'HLA-B15:158',
    'HLA-B15:159',
    'HLA-B15:160',
    'HLA-B15:161',
    'HLA-B15:162',
    'HLA-B15:163',
    'HLA-B15:164',
    'HLA-B15:165',
    'HLA-B15:166',
    'HLA-B15:167',
    'HLA-B15:168',
    'HLA-B15:169',
    'HLA-B15:170',
    'HLA-B15:171',
    'HLA-B15:172',
    'HLA-B15:173',
    'HLA-B15:174',
    'HLA-B15:175',
    'HLA-B15:176',
    'HLA-B15:177',
    'HLA-B15:178',
    'HLA-B15:179',
    'HLA-B15:180',
    'HLA-B15:183',
    'HLA-B15:184',
    'HLA-B15:185',
    'HLA-B15:186',
    'HLA-B15:187',
    'HLA-B15:188',
    'HLA-B15:189',
    'HLA-B15:191',
    'HLA-B15:192',
    'HLA-B15:193',
    'HLA-B15:194',
    'HLA-B15:195',
    'HLA-B15:196',
    'HLA-B15:197',
    'HLA-B15:198',
    'HLA-B15:199',
    'HLA-B15:200',
    'HLA-B15:201',
    'HLA-B15:202',
    'HLA-B18:01',
    'HLA-B18:02',
    'HLA-B18:03',
    'HLA-B18:04',
    'HLA-B18:05',
    'HLA-B18:06',
    'HLA-B18:07',
    'HLA-B18:08',
    'HLA-B18:09',
    'HLA-B18:10',
    'HLA-B18:11',
    'HLA-B18:12',
    'HLA-B18:13',
    'HLA-B18:14',
    'HLA-B18:15',
    'HLA-B18:18',
    'HLA-B18:19',
    'HLA-B18:20',
    'HLA-B18:21',
    'HLA-B18:22',
    'HLA-B18:24',
    'HLA-B18:25',
    'HLA-B18:26',
    'HLA-B18:27',
    'HLA-B18:28',
    'HLA-B18:29',
    'HLA-B18:30',
    'HLA-B18:31',
    'HLA-B18:32',
    'HLA-B18:33',
    'HLA-B18:34',
    'HLA-B18:35',
    'HLA-B18:36',
    'HLA-B18:37',
    'HLA-B18:38',
    'HLA-B18:39',
    'HLA-B18:40',
    'HLA-B18:41',
    'HLA-B18:42',
    'HLA-B18:43',
    'HLA-B18:44',
    'HLA-B18:45',
    'HLA-B18:46',
    'HLA-B18:47',
    'HLA-B18:48',
    'HLA-B18:49',
    'HLA-B18:50',
    'HLA-B27:01',
    'HLA-B27:02',
    'HLA-B27:03',
    'HLA-B27:04',
    'HLA-B27:05',
    'HLA-B27:06',
    'HLA-B27:07',
    'HLA-B27:08',
    'HLA-B27:09',
    'HLA-B27:10',
    'HLA-B27:11',
    'HLA-B27:12',
    'HLA-B27:13',
    'HLA-B27:14',
    'HLA-B27:15',
    'HLA-B27:16',
    'HLA-B27:17',
    'HLA-B27:18',
    'HLA-B27:19',
    'HLA-B27:20',
    'HLA-B27:21',
    'HLA-B27:23',
    'HLA-B27:24',
    'HLA-B27:25',
    'HLA-B27:26',
    'HLA-B27:27',
    'HLA-B27:28',
    'HLA-B27:29',
    'HLA-B27:30',
    'HLA-B27:31',
    'HLA-B27:32',
    'HLA-B27:33',
    'HLA-B27:34',
    'HLA-B27:35',
    'HLA-B27:36',
    'HLA-B27:37',
    'HLA-B27:38',
    'HLA-B27:39',
    'HLA-B27:40',
    'HLA-B27:41',
    'HLA-B27:42',
    'HLA-B27:43',
    'HLA-B27:44',
    'HLA-B27:45',
    'HLA-B27:46',
    'HLA-B27:47',
    'HLA-B27:48',
    'HLA-B27:49',
    'HLA-B27:50',
    'HLA-B27:51',
    'HLA-B27:52',
    'HLA-B27:53',
    'HLA-B27:54',
    'HLA-B27:55',
    'HLA-B27:56',
    'HLA-B27:57',
    'HLA-B27:58',
    'HLA-B27:60',
    'HLA-B27:61',
    'HLA-B27:62',
    'HLA-B27:63',
    'HLA-B27:67',
    'HLA-B27:68',
    'HLA-B27:69',
    'HLA-B35:01',
    'HLA-B35:02',
    'HLA-B35:03',
    'HLA-B35:04',
    'HLA-B35:05',
    'HLA-B35:06',
    'HLA-B35:07',
    'HLA-B35:08',
    'HLA-B35:09',
    'HLA-B35:10',
    'HLA-B35:11',
    'HLA-B35:12',
    'HLA-B35:13',
    'HLA-B35:14',
    'HLA-B35:15',
    'HLA-B35:16',
    'HLA-B35:17',
    'HLA-B35:18',
    'HLA-B35:19',
    'HLA-B35:20',
    'HLA-B35:21',
    'HLA-B35:22',
    'HLA-B35:23',
    'HLA-B35:24',
    'HLA-B35:25',
    'HLA-B35:26',
    'HLA-B35:27',
    'HLA-B35:28',
    'HLA-B35:29',
    'HLA-B35:30',
    'HLA-B35:31',
    'HLA-B35:32',
    'HLA-B35:33',
    'HLA-B35:34',
    'HLA-B35:35',
    'HLA-B35:36',
    'HLA-B35:37',
    'HLA-B35:38',
    'HLA-B35:39',
    'HLA-B35:41',
    'HLA-B35:42',
    'HLA-B35:43',
    'HLA-B35:44',
    'HLA-B35:45',
    'HLA-B35:46',
    'HLA-B35:47',
    'HLA-B35:48',
    'HLA-B35:49',
    'HLA-B35:50',
    'HLA-B35:51',
    'HLA-B35:52',
    'HLA-B35:54',
    'HLA-B35:55',
    'HLA-B35:56',
    'HLA-B35:57',
    'HLA-B35:58',
    'HLA-B35:59',
    'HLA-B35:60',
    'HLA-B35:61',
    'HLA-B35:62',
    'HLA-B35:63',
    'HLA-B35:64',
    'HLA-B35:66',
    'HLA-B35:67',
    'HLA-B35:68',
    'HLA-B35:69',
    'HLA-B35:70',
    'HLA-B35:71',
    'HLA-B35:72',
    'HLA-B35:74',
    'HLA-B35:75',
    'HLA-B35:76',
    'HLA-B35:77',
    'HLA-B35:78',
    'HLA-B35:79',
    'HLA-B35:80',
    'HLA-B35:81',
    'HLA-B35:82',
    'HLA-B35:83',
    'HLA-B35:84',
    'HLA-B35:85',
    'HLA-B35:86',
    'HLA-B35:87',
    'HLA-B35:88',
    'HLA-B35:89',
    'HLA-B35:90',
    'HLA-B35:91',
    'HLA-B35:92',
    'HLA-B35:93',
    'HLA-B35:94',
    'HLA-B35:95',
    'HLA-B35:96',
    'HLA-B35:97',
    'HLA-B35:98',
    'HLA-B35:99',
    'HLA-B35:100',
    'HLA-B35:101',
    'HLA-B35:102',
    'HLA-B35:103',
    'HLA-B35:104',
    'HLA-B35:105',
    'HLA-B35:106',
    'HLA-B35:107',
    'HLA-B35:108',
    'HLA-B35:109',
    'HLA-B35:110',
    'HLA-B35:111',
    'HLA-B35:112',
    'HLA-B35:113',
    'HLA-B35:114',
    'HLA-B35:115',
    'HLA-B35:116',
    'HLA-B35:117',
    'HLA-B35:118',
    'HLA-B35:119',
    'HLA-B35:120',
    'HLA-B35:121',
    'HLA-B35:122',
    'HLA-B35:123',
    'HLA-B35:124',
    'HLA-B35:125',
    'HLA-B35:126',
    'HLA-B35:127',
    'HLA-B35:128',
    'HLA-B35:131',
    'HLA-B35:132',
    'HLA-B35:133',
    'HLA-B35:135',
    'HLA-B35:136',
    'HLA-B35:137',
    'HLA-B35:138',
    'HLA-B35:139',
    'HLA-B35:140',
    'HLA-B35:141',
    'HLA-B35:142',
    'HLA-B35:143',
    'HLA-B35:144',
    'HLA-B37:01',
    'HLA-B37:02',
    'HLA-B37:04',
    'HLA-B37:05',
    'HLA-B37:06',
    'HLA-B37:07',
    'HLA-B37:08',
    'HLA-B37:09',
    'HLA-B37:10',
    'HLA-B37:11',
    'HLA-B37:12',
    'HLA-B37:13',
    'HLA-B37:14',
    'HLA-B37:15',
    'HLA-B37:17',
    'HLA-B37:18',
    'HLA-B37:19',
    'HLA-B37:20',
    'HLA-B37:21',
    'HLA-B37:22',
    'HLA-B37:23',
    'HLA-B38:01',
    'HLA-B38:02',
    'HLA-B38:03',
    'HLA-B38:04',
    'HLA-B38:05',
    'HLA-B38:06',
    'HLA-B38:07',
    'HLA-B38:08',
    'HLA-B38:09',
    'HLA-B38:10',
    'HLA-B38:11',
    'HLA-B38:12',
    'HLA-B38:13',
    'HLA-B38:14',
    'HLA-B38:15',
    'HLA-B38:16',
    'HLA-B38:17',
    'HLA-B38:18',
    'HLA-B38:19',
    'HLA-B38:20',
    'HLA-B38:21',
    'HLA-B38:22',
    'HLA-B38:23',
    'HLA-B39:01',
    'HLA-B39:02',
    'HLA-B39:03',
    'HLA-B39:04',
    'HLA-B39:05',
    'HLA-B39:06',
    'HLA-B39:07',
    'HLA-B39:08',
    'HLA-B39:09',
    'HLA-B39:10',
    'HLA-B39:11',
    'HLA-B39:12',
    'HLA-B39:13',
    'HLA-B39:14',
    'HLA-B39:15',
    'HLA-B39:16',
    'HLA-B39:17',
    'HLA-B39:18',
    'HLA-B39:19',
    'HLA-B39:20',
    'HLA-B39:22',
    'HLA-B39:23',
    'HLA-B39:24',
    'HLA-B39:26',
    'HLA-B39:27',
    'HLA-B39:28',
    'HLA-B39:29',
    'HLA-B39:30',
    'HLA-B39:31',
    'HLA-B39:32',
    'HLA-B39:33',
    'HLA-B39:34',
    'HLA-B39:35',
    'HLA-B39:36',
    'HLA-B39:37',
    'HLA-B39:39',
    'HLA-B39:41',
    'HLA-B39:42',
    'HLA-B39:43',
    'HLA-B39:44',
    'HLA-B39:45',
    'HLA-B39:46',
    'HLA-B39:47',
    'HLA-B39:48',
    'HLA-B39:49',
    'HLA-B39:50',
    'HLA-B39:51',
    'HLA-B39:52',
    'HLA-B39:53',
    'HLA-B39:54',
    'HLA-B39:55',
    'HLA-B39:56',
    'HLA-B39:57',
    'HLA-B39:58',
    'HLA-B39:59',
    'HLA-B39:60',
    'HLA-B40:01',
    'HLA-B40:02',
    'HLA-B40:03',
    'HLA-B40:04',
    'HLA-B40:05',
    'HLA-B40:06',
    'HLA-B40:07',
    'HLA-B40:08',
    'HLA-B40:09',
    'HLA-B40:10',
    'HLA-B40:11',
    'HLA-B40:12',
    'HLA-B40:13',
    'HLA-B40:14',
    'HLA-B40:15',
    'HLA-B40:16',
    'HLA-B40:18',
    'HLA-B40:19',
    'HLA-B40:20',
    'HLA-B40:21',
    'HLA-B40:23',
    'HLA-B40:24',
    'HLA-B40:25',
    'HLA-B40:26',
    'HLA-B40:27',
    'HLA-B40:28',
    'HLA-B40:29',
    'HLA-B40:30',
    'HLA-B40:31',
    'HLA-B40:32',
    'HLA-B40:33',
    'HLA-B40:34',
    'HLA-B40:35',
    'HLA-B40:36',
    'HLA-B40:37',
    'HLA-B40:38',
    'HLA-B40:39',
    'HLA-B40:40',
    'HLA-B40:42',
    'HLA-B40:43',
    'HLA-B40:44',
    'HLA-B40:45',
    'HLA-B40:46',
    'HLA-B40:47',
    'HLA-B40:48',
    'HLA-B40:49',
    'HLA-B40:50',
    'HLA-B40:51',
    'HLA-B40:52',
    'HLA-B40:53',
    'HLA-B40:54',
    'HLA-B40:55',
    'HLA-B40:56',
    'HLA-B40:57',
    'HLA-B40:58',
    'HLA-B40:59',
    'HLA-B40:60',
    'HLA-B40:61',
    'HLA-B40:62',
    'HLA-B40:63',
    'HLA-B40:64',
    'HLA-B40:65',
    'HLA-B40:66',
    'HLA-B40:67',
    'HLA-B40:68',
    'HLA-B40:69',
    'HLA-B40:70',
    'HLA-B40:71',
    'HLA-B40:72',
    'HLA-B40:73',
    'HLA-B40:74',
    'HLA-B40:75',
    'HLA-B40:76',
    'HLA-B40:77',
    'HLA-B40:78',
    'HLA-B40:79',
    'HLA-B40:80',
    'HLA-B40:81',
    'HLA-B40:82',
    'HLA-B40:83',
    'HLA-B40:84',
    'HLA-B40:85',
    'HLA-B40:86',
    'HLA-B40:87',
    'HLA-B40:88',
    'HLA-B40:89',
    'HLA-B40:90',
    'HLA-B40:91',
    'HLA-B40:92',
    'HLA-B40:93',
    'HLA-B40:94',
    'HLA-B40:95',
    'HLA-B40:96',
    'HLA-B40:97',
    'HLA-B40:98',
    'HLA-B40:99',
    'HLA-B40:100',
    'HLA-B40:101',
    'HLA-B40:102',
    'HLA-B40:103',
    'HLA-B40:104',
    'HLA-B40:105',
    'HLA-B40:106',
    'HLA-B40:107',
    'HLA-B40:108',
    'HLA-B40:109',
    'HLA-B40:110',
    'HLA-B40:111',
    'HLA-B40:112',
    'HLA-B40:113',
    'HLA-B40:114',
    'HLA-B40:115',
    'HLA-B40:116',
    'HLA-B40:117',
    'HLA-B40:119',
    'HLA-B40:120',
    'HLA-B40:121',
    'HLA-B40:122',
    'HLA-B40:123',
    'HLA-B40:124',
    'HLA-B40:125',
    'HLA-B40:126',
    'HLA-B40:127',
    'HLA-B40:128',
    'HLA-B40:129',
    'HLA-B40:130',
    'HLA-B40:131',
    'HLA-B40:132',
    'HLA-B40:134',
    'HLA-B40:135',
    'HLA-B40:136',
    'HLA-B40:137',
    'HLA-B40:138',
    'HLA-B40:139',
    'HLA-B40:140',
    'HLA-B40:141',
    'HLA-B40:143',
    'HLA-B40:145',
    'HLA-B40:146',
    'HLA-B40:147',
    'HLA-B41:01',
    'HLA-B41:02',
    'HLA-B41:03',
    'HLA-B41:04',
    'HLA-B41:05',
    'HLA-B41:06',
    'HLA-B41:07',
    'HLA-B41:08',
    'HLA-B41:09',
    'HLA-B41:10',
    'HLA-B41:11',
    'HLA-B41:12',
    'HLA-B42:01',
    'HLA-B42:02',
    'HLA-B42:04',
    'HLA-B42:05',
    'HLA-B42:06',
    'HLA-B42:07',
    'HLA-B42:08',
    'HLA-B42:09',
    'HLA-B42:10',
    'HLA-B42:11',
    'HLA-B42:12',
    'HLA-B42:13',
    'HLA-B42:14',
    'HLA-B44:02',
    'HLA-B44:03',
    'HLA-B44:04',
    'HLA-B44:05',
    'HLA-B44:06',
    'HLA-B44:07',
    'HLA-B44:08',
    'HLA-B44:09',
    'HLA-B44:10',
    'HLA-B44:11',
    'HLA-B44:12',
    'HLA-B44:13',
    'HLA-B44:14',
    'HLA-B44:15',
    'HLA-B44:16',
    'HLA-B44:17',
    'HLA-B44:18',
    'HLA-B44:20',
    'HLA-B44:21',
    'HLA-B44:22',
    'HLA-B44:24',
    'HLA-B44:25',
    'HLA-B44:26',
    'HLA-B44:27',
    'HLA-B44:28',
    'HLA-B44:29',
    'HLA-B44:30',
    'HLA-B44:31',
    'HLA-B44:32',
    'HLA-B44:33',
    'HLA-B44:34',
    'HLA-B44:35',
    'HLA-B44:36',
    'HLA-B44:37',
    'HLA-B44:38',
    'HLA-B44:39',
    'HLA-B44:40',
    'HLA-B44:41',
    'HLA-B44:42',
    'HLA-B44:43',
    'HLA-B44:44',
    'HLA-B44:45',
    'HLA-B44:46',
    'HLA-B44:47',
    'HLA-B44:48',
    'HLA-B44:49',
    'HLA-B44:50',
    'HLA-B44:51',
    'HLA-B44:53',
    'HLA-B44:54',
    'HLA-B44:55',
    'HLA-B44:57',
    'HLA-B44:59',
    'HLA-B44:60',
    'HLA-B44:62',
    'HLA-B44:63',
    'HLA-B44:64',
    'HLA-B44:65',
    'HLA-B44:66',
    'HLA-B44:67',
    'HLA-B44:68',
    'HLA-B44:69',
    'HLA-B44:70',
    'HLA-B44:71',
    'HLA-B44:72',
    'HLA-B44:73',
    'HLA-B44:74',
    'HLA-B44:75',
    'HLA-B44:76',
    'HLA-B44:77',
    'HLA-B44:78',
    'HLA-B44:79',
    'HLA-B44:80',
    'HLA-B44:81',
    'HLA-B44:82',
    'HLA-B44:83',
    'HLA-B44:84',
    'HLA-B44:85',
    'HLA-B44:86',
    'HLA-B44:87',
    'HLA-B44:88',
    'HLA-B44:89',
    'HLA-B44:90',
    'HLA-B44:91',
    'HLA-B44:92',
    'HLA-B44:93',
    'HLA-B44:94',
    'HLA-B44:95',
    'HLA-B44:96',
    'HLA-B44:97',
    'HLA-B44:98',
    'HLA-B44:99',
    'HLA-B44:100',
    'HLA-B44:101',
    'HLA-B44:102',
    'HLA-B44:103',
    'HLA-B44:104',
    'HLA-B44:105',
    'HLA-B44:106',
    'HLA-B44:107',
    'HLA-B44:109',
    'HLA-B44:110',
    'HLA-B45:01',
    'HLA-B45:02',
    'HLA-B45:03',
    'HLA-B45:04',
    'HLA-B45:05',
    'HLA-B45:06',
    'HLA-B45:07',
    'HLA-B45:08',
    'HLA-B45:09',
    'HLA-B45:10',
    'HLA-B45:11',
    'HLA-B45:12',
    'HLA-B46:01',
    'HLA-B46:02',
    'HLA-B46:03',
    'HLA-B46:04',
    'HLA-B46:05',
    'HLA-B46:06',
    'HLA-B46:08',
    'HLA-B46:09',
    'HLA-B46:10',
    'HLA-B46:11',
    'HLA-B46:12',
    'HLA-B46:13',
    'HLA-B46:14',
    'HLA-B46:16',
    'HLA-B46:17',
    'HLA-B46:18',
    'HLA-B46:19',
    'HLA-B46:20',
    'HLA-B46:21',
    'HLA-B46:22',
    'HLA-B46:23',
    'HLA-B46:24',
    'HLA-B47:01',
    'HLA-B47:02',
    'HLA-B47:03',
    'HLA-B47:04',
    'HLA-B47:05',
    'HLA-B47:06',
    'HLA-B47:07',
    'HLA-B48:01',
    'HLA-B48:02',
    'HLA-B48:03',
    'HLA-B48:04',
    'HLA-B48:05',
    'HLA-B48:06',
    'HLA-B48:07',
    'HLA-B48:08',
    'HLA-B48:09',
    'HLA-B48:10',
    'HLA-B48:11',
    'HLA-B48:12',
    'HLA-B48:13',
    'HLA-B48:14',
    'HLA-B48:15',
    'HLA-B48:16',
    'HLA-B48:17',
    'HLA-B48:18',
    'HLA-B48:19',
    'HLA-B48:20',
    'HLA-B48:21',
    'HLA-B48:22',
    'HLA-B48:23',
    'HLA-B49:01',
    'HLA-B49:02',
    'HLA-B49:03',
    'HLA-B49:04',
    'HLA-B49:05',
    'HLA-B49:06',
    'HLA-B49:07',
    'HLA-B49:08',
    'HLA-B49:09',
    'HLA-B49:10',
    'HLA-B50:01',
    'HLA-B50:02',
    'HLA-B50:04',
    'HLA-B50:05',
    'HLA-B50:06',
    'HLA-B50:07',
    'HLA-B50:08',
    'HLA-B50:09',
    'HLA-B51:01',
    'HLA-B51:02',
    'HLA-B51:03',
    'HLA-B51:04',
    'HLA-B51:05',
    'HLA-B51:06',
    'HLA-B51:07',
    'HLA-B51:08',
    'HLA-B51:09',
    'HLA-B51:12',
    'HLA-B51:13',
    'HLA-B51:14',
    'HLA-B51:15',
    'HLA-B51:16',
    'HLA-B51:17',
    'HLA-B51:18',
    'HLA-B51:19',
    'HLA-B51:20',
    'HLA-B51:21',
    'HLA-B51:22',
    'HLA-B51:23',
    'HLA-B51:24',
    'HLA-B51:26',
    'HLA-B51:28',
    'HLA-B51:29',
    'HLA-B51:30',
    'HLA-B51:31',
    'HLA-B51:32',
    'HLA-B51:33',
    'HLA-B51:34',
    'HLA-B51:35',
    'HLA-B51:36',
    'HLA-B51:37',
    'HLA-B51:38',
    'HLA-B51:39',
    'HLA-B51:40',
    'HLA-B51:42',
    'HLA-B51:43',
    'HLA-B51:45',
    'HLA-B51:46',
    'HLA-B51:48',
    'HLA-B51:49',
    'HLA-B51:50',
    'HLA-B51:51',
    'HLA-B51:52',
    'HLA-B51:53',
    'HLA-B51:54',
    'HLA-B51:55',
    'HLA-B51:56',
    'HLA-B51:57',
    'HLA-B51:58',
    'HLA-B51:59',
    'HLA-B51:60',
    'HLA-B51:61',
    'HLA-B51:62',
    'HLA-B51:63',
    'HLA-B51:64',
    'HLA-B51:65',
    'HLA-B51:66',
    'HLA-B51:67',
    'HLA-B51:68',
    'HLA-B51:69',
    'HLA-B51:70',
    'HLA-B51:71',
    'HLA-B51:72',
    'HLA-B51:73',
    'HLA-B51:74',
    'HLA-B51:75',
    'HLA-B51:76',
    'HLA-B51:77',
    'HLA-B51:78',
    'HLA-B51:79',
    'HLA-B51:80',
    'HLA-B51:81',
    'HLA-B51:82',
    'HLA-B51:83',
    'HLA-B51:84',
    'HLA-B51:85',
    'HLA-B51:86',
    'HLA-B51:87',
    'HLA-B51:88',
    'HLA-B51:89',
    'HLA-B51:90',
    'HLA-B51:91',
    'HLA-B51:92',
    'HLA-B51:93',
    'HLA-B51:94',
    'HLA-B51:95',
    'HLA-B51:96',
    'HLA-B52:01',
    'HLA-B52:02',
    'HLA-B52:03',
    'HLA-B52:04',
    'HLA-B52:05',
    'HLA-B52:06',
    'HLA-B52:07',
    'HLA-B52:08',
    'HLA-B52:09',
    'HLA-B52:10',
    'HLA-B52:11',
    'HLA-B52:12',
    'HLA-B52:13',
    'HLA-B52:14',
    'HLA-B52:15',
    'HLA-B52:16',
    'HLA-B52:17',
    'HLA-B52:18',
    'HLA-B52:19',
    'HLA-B52:20',
    'HLA-B52:21',
    'HLA-B53:01',
    'HLA-B53:02',
    'HLA-B53:03',
    'HLA-B53:04',
    'HLA-B53:05',
    'HLA-B53:06',
    'HLA-B53:07',
    'HLA-B53:08',
    'HLA-B53:09',
    'HLA-B53:10',
    'HLA-B53:11',
    'HLA-B53:12',
    'HLA-B53:13',
    'HLA-B53:14',
    'HLA-B53:15',
    'HLA-B53:16',
    'HLA-B53:17',
    'HLA-B53:18',
    'HLA-B53:19',
    'HLA-B53:20',
    'HLA-B53:21',
    'HLA-B53:22',
    'HLA-B53:23',
    'HLA-B54:01',
    'HLA-B54:02',
    'HLA-B54:03',
    'HLA-B54:04',
    'HLA-B54:06',
    'HLA-B54:07',
    'HLA-B54:09',
    'HLA-B54:10',
    'HLA-B54:11',
    'HLA-B54:12',
    'HLA-B54:13',
    'HLA-B54:14',
    'HLA-B54:15',
    'HLA-B54:16',
    'HLA-B54:17',
    'HLA-B54:18',
    'HLA-B54:19',
    'HLA-B54:20',
    'HLA-B54:21',
    'HLA-B54:22',
    'HLA-B54:23',
    'HLA-B55:01',
    'HLA-B55:02',
    'HLA-B55:03',
    'HLA-B55:04',
    'HLA-B55:05',
    'HLA-B55:07',
    'HLA-B55:08',
    'HLA-B55:09',
    'HLA-B55:10',
    'HLA-B55:11',
    'HLA-B55:12',
    'HLA-B55:13',
    'HLA-B55:14',
    'HLA-B55:15',
    'HLA-B55:16',
    'HLA-B55:17',
    'HLA-B55:18',
    'HLA-B55:19',
    'HLA-B55:20',
    'HLA-B55:21',
    'HLA-B55:22',
    'HLA-B55:23',
    'HLA-B55:24',
    'HLA-B55:25',
    'HLA-B55:26',
    'HLA-B55:27',
    'HLA-B55:28',
    'HLA-B55:29',
    'HLA-B55:30',
    'HLA-B55:31',
    'HLA-B55:32',
    'HLA-B55:33',
    'HLA-B55:34',
    'HLA-B55:35',
    'HLA-B55:36',
    'HLA-B55:37',
    'HLA-B55:38',
    'HLA-B55:39',
    'HLA-B55:40',
    'HLA-B55:41',
    'HLA-B55:42',
    'HLA-B55:43',
    'HLA-B56:01',
    'HLA-B56:02',
    'HLA-B56:03',
    'HLA-B56:04',
    'HLA-B56:05',
    'HLA-B56:06',
    'HLA-B56:07',
    'HLA-B56:08',
    'HLA-B56:09',
    'HLA-B56:10',
    'HLA-B56:11',
    'HLA-B56:12',
    'HLA-B56:13',
    'HLA-B56:14',
    'HLA-B56:15',
    'HLA-B56:16',
    'HLA-B56:17',
    'HLA-B56:18',
    'HLA-B56:20',
    'HLA-B56:21',
    'HLA-B56:22',
    'HLA-B56:23',
    'HLA-B56:24',
    'HLA-B56:25',
    'HLA-B56:26',
    'HLA-B56:27',
    'HLA-B56:29',
    'HLA-B57:01',
    'HLA-B57:02',
    'HLA-B57:03',
    'HLA-B57:04',
    'HLA-B57:05',
    'HLA-B57:06',
    'HLA-B57:07',
    'HLA-B57:08',
    'HLA-B57:09',
    'HLA-B57:10',
    'HLA-B57:11',
    'HLA-B57:12',
    'HLA-B57:13',
    'HLA-B57:14',
    'HLA-B57:15',
    'HLA-B57:16',
    'HLA-B57:17',
    'HLA-B57:18',
    'HLA-B57:19',
    'HLA-B57:20',
    'HLA-B57:21',
    'HLA-B57:22',
    'HLA-B57:23',
    'HLA-B57:24',
    'HLA-B57:25',
    'HLA-B57:26',
    'HLA-B57:27',
    'HLA-B57:29',
    'HLA-B57:30',
    'HLA-B57:31',
    'HLA-B57:32',
    'HLA-B58:01',
    'HLA-B58:02',
    'HLA-B58:04',
    'HLA-B58:05',
    'HLA-B58:06',
    'HLA-B58:07',
    'HLA-B58:08',
    'HLA-B58:09',
    'HLA-B58:11',
    'HLA-B58:12',
    'HLA-B58:13',
    'HLA-B58:14',
    'HLA-B58:15',
    'HLA-B58:16',
    'HLA-B58:18',
    'HLA-B58:19',
    'HLA-B58:20',
    'HLA-B58:21',
    'HLA-B58:22',
    'HLA-B58:23',
    'HLA-B58:24',
    'HLA-B58:25',
    'HLA-B58:26',
    'HLA-B58:27',
    'HLA-B58:28',
    'HLA-B58:29',
    'HLA-B58:30',
    'HLA-B59:01',
    'HLA-B59:02',
    'HLA-B59:03',
    'HLA-B59:04',
    'HLA-B59:05',
    'HLA-B67:01',
    'HLA-B67:02',
    'HLA-B73:01',
    'HLA-B73:02',
    'HLA-B78:01',
    'HLA-B78:02',
    'HLA-B78:03',
    'HLA-B78:04',
    'HLA-B78:05',
    'HLA-B78:06',
    'HLA-B78:07',
    'HLA-B81:01',
    'HLA-B81:02',
    'HLA-B81:03',
    'HLA-B81:05',
    'HLA-B82:01',
    'HLA-B82:02',
    'HLA-B82:03',
    'HLA-B83:01',
    'HLA-C01:02',
    'HLA-C01:03',
    'HLA-C01:04',
    'HLA-C01:05',
    'HLA-C01:06',
    'HLA-C01:07',
    'HLA-C01:08',
    'HLA-C01:09',
    'HLA-C01:10',
    'HLA-C01:11',
    'HLA-C01:12',
    'HLA-C01:13',
    'HLA-C01:14',
    'HLA-C01:15',
    'HLA-C01:16',
    'HLA-C01:17',
    'HLA-C01:18',
    'HLA-C01:19',
    'HLA-C01:20',
    'HLA-C01:21',
    'HLA-C01:22',
    'HLA-C01:23',
    'HLA-C01:24',
    'HLA-C01:25',
    'HLA-C01:26',
    'HLA-C01:27',
    'HLA-C01:28',
    'HLA-C01:29',
    'HLA-C01:30',
    'HLA-C01:31',
    'HLA-C01:32',
    'HLA-C01:33',
    'HLA-C01:34',
    'HLA-C01:35',
    'HLA-C01:36',
    'HLA-C01:38',
    'HLA-C01:39',
    'HLA-C01:40',
    'HLA-C02:02',
    'HLA-C02:03',
    'HLA-C02:04',
    'HLA-C02:05',
    'HLA-C02:06',
    'HLA-C02:07',
    'HLA-C02:08',
    'HLA-C02:09',
    'HLA-C02:10',
    'HLA-C02:11',
    'HLA-C02:12',
    'HLA-C02:13',
    'HLA-C02:14',
    'HLA-C02:15',
    'HLA-C02:16',
    'HLA-C02:17',
    'HLA-C02:18',
    'HLA-C02:19',
    'HLA-C02:20',
    'HLA-C02:21',
    'HLA-C02:22',
    'HLA-C02:23',
    'HLA-C02:24',
    'HLA-C02:26',
    'HLA-C02:27',
    'HLA-C02:28',
    'HLA-C02:29',
    'HLA-C02:30',
    'HLA-C02:31',
    'HLA-C02:32',
    'HLA-C02:33',
    'HLA-C02:34',
    'HLA-C02:35',
    'HLA-C02:36',
    'HLA-C02:37',
    'HLA-C02:39',
    'HLA-C02:40',
    'HLA-C03:01',
    'HLA-C03:02',
    'HLA-C03:03',
    'HLA-C03:04',
    'HLA-C03:05',
    'HLA-C03:06',
    'HLA-C03:07',
    'HLA-C03:08',
    'HLA-C03:09',
    'HLA-C03:10',
    'HLA-C03:11',
    'HLA-C03:12',
    'HLA-C03:13',
    'HLA-C03:14',
    'HLA-C03:15',
    'HLA-C03:16',
    'HLA-C03:17',
    'HLA-C03:18',
    'HLA-C03:19',
    'HLA-C03:21',
    'HLA-C03:23',
    'HLA-C03:24',
    'HLA-C03:25',
    'HLA-C03:26',
    'HLA-C03:27',
    'HLA-C03:28',
    'HLA-C03:29',
    'HLA-C03:30',
    'HLA-C03:31',
    'HLA-C03:32',
    'HLA-C03:33',
    'HLA-C03:34',
    'HLA-C03:35',
    'HLA-C03:36',
    'HLA-C03:37',
    'HLA-C03:38',
    'HLA-C03:39',
    'HLA-C03:40',
    'HLA-C03:41',
    'HLA-C03:42',
    'HLA-C03:43',
    'HLA-C03:44',
    'HLA-C03:45',
    'HLA-C03:46',
    'HLA-C03:47',
    'HLA-C03:48',
    'HLA-C03:49',
    'HLA-C03:50',
    'HLA-C03:51',
    'HLA-C03:52',
    'HLA-C03:53',
    'HLA-C03:54',
    'HLA-C03:55',
    'HLA-C03:56',
    'HLA-C03:57',
    'HLA-C03:58',
    'HLA-C03:59',
    'HLA-C03:60',
    'HLA-C03:61',
    'HLA-C03:62',
    'HLA-C03:63',
    'HLA-C03:64',
    'HLA-C03:65',
    'HLA-C03:66',
    'HLA-C03:67',
    'HLA-C03:68',
    'HLA-C03:69',
    'HLA-C03:70',
    'HLA-C03:71',
    'HLA-C03:72',
    'HLA-C03:73',
    'HLA-C03:74',
    'HLA-C03:75',
    'HLA-C03:76',
    'HLA-C03:77',
    'HLA-C03:78',
    'HLA-C03:79',
    'HLA-C03:80',
    'HLA-C03:81',
    'HLA-C03:82',
    'HLA-C03:83',
    'HLA-C03:84',
    'HLA-C03:85',
    'HLA-C03:86',
    'HLA-C03:87',
    'HLA-C03:88',
    'HLA-C03:89',
    'HLA-C03:90',
    'HLA-C03:91',
    'HLA-C03:92',
    'HLA-C03:93',
    'HLA-C03:94',
    'HLA-C04:01',
    'HLA-C04:03',
    'HLA-C04:04',
    'HLA-C04:05',
    'HLA-C04:06',
    'HLA-C04:07',
    'HLA-C04:08',
    'HLA-C04:10',
    'HLA-C04:11',
    'HLA-C04:12',
    'HLA-C04:13',
    'HLA-C04:14',
    'HLA-C04:15',
    'HLA-C04:16',
    'HLA-C04:17',
    'HLA-C04:18',
    'HLA-C04:19',
    'HLA-C04:20',
    'HLA-C04:23',
    'HLA-C04:24',
    'HLA-C04:25',
    'HLA-C04:26',
    'HLA-C04:27',
    'HLA-C04:28',
    'HLA-C04:29',
    'HLA-C04:30',
    'HLA-C04:31',
    'HLA-C04:32',
    'HLA-C04:33',
    'HLA-C04:34',
    'HLA-C04:35',
    'HLA-C04:36',
    'HLA-C04:37',
    'HLA-C04:38',
    'HLA-C04:39',
    'HLA-C04:40',
    'HLA-C04:41',
    'HLA-C04:42',
    'HLA-C04:43',
    'HLA-C04:44',
    'HLA-C04:45',
    'HLA-C04:46',
    'HLA-C04:47',
    'HLA-C04:48',
    'HLA-C04:49',
    'HLA-C04:50',
    'HLA-C04:51',
    'HLA-C04:52',
    'HLA-C04:53',
    'HLA-C04:54',
    'HLA-C04:55',
    'HLA-C04:56',
    'HLA-C04:57',
    'HLA-C04:58',
    'HLA-C04:60',
    'HLA-C04:61',
    'HLA-C04:62',
    'HLA-C04:63',
    'HLA-C04:64',
    'HLA-C04:65',
    'HLA-C04:66',
    'HLA-C04:67',
    'HLA-C04:68',
    'HLA-C04:69',
    'HLA-C04:70',
    'HLA-C05:01',
    'HLA-C05:03',
    'HLA-C05:04',
    'HLA-C05:05',
    'HLA-C05:06',
    'HLA-C05:08',
    'HLA-C05:09',
    'HLA-C05:10',
    'HLA-C05:11',
    'HLA-C05:12',
    'HLA-C05:13',
    'HLA-C05:14',
    'HLA-C05:15',
    'HLA-C05:16',
    'HLA-C05:17',
    'HLA-C05:18',
    'HLA-C05:19',
    'HLA-C05:20',
    'HLA-C05:21',
    'HLA-C05:22',
    'HLA-C05:23',
    'HLA-C05:24',
    'HLA-C05:25',
    'HLA-C05:26',
    'HLA-C05:27',
    'HLA-C05:28',
    'HLA-C05:29',
    'HLA-C05:30',
    'HLA-C05:31',
    'HLA-C05:32',
    'HLA-C05:33',
    'HLA-C05:34',
    'HLA-C05:35',
    'HLA-C05:36',
    'HLA-C05:37',
    'HLA-C05:38',
    'HLA-C05:39',
    'HLA-C05:40',
    'HLA-C05:41',
    'HLA-C05:42',
    'HLA-C05:43',
    'HLA-C05:44',
    'HLA-C05:45',
    'HLA-C06:02',
    'HLA-C06:03',
    'HLA-C06:04',
    'HLA-C06:05',
    'HLA-C06:06',
    'HLA-C06:07',
    'HLA-C06:08',
    'HLA-C06:09',
    'HLA-C06:10',
    'HLA-C06:11',
    'HLA-C06:12',
    'HLA-C06:13',
    'HLA-C06:14',
    'HLA-C06:15',
    'HLA-C06:17',
    'HLA-C06:18',
    'HLA-C06:19',
    'HLA-C06:20',
    'HLA-C06:21',
    'HLA-C06:22',
    'HLA-C06:23',
    'HLA-C06:24',
    'HLA-C06:25',
    'HLA-C06:26',
    'HLA-C06:27',
    'HLA-C06:28',
    'HLA-C06:29',
    'HLA-C06:30',
    'HLA-C06:31',
    'HLA-C06:32',
    'HLA-C06:33',
    'HLA-C06:34',
    'HLA-C06:35',
    'HLA-C06:36',
    'HLA-C06:37',
    'HLA-C06:38',
    'HLA-C06:39',
    'HLA-C06:40',
    'HLA-C06:41',
    'HLA-C06:42',
    'HLA-C06:43',
    'HLA-C06:44',
    'HLA-C06:45',
    'HLA-C07:01',
    'HLA-C07:02',
    'HLA-C07:03',
    'HLA-C07:04',
    'HLA-C07:05',
    'HLA-C07:06',
    'HLA-C07:07',
    'HLA-C07:08',
    'HLA-C07:09',
    'HLA-C07:10',
    'HLA-C07:11',
    'HLA-C07:12',
    'HLA-C07:13',
    'HLA-C07:14',
    'HLA-C07:15',
    'HLA-C07:16',
    'HLA-C07:17',
    'HLA-C07:18',
    'HLA-C07:19',
    'HLA-C07:20',
    'HLA-C07:21',
    'HLA-C07:22',
    'HLA-C07:23',
    'HLA-C07:24',
    'HLA-C07:25',
    'HLA-C07:26',
    'HLA-C07:27',
    'HLA-C07:28',
    'HLA-C07:29',
    'HLA-C07:30',
    'HLA-C07:31',
    'HLA-C07:35',
    'HLA-C07:36',
    'HLA-C07:37',
    'HLA-C07:38',
    'HLA-C07:39',
    'HLA-C07:40',
    'HLA-C07:41',
    'HLA-C07:42',
    'HLA-C07:43',
    'HLA-C07:44',
    'HLA-C07:45',
    'HLA-C07:46',
    'HLA-C07:47',
    'HLA-C07:48',
    'HLA-C07:49',
    'HLA-C07:50',
    'HLA-C07:51',
    'HLA-C07:52',
    'HLA-C07:53',
    'HLA-C07:54',
    'HLA-C07:56',
    'HLA-C07:57',
    'HLA-C07:58',
    'HLA-C07:59',
    'HLA-C07:60',
    'HLA-C07:62',
    'HLA-C07:63',
    'HLA-C07:64',
    'HLA-C07:65',
    'HLA-C07:66',
    'HLA-C07:67',
    'HLA-C07:68',
    'HLA-C07:69',
    'HLA-C07:70',
    'HLA-C07:71',
    'HLA-C07:72',
    'HLA-C07:73',
    'HLA-C07:74',
    'HLA-C07:75',
    'HLA-C07:76',
    'HLA-C07:77',
    'HLA-C07:78',
    'HLA-C07:79',
    'HLA-C07:80',
    'HLA-C07:81',
    'HLA-C07:82',
    'HLA-C07:83',
    'HLA-C07:84',
    'HLA-C07:85',
    'HLA-C07:86',
    'HLA-C07:87',
    'HLA-C07:88',
    'HLA-C07:89',
    'HLA-C07:90',
    'HLA-C07:91',
    'HLA-C07:92',
    'HLA-C07:93',
    'HLA-C07:94',
    'HLA-C07:95',
    'HLA-C07:96',
    'HLA-C07:97',
    'HLA-C07:99',
    'HLA-C07:100',
    'HLA-C07:101',
    'HLA-C07:102',
    'HLA-C07:103',
    'HLA-C07:105',
    'HLA-C07:106',
    'HLA-C07:107',
    'HLA-C07:108',
    'HLA-C07:109',
    'HLA-C07:110',
    'HLA-C07:111',
    'HLA-C07:112',
    'HLA-C07:113',
    'HLA-C07:114',
    'HLA-C07:115',
    'HLA-C07:116',
    'HLA-C07:117',
    'HLA-C07:118',
    'HLA-C07:119',
    'HLA-C07:120',
    'HLA-C07:122',
    'HLA-C07:123',
    'HLA-C07:124',
    'HLA-C07:125',
    'HLA-C07:126',
    'HLA-C07:127',
    'HLA-C07:128',
    'HLA-C07:129',
    'HLA-C07:130',
    'HLA-C07:131',
    'HLA-C07:132',
    'HLA-C07:133',
    'HLA-C07:134',
    'HLA-C07:135',
    'HLA-C07:136',
    'HLA-C07:137',
    'HLA-C07:138',
    'HLA-C07:139',
    'HLA-C07:140',
    'HLA-C07:141',
    'HLA-C07:142',
    'HLA-C07:143',
    'HLA-C07:144',
    'HLA-C07:145',
    'HLA-C07:146',
    'HLA-C07:147',
    'HLA-C07:148',
    'HLA-C07:149',
    'HLA-C08:01',
    'HLA-C08:02',
    'HLA-C08:03',
    'HLA-C08:04',
    'HLA-C08:05',
    'HLA-C08:06',
    'HLA-C08:07',
    'HLA-C08:08',
    'HLA-C08:09',
    'HLA-C08:10',
    'HLA-C08:11',
    'HLA-C08:12',
    'HLA-C08:13',
    'HLA-C08:14',
    'HLA-C08:15',
    'HLA-C08:16',
    'HLA-C08:17',
    'HLA-C08:18',
    'HLA-C08:19',
    'HLA-C08:20',
    'HLA-C08:21',
    'HLA-C08:22',
    'HLA-C08:23',
    'HLA-C08:24',
    'HLA-C08:25',
    'HLA-C08:27',
    'HLA-C08:28',
    'HLA-C08:29',
    'HLA-C08:30',
    'HLA-C08:31',
    'HLA-C08:32',
    'HLA-C08:33',
    'HLA-C08:34',
    'HLA-C08:35',
    'HLA-C12:02',
    'HLA-C12:03',
    'HLA-C12:04',
    'HLA-C12:05',
    'HLA-C12:06',
    'HLA-C12:07',
    'HLA-C12:08',
    'HLA-C12:09',
    'HLA-C12:10',
    'HLA-C12:11',
    'HLA-C12:12',
    'HLA-C12:13',
    'HLA-C12:14',
    'HLA-C12:15',
    'HLA-C12:16',
    'HLA-C12:17',
    'HLA-C12:18',
    'HLA-C12:19',
    'HLA-C12:20',
    'HLA-C12:21',
    'HLA-C12:22',
    'HLA-C12:23',
    'HLA-C12:24',
    'HLA-C12:25',
    'HLA-C12:26',
    'HLA-C12:27',
    'HLA-C12:28',
    'HLA-C12:29',
    'HLA-C12:30',
    'HLA-C12:31',
    'HLA-C12:32',
    'HLA-C12:33',
    'HLA-C12:34',
    'HLA-C12:35',
    'HLA-C12:36',
    'HLA-C12:37',
    'HLA-C12:38',
    'HLA-C12:40',
    'HLA-C12:41',
    'HLA-C12:43',
    'HLA-C12:44',
    'HLA-C14:02',
    'HLA-C14:03',
    'HLA-C14:04',
    'HLA-C14:05',
    'HLA-C14:06',
    'HLA-C14:08',
    'HLA-C14:09',
    'HLA-C14:10',
    'HLA-C14:11',
    'HLA-C14:12',
    'HLA-C14:13',
    'HLA-C14:14',
    'HLA-C14:15',
    'HLA-C14:16',
    'HLA-C14:17',
    'HLA-C14:18',
    'HLA-C14:19',
    'HLA-C14:20',
    'HLA-C15:02',
    'HLA-C15:03',
    'HLA-C15:04',
    'HLA-C15:05',
    'HLA-C15:06',
    'HLA-C15:07',
    'HLA-C15:08',
    'HLA-C15:09',
    'HLA-C15:10',
    'HLA-C15:11',
    'HLA-C15:12',
    'HLA-C15:13',
    'HLA-C15:15',
    'HLA-C15:16',
    'HLA-C15:17',
    'HLA-C15:18',
    'HLA-C15:19',
    'HLA-C15:20',
    'HLA-C15:21',
    'HLA-C15:22',
    'HLA-C15:23',
    'HLA-C15:24',
    'HLA-C15:25',
    'HLA-C15:26',
    'HLA-C15:27',
    'HLA-C15:28',
    'HLA-C15:29',
    'HLA-C15:30',
    'HLA-C15:31',
    'HLA-C15:33',
    'HLA-C15:34',
    'HLA-C15:35',
    'HLA-C16:01',
    'HLA-C16:02',
    'HLA-C16:04',
    'HLA-C16:06',
    'HLA-C16:07',
    'HLA-C16:08',
    'HLA-C16:09',
    'HLA-C16:10',
    'HLA-C16:11',
    'HLA-C16:12',
    'HLA-C16:13',
    'HLA-C16:14',
    'HLA-C16:15',
    'HLA-C16:17',
    'HLA-C16:18',
    'HLA-C16:19',
    'HLA-C16:20',
    'HLA-C16:21',
    'HLA-C16:22',
    'HLA-C16:23',
    'HLA-C16:24',
    'HLA-C16:25',
    'HLA-C16:26',
    'HLA-C17:01',
    'HLA-C17:02',
    'HLA-C17:03',
    'HLA-C17:04',
    'HLA-C17:05',
    'HLA-C17:06',
    'HLA-C17:07',
    'HLA-C18:01',
    'HLA-C18:02',
    'HLA-C18:03',
    'Mamu-A1:00101',
    'Mamu-A1:00102',
    'Mamu-A1:00103',
    'Mamu-A1:00104',
    'Mamu-A1:00105',
    'Mamu-A1:00201',
    'Mamu-A1:00301',
    'Mamu-A1:00302',
    'Mamu-A1:00303',
    'Mamu-A1:00304',
    'Mamu-A1:00305',
    'Mamu-A1:00306',
    'Mamu-A1:00307',
    'Mamu-A1:00308',
    'Mamu-A1:00310',
    'Mamu-A1:00401',
    'Mamu-A1:00402',
    'Mamu-A1:00403',
    'Mamu-A1:00601',
    'Mamu-A1:00602',
    'Mamu-A1:00701',
    'Mamu-A1:00702',
    'Mamu-A1:00703',
    'Mamu-A1:00704',
    'Mamu-A1:00705',
    'Mamu-A1:00801',
    'Mamu-A1:01001',
    'Mamu-A1:01002',
    'Mamu-A1:01101',
    'Mamu-A1:01102',
    'Mamu-A1:01103',
    'Mamu-A1:01104',
    'Mamu-A1:01201',
    'Mamu-A1:01601',
    'Mamu-A1:01801',
    'Mamu-A1:01802',
    'Mamu-A1:01803',
    'Mamu-A1:01804',
    'Mamu-A1:01805',
    'Mamu-A1:01806',
    'Mamu-A1:01807',
    'Mamu-A1:01808',
    'Mamu-A1:01901',
    'Mamu-A1:01902',
    'Mamu-A1:01903',
    'Mamu-A1:01904',
    'Mamu-A1:01905',
    'Mamu-A1:01906',
    'Mamu-A1:01907',
    'Mamu-A1:02201',
    'Mamu-A1:02202',
    'Mamu-A1:02203',
    'Mamu-A1:02301',
    'Mamu-A1:02302',
    'Mamu-A1:02501',
    'Mamu-A1:02502',
    'Mamu-A1:02601',
    'Mamu-A1:02602',
    'Mamu-A1:02603',
    'Mamu-A1:02801',
    'Mamu-A1:02802',
    'Mamu-A1:02803',
    'Mamu-A1:02804',
    'Mamu-A1:02805',
    'Mamu-A1:02806',
    'Mamu-A1:03201',
    'Mamu-A1:03202',
    'Mamu-A1:03203',
    'Mamu-A1:03301',
    'Mamu-A1:04001',
    'Mamu-A1:04002',
    'Mamu-A1:04003',
    'Mamu-A1:04101',
    'Mamu-A1:04102',
    'Mamu-A1:04201',
    'Mamu-A1:04301',
    'Mamu-A1:04501',
    'Mamu-A1:04801',
    'Mamu-A1:04901',
    'Mamu-A1:04902',
    'Mamu-A1:04903',
    'Mamu-A1:04904',
    'Mamu-A1:05001',
    'Mamu-A1:05101',
    'Mamu-A1:05201',
    'Mamu-A1:05301',
    'Mamu-A1:05302',
    'Mamu-A1:05401',
    'Mamu-A1:05402',
    'Mamu-A1:05501',
    'Mamu-A1:05601',
    'Mamu-A1:05602',
    'Mamu-A1:05603',
    'Mamu-A1:05701',
    'Mamu-A1:05702',
    'Mamu-A1:05901',
    'Mamu-A1:06001',
    'Mamu-A1:06101',
    'Mamu-A1:06301',
    'Mamu-A1:06501',
    'Mamu-A1:06601',
    'Mamu-A1:07301',
    'Mamu-A1:07401',
    'Mamu-A1:07402',
    'Mamu-A1:07403',
    'Mamu-A1:08101',
    'Mamu-A1:08501',
    'Mamu-A1:09101',
    'Mamu-A1:09201',
    'Mamu-A1:10501',
    'Mamu-A1:10502',
    'Mamu-A1:10503',
    'Mamu-A1:10504',
    'Mamu-A1:10601',
    'Mamu-A1:10701',
    'Mamu-A1:10801',
    'Mamu-A1:10901',
    'Mamu-A1:11001',
    'Mamu-A1:11101',
    'Mamu-A1:11201',
    'Mamu-A1:11301',
    'Mamu-A2:0101',
    'Mamu-A2:0102',
    'Mamu-A2:0103',
    'Mamu-A2:0501',
    'Mamu-A2:05020',
    'Mamu-A2:05030',
    'Mamu-A2:05040',
    'Mamu-A2:0505',
    'Mamu-A2:0506',
    'Mamu-A2:0507',
    'Mamu-A2:0509',
    'Mamu-A2:0510',
    'Mamu-A2:0511',
    'Mamu-A2:0512',
    'Mamu-A2:0513',
    'Mamu-A2:0514',
    'Mamu-A2:05150',
    'Mamu-A2:05160',
    'Mamu-A2:0517',
    'Mamu-A2:0518',
    'Mamu-A2:0519',
    'Mamu-A2:0520',
    'Mamu-A2:0521',
    'Mamu-A2:0522',
    'Mamu-A2:0523',
    'Mamu-A2:0524',
    'Mamu-A2:0525',
    'Mamu-A2:0526',
    'Mamu-A2:0527',
    'Mamu-A2:0528',
    'Mamu-A2:0529',
    'Mamu-A2:0531',
    'Mamu-A2:05320',
    'Mamu-A2:0533',
    'Mamu-A2:0534',
    'Mamu-A2:0535',
    'Mamu-A2:0536',
    'Mamu-A2:0537',
    'Mamu-A2:0538',
    'Mamu-A2:0539',
    'Mamu-A2:0540',
    'Mamu-A2:0541',
    'Mamu-A2:0542',
    'Mamu-A2:0543',
    'Mamu-A2:0544',
    'Mamu-A2:0545',
    'Mamu-A2:0546',
    'Mamu-A2:0547',
    'Mamu-A2:2401',
    'Mamu-A2:2402',
    'Mamu-A2:2403',
    'Mamu-A3:1301',
    'Mamu-A3:1302',
    'Mamu-A3:1303',
    'Mamu-A3:1304',
    'Mamu-A3:1305',
    'Mamu-A3:1306',
    'Mamu-A3:1307',
    'Mamu-A3:1308',
    'Mamu-A3:1309',
    'Mamu-A3:1310',
    'Mamu-A3:1311',
    'Mamu-A3:1312',
    'Mamu-A3:1313',
    'Mamu-A4:0101',
    'Mamu-A4:01020',
    'Mamu-A4:0103',
    'Mamu-A4:0202',
    'Mamu-A4:0203',
    'Mamu-A4:0205',
    'Mamu-A4:0301',
    'Mamu-A4:0302',
    'Mamu-A4:1402',
    'Mamu-A4:14030',
    'Mamu-A4:1404',
    'Mamu-A4:1405',
    'Mamu-A4:1406',
    'Mamu-A4:1407',
    'Mamu-A4:1408',
    'Mamu-A4:1409',
    'Mamu-A5:30010',
    'Mamu-A5:3002',
    'Mamu-A5:3003',
    'Mamu-A5:3004',
    'Mamu-A5:3005',
    'Mamu-A5:3006',
    'Mamu-A6:0101',
    'Mamu-A6:0102',
    'Mamu-A6:0103',
    'Mamu-A6:0104',
    'Mamu-A6:0105',
    'Mamu-A7:0101',
    'Mamu-A7:0102',
    'Mamu-A7:0103',
    'Mamu-A7:0201',
    'Mamu-AG:01',
    'Mamu-AG:02011',
    'Mamu-AG:02012',
    'Mamu-AG:0202',
    'Mamu-AG:03011',
    'Mamu-AG:0302',
    'Mamu-B:00101',
    'Mamu-B:00102',
    'Mamu-B:00201',
    'Mamu-B:00202',
    'Mamu-B:00301',
    'Mamu-B:00302',
    'Mamu-B:00401',
    'Mamu-B:00501',
    'Mamu-B:00502',
    'Mamu-B:00601',
    'Mamu-B:00602',
    'Mamu-B:00701',
    'Mamu-B:00702',
    'Mamu-B:00703',
    'Mamu-B:00704',
    'Mamu-B:00801',
    'Mamu-B:01001',
    'Mamu-B:01101',
    'Mamu-B:01201',
    'Mamu-B:01301',
    'Mamu-B:01401',
    'Mamu-B:01501',
    'Mamu-B:01502',
    'Mamu-B:01601',
    'Mamu-B:01701',
    'Mamu-B:01702',
    'Mamu-B:01703',
    'Mamu-B:01801',
    'Mamu-B:01901',
    'Mamu-B:01902',
    'Mamu-B:01903',
    'Mamu-B:02001',
    'Mamu-B:02101',
    'Mamu-B:02102',
    'Mamu-B:02103',
    'Mamu-B:02201',
    'Mamu-B:02301',
    'Mamu-B:02401',
    'Mamu-B:02501',
    'Mamu-B:02601',
    'Mamu-B:02602',
    'Mamu-B:02701',
    'Mamu-B:02702',
    'Mamu-B:02703',
    'Mamu-B:02801',
    'Mamu-B:02802',
    'Mamu-B:02803',
    'Mamu-B:02901',
    'Mamu-B:02902',
    'Mamu-B:03001',
    'Mamu-B:03002',
    'Mamu-B:03003',
    'Mamu-B:03004',
    'Mamu-B:03005',
    'Mamu-B:03101',
    'Mamu-B:03102',
    'Mamu-B:03103',
    'Mamu-B:03201',
    'Mamu-B:03301',
    'Mamu-B:03401',
    'Mamu-B:03501',
    'Mamu-B:03601',
    'Mamu-B:03602',
    'Mamu-B:03701',
    'Mamu-B:03801',
    'Mamu-B:03802',
    'Mamu-B:03901',
    'Mamu-B:04001',
    'Mamu-B:04002',
    'Mamu-B:04101',
    'Mamu-B:04201',
    'Mamu-B:04301',
    'Mamu-B:04401',
    'Mamu-B:04402',
    'Mamu-B:04403',
    'Mamu-B:04404',
    'Mamu-B:04405',
    'Mamu-B:04501',
    'Mamu-B:04502',
    'Mamu-B:04503',
    'Mamu-B:04504',
    'Mamu-B:04601',
    'Mamu-B:04602',
    'Mamu-B:04603',
    'Mamu-B:04604',
    'Mamu-B:04605',
    'Mamu-B:04607',
    'Mamu-B:04608',
    'Mamu-B:04609',
    'Mamu-B:04610',
    'Mamu-B:04611',
    'Mamu-B:04612',
    'Mamu-B:04613',
    'Mamu-B:04614',
    'Mamu-B:04615',
    'Mamu-B:04616',
    'Mamu-B:04617',
    'Mamu-B:04701',
    'Mamu-B:04702',
    'Mamu-B:04703',
    'Mamu-B:04704',
    'Mamu-B:04705',
    'Mamu-B:04801',
    'Mamu-B:04802',
    'Mamu-B:04901',
    'Mamu-B:05002',
    'Mamu-B:05101',
    'Mamu-B:05102',
    'Mamu-B:05103',
    'Mamu-B:05104',
    'Mamu-B:05105',
    'Mamu-B:05201',
    'Mamu-B:05301',
    'Mamu-B:05302',
    'Mamu-B:05401',
    'Mamu-B:05501',
    'Mamu-B:05601',
    'Mamu-B:05602',
    'Mamu-B:05701',
    'Mamu-B:05702',
    'Mamu-B:05802',
    'Mamu-B:05901',
    'Mamu-B:06001',
    'Mamu-B:06002',
    'Mamu-B:06003',
    'Mamu-B:06101',
    'Mamu-B:06102',
    'Mamu-B:06103',
    'Mamu-B:06301',
    'Mamu-B:06302',
    'Mamu-B:06401',
    'Mamu-B:06402',
    'Mamu-B:06501',
    'Mamu-B:06502',
    'Mamu-B:06503',
    'Mamu-B:06601',
    'Mamu-B:06701',
    'Mamu-B:06702',
    'Mamu-B:06801',
    'Mamu-B:06802',
    'Mamu-B:06803',
    'Mamu-B:06804',
    'Mamu-B:06805',
    'Mamu-B:06901',
    'Mamu-B:06902',
    'Mamu-B:06903',
    'Mamu-B:06904',
    'Mamu-B:07001',
    'Mamu-B:07002',
    'Mamu-B:07101',
    'Mamu-B:07201',
    'Mamu-B:07202',
    'Mamu-B:07301',
    'Mamu-B:07401',
    'Mamu-B:07402',
    'Mamu-B:07501',
    'Mamu-B:07502',
    'Mamu-B:07601',
    'Mamu-B:07602',
    'Mamu-B:07701',
    'Mamu-B:07702',
    'Mamu-B:07801',
    'Mamu-B:07901',
    'Mamu-B:07902',
    'Mamu-B:07903',
    'Mamu-B:08001',
    'Mamu-B:08101',
    'Mamu-B:08102',
    'Mamu-B:08201',
    'Mamu-B:08202',
    'Mamu-B:08301',
    'Mamu-B:08401',
    'Mamu-B:08501',
    'Mamu-B:08502',
    'Mamu-B:08601',
    'Mamu-B:08602',
    'Mamu-B:08603',
    'Mamu-B:08701',
    'Mamu-B:08801',
    'Mamu-B:08901',
    'Mamu-B:09001',
    'Mamu-B:09101',
    'Mamu-B:09102',
    'Mamu-B:09201',
    'Mamu-B:09301',
    'Mamu-B:09401',
    'Mamu-B:09501',
    'Mamu-B:09601',
    'Mamu-B:09701',
    'Mamu-B:09801',
    'Mamu-B:09901',
    'Mamu-B:10001',
    'Mamu-B:10101',
    'Patr-A0101',
    'Patr-A0201',
    'Patr-A0301',
    'Patr-A0302',
    'Patr-A0401',
    'Patr-A0402',
    'Patr-A0404',
    'Patr-A0501',
    'Patr-A0601',
    'Patr-A0602',
    'Patr-A0701',
    'Patr-A0801',
    'Patr-A0802',
    'Patr-A0803',
    'Patr-A0901',
    'Patr-A0902',
    'Patr-A1001',
    'Patr-A1101',
    'Patr-A1201',
    'Patr-A1301',
    'Patr-A1401',
    'Patr-A1501',
    'Patr-A1502',
    'Patr-A1601',
    'Patr-A1701',
    'Patr-A1702',
    'Patr-A1703',
    'Patr-A1801',
    'Patr-A2301',
    'Patr-A2401',
    'Patr-B0101',
    'Patr-B0102',
    'Patr-B0201',
    'Patr-B0203',
    'Patr-B0301',
    'Patr-B0302',
    'Patr-B0401',
    'Patr-B0402',
    'Patr-B0501',
    'Patr-B0502',
    'Patr-B0601',
    'Patr-B0701',
    'Patr-B0702',
    'Patr-B0801',
    'Patr-B0802',
    'Patr-B0901',
    'Patr-B1001',
    'Patr-B1101',
    'Patr-B1102',
    'Patr-B1202',
    'Patr-B1301',
    'Patr-B1401',
    'Patr-B1601',
    'Patr-B1602',
    'Patr-B1701',
    'Patr-B1702',
    'Patr-B1703',
    'Patr-B1801',
    'Patr-B1901',
    'Patr-B2001',
    'Patr-B2101',
    'Patr-B2201',
    'Patr-B2202',
    'Patr-B2301',
    'Patr-B2302',
    'Patr-B2303',
    'Patr-B2401',
    'Patr-B2402',
    'Patr-B2501',
    'Patr-B2601',
    'Patr-B2701',
    'Patr-B2801',
    'Patr-B2901',
    'Patr-B3001',
    'Patr-B3501',
    'Patr-B3601',
    'Patr-B3701',
    'Patr-C0201',
    'Patr-C0202',
    'Patr-C0203',
    'Patr-C0204',
    'Patr-C0205',
    'Patr-C0206',
    'Patr-C0301',
    'Patr-C0302',
    'Patr-C0303',
    'Patr-C0304',
    'Patr-C0401',
    'Patr-C0501',
    'Patr-C0502',
    'Patr-C0601',
    'Patr-C0701',
    'Patr-C0801',
    'Patr-C0901',
    'Patr-C0902',
    'Patr-C0903',
    'Patr-C0904',
    'Patr-C0905',
    'Patr-C1001',
    'Patr-C1101',
    'Patr-C1201',
    'Patr-C1301',
    'Patr-C1302',
    'Patr-C1501',
    'Patr-C1601',
    'SLA-1:0101',
    'SLA-1:0201',
    'SLA-1:0202',
    'SLA-1:0401',
    'SLA-1:0501',
    'SLA-1:0601',
    'SLA-1:0701',
    'SLA-1:0702',
    'SLA-1:0801',
    'SLA-1:1101',
    'SLA-1:1201',
    'SLA-1:1301',
    'SLA-2:0101',
    'SLA-2:0102',
    'SLA-2:0201',
    'SLA-2:0202',
    'SLA-2:0301',
    'SLA-2:0302',
    'SLA-2:0401',
    'SLA-2:0402',
    'SLA-2:0501',
    'SLA-2:0502',
    'SLA-2:0601',
    'SLA-2:0701',
    'SLA-2:1001',
    'SLA-2:1002',
    'SLA-2:1101',
    'SLA-2:1201',
    'SLA-3:0101',
    'SLA-3:0301',
    'SLA-3:0302',
    'SLA-3:0303',
    'SLA-3:0304',
    'SLA-3:0401',
    'SLA-3:0501',
    'SLA-3:0502',
    'SLA-3:0503',
    'SLA-3:0601',
    'SLA-3:0602',
    'SLA-3:0701',
    'SLA-6:0101',
    'SLA-6:0102',
    'SLA-6:0103',
    'SLA-6:0104',
    'SLA-6:0105',
    'SLA-1-CHANGDA',
    'SLA-1-HB01',
    'SLA-1-HB02',
    'SLA-1-HB03',
    'SLA-1-HB04',
    'SLA-1-LWH',
    'SLA-1-TPK',
    'SLA-1-YC',
    'SLA-1-YDL01',
    'SLA-1-YTH',
    'SLA-1:es11',
    'SLA-2-YDL02',
    'SLA-2:CDY.AA',
    'SLA-2:HB01',
    'SLA-2:LWH.AA',
    'SLA-2:TPK.AA',
    'SLA-2:YC.AA',
    'SLA-2:YDL.AA',
    'SLA-2:YDY.AA',
    'SLA-2:YTH.AA',
    'SLA-2:es22',
    'SLA-3-CDY',
    'SLA-3-HB01',
    'SLA-3-LWH',
    'SLA-3-TPK',
    'SLA-3-YC',
    'SLA-3-YDL',
    'SLA-3-YDY01',
    'SLA-3-YDY02',
    'SLA-3-YTH',
    'H-2-Db',
    'H-2-Dd',
    'H-2-Kb',
    'H-2-Kd',
    'H-2-Kk',
    'H-2-Ld',
    'H-2-Qa2',
    'H-2-Qa1',
    'Gogo-B0101',
    'HLA-G01:01',
    'HLA-G01:02',
    'HLA-G01:03',
    'HLA-G01:04',
    'HLA-G01:06',
    'HLA-G01:07',
    'HLA-G01:08',
    'HLA-G01:09',
    'HLA-E01:01',
    'BoLA-N:00101',
    'BoLA-N:00102',
    'BoLA-N:00103',
    'BoLA-N:00201',
    'BoLA-N:00301',
    'BoLA-N:00401',
    'BoLA-N:00402',
    'BoLA-N:00501',
    'BoLA-N:00601',
    'BoLA-N:00602',
    'BoLA-N:00801',
    'BoLA-N:00802',
    'BoLA-N:00901',
    'BoLA-N:00902',
    'BoLA-N:01001',
    'BoLA-N:01101',
    'BoLA-N:01201',
    'BoLA-N:01301',
    'BoLA-N:01302',
    'BoLA-N:01401',
    'BoLA-N:01402',
    'BoLA-N:01501',
    'BoLA-N:01502',
    'BoLA-N:01601',
    'BoLA-N:01602',
    'BoLA-N:01701',
    'BoLA-N:01702',
    'BoLA-N:01801',
    'BoLA-N:01802',
    'BoLA-N:01901',
    'BoLA-N:02001',
    'BoLA-N:02101',
    'BoLA-N:02201',
    'BoLA-N:02301',
    'BoLA-N:02401',
    'BoLA-N:02402',
    'BoLA-N:02501',
    'BoLA-N:02601',
    'BoLA-N:02602',
    'BoLA-N:02701',
    'BoLA-N:02702',
    'BoLA-N:02801',
    'BoLA-N:02901',
    'BoLA-N:03001',
    'BoLA-N:03101',
    'BoLA-N:03401',
    'BoLA-N:03501',
    'BoLA-N:03601',
    'BoLA-N:03701',
    'BoLA-N:03801',
    'BoLA-N:03901',
    'BoLA-N:04001',
    'BoLA-N:04101',
    'BoLA-N:04201',
    'BoLA-N:04301',
    'BoLA-N:04401',
    'BoLA-N:04501',
    'BoLA-N:04601',
    'BoLA-N:04701',
    'BoLA-N:04801',
    'BoLA-N:04901',
    'BoLA-N:05001',
    'BoLA-N:05101',
    'BoLA-N:05201',
    'BoLA-N:05301',
    'BoLA-N:05401',
    'BoLA-N:05501',
    'BoLA-N:05601',
    'BoLA-NC1:00101',
    'BoLA-NC1:00201',
    'BoLA-NC1:00301',
    'BoLA-NC1:00401',
    'BoLA-NC2:00101',
    'BoLA-NC2:00102',
    'BoLA-NC3:00101',
    'BoLA-NC4:00101',
    'BoLA-NC4:00201',
    'BoLA-HD6',
    'BoLA-JSP.1',
    'BoLA-T2c',
    'BoLA-T2b',
    'BoLA-T2a',
    'BoLA-T7',
    'BoLA-D18.4',
    'BoLA-AW10',
    'BoLA-T5',
    'BoLA-1:00901',
    'BoLA-1:00902',
    'BoLA-1:01901',
    'BoLA-1:02001',
    'BoLA-1:02101',
    'BoLA-1:02301',
    'BoLA-1:02801',
    'BoLA-1:02901',
    'BoLA-1:03101',
    'BoLA-1:03102',
    'BoLA-1:04201',
    'BoLA-1:04901',
    'BoLA-1:06101',
    'BoLA-1:06701',
    'BoLA-1:07401',
    'BoLA-2:00501',
    'BoLA-2:00601',
    'BoLA-2:00602',
    'BoLA-2:00801',
    'BoLA-2:00802',
    'BoLA-2:01201',
    'BoLA-2:01601',
    'BoLA-2:01602',
    'BoLA-2:01801',
    'BoLA-2:01802',
    'BoLA-2:02201',
    'BoLA-2:02501',
    'BoLA-2:02601',
    'BoLA-2:02602',
    'BoLA-2:02603',
    'BoLA-2:03001',
    'BoLA-2:03202',
    'BoLA-2:04301',
    'BoLA-2:04401',
    'BoLA-2:04402',
    'BoLA-2:04501',
    'BoLA-2:04601',
    'BoLA-2:04701',
    'BoLA-2:04801',
    'BoLA-2:05401',
    'BoLA-2:05501',
    'BoLA-2:05601',
    'BoLA-2:05701',
    'BoLA-2:06001',
    'BoLA-2:06201',
    'BoLA-2:06901',
    'BoLA-2:07001',
    'BoLA-2:07101',
    'BoLA-3:00101',
    'BoLA-3:00102',
    'BoLA-3:00103',
    'BoLA-3:00201',
    'BoLA-3:00401',
    'BoLA-3:00402',
    'BoLA-3:00403',
    'BoLA-3:01001',
    'BoLA-3:01101',
    'BoLA-3:01701',
    'BoLA-3:01702',
    'BoLA-3:01703',
    'BoLA-3:02701',
    'BoLA-3:02702',
    'BoLA-3:03501',
    'BoLA-3:03601',
    'BoLA-3:03701',
    'BoLA-3:03801',
    'BoLA-3:05001',
    'BoLA-3:05002',
    'BoLA-3:05101',
    'BoLA-3:05201',
    'BoLA-3:05301',
    'BoLA-3:05801',
    'BoLA-3:05901',
    'BoLA-3:06501',
    'BoLA-3:06601',
    'BoLA-3:06602',
    'BoLA-3:06801',
    'BoLA-3:07301',
    'BoLA-4:02401',
    'BoLA-4:02402',
    'BoLA-4:06301',
    'BoLA-5:00301',
    'BoLA-5:03901',
    'BoLA-5:06401',
    'BoLA-5:07201',
    'BoLA-6:01301',
    'BoLA-6:01302',
    'BoLA-6:01401',
    'BoLA-6:01402',
    'BoLA-6:01501',
    'BoLA-6:01502',
    'BoLA-6:03401',
    'BoLA-6:04001',
    'BoLA-6:04101'
]
