import sys
import difflib
import subprocess
import pandas as pd
from pathlib import Path
from AlleleValidator import AlleleValidator
from allele_autocomplete import Allele_Autocomplete


DATA_DIR = str(Path(__file__).parent)


def compare_allele_files():
    MHC_NAMES_FILE = DATA_DIR + '/MHC_allele_names.txt'
    MHC_PSEUDO_FILE = DATA_DIR + '/MHC_pseudo.dat'

    with open(MHC_NAMES_FILE, 'r') as f:
        # Read allele file from the DTU
        content_first = f.readlines()
        mhc_names_alleles = [_.strip() for _ in content_first]
        
    with open(MHC_PSEUDO_FILE, 'r') as f:
        # read allele file from NetMHCpan-4.1 in GitLab
        content_sec = f.readlines()
        
        # File Format: left column (allele) right column (seq)
        content_sec = [_.strip().split(' ') for _ in content_sec]

        pseudo_alleles = []
        for _ in content_sec:
            allele = _[0]
            if '\t' in allele:    
                allele = allele.split('\t')[0]
                
            pseudo_alleles.append(allele)

    '''
    There are about 2000 more alleles in the MHC_pseudo.dat file (13101)
    than the MHC_allele_names.txt (11101). Note that the MHC_allele_names.txt
    is from DTU, and that's probably the accurate one.

    Let's mark wich alleles exist both files, and which ones don't.
    
    HOW?
    ----
    Take the one with larger amount of alleles, and turn it into a dictionary.
    We will then set key: allele name, value: T/F (depending on if it exists in
    both files).
    '''
    # Handle duplicates:
    # This removes 104 alleles, leaving 12997 unique alleles
    pseudo_alleles = set(pseudo_alleles) 
    pseudo_alleles_dict= dict(zip(pseudo_alleles, [False]*len(pseudo_alleles)))
    
    for allele in mhc_names_alleles:
        if allele in pseudo_alleles_dict:
            pseudo_alleles_dict[allele] = True
    
    counter = 0
    extra_alleles = []
    for k, v in pseudo_alleles_dict.items():
        if not v:
            counter += 1
            extra_alleles.append(k)

    # This shows that there are 1896 alleles that exist only in the pseudo file.
    print(counter)
    
    av = AlleleValidator()
    ac = Allele_Autocomplete()

    # unknown alleles
    unknown_alleles = []

    for toolname in extra_alleles:
        iedb_label = av.convert_methodlabel_to_iedblabel(toolname, 'netmhcpan')
        if not iedb_label:
            # iedb_label = av.convert_synonym_to_iedblabel(toolname)
            potential_names = ac.get_allele_suggestions(query=toolname)
            try:
                iedb_label = potential_names[0]['label']
                # Define the curl command
                curl_command = [
                    'curl',
                    '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={iedb_label}&length=9',
                    'http://localhost/tools_api/mhci/'
                ]

                # Run the curl command
                result = subprocess.run(curl_command, capture_output=True, text=True)

                # Run the curl command
                result = subprocess.run(curl_command, capture_output=True, text=True)

                # Print the output
                print(result.stdout)
                print(type(result.stdout))
                
                if 'Invalid allele name' in result.stdout:
                    sys.exit()

            except:
                unknown_alleles.append(toolname)

    print('----')
    print(unknown_alleles)

    UNKNOWN_ALLELE_FILE = DATA_DIR + '/unknown_mhc_alleles.txt'

    with open(UNKNOWN_ALLELE_FILE, 'w') as f:
        for _ in unknown_alleles:
            f.write(f'{_}\n')


# def count_list():
#     a = ['H2-Dq', 'Chi-B0401', 'BoLA-N:01402', 'BoLA-N:04601', 'HLA-B9514', 'HLA-B9518', 'BoLA-601302', 'Mamu-B8301', 'SLA-30301', 'BoLA-200801', 'SLA-20601', 'BoLA-300103', 'HLA-B9526', 'HLA-B9522', 'SLA-30101', 'BoLA-N:04001', 'Mamu-A0602', 'H2-Qa2', 'BoLA-203202', 'Patr-A01:01', 'Patr-B17:02', 'BoLA-303601', 'SLA-21002', 'Patr-C09:04', 'BoLA-303701', 'BoLA-201801', 'BoLA-N:05001', 'BoLA-201601', 'Mamu-A1001:01', 'Mamu-B8701', 'Patr-B04:01', 'BoLA-201802', 'HLA-B9530', 'SLA-10501', 'BoLA-N:00801', 'Patr-C09:03', 'Mamu-B02', 'Patr-A04:04', 'BoLA-305301', 'BoLA-N:00102', 'Patr-B16:02', 'Patr-A12:01', 'BoLA-204401', 'BoLA-601301', 'HLA-B0744', 'H2-Qa1', 'HLA-B9509', 'Mamu-A04', 'BoLA-300401', 'BoLA-N:02301', 'BoLA-N:05501', 'Patr-C02:04', 'Patr-A17:03', 'BoLA-N:03101', 'BoLA-601401', 'Mamu-A26', 'Patr-C03:04', 'BoLA-307301', 'Mamu-B21', 'Mamu-B01', 'HLA-A2307', 'BoLA-N:03001', 'Mamu-A20101', 'SLA-20202', 'HLA-A9222', 'BoLA-603401', 'BoLA-303501', 'BoLA-N:02901', 'Mamu-B57', 'Patr-A09:02', 'H2-Ld', 'BoLA-202602', 'HLA-B3565', 'BoLA-N:02801', 'BoLA-N:00103', 'BoLA-207001', 'BoLA-N:01802', 'Mamu-B3901', 'Patr-A06:02', 'Patr-A08:01', 'Mamu-B03', 'Mamu-B43', 'HLA-A9208', 'HLA-B0749', 'Mamu-B12', 'Mamu-A11', 'Patr-A18:01', 'Patr-B09:01', 'BoLA-102101', 'BoLA-202201', 'BoLA-204801', 'BoLA-N:02602', 'Patr-C11:01', 'BoLA-107401', 'BoLA-N:00802', 'Patr-B17:03', 'SLA-10101', 'BoLA-100901', 'HLA-B1823', 'Patr-B20:01', 'Patr-B24:01', 'H2-Kk', 'BoLA-303801', 'Patr-B35:01', 'SLA-20501', 'BoLA-N:04301', 'SLA-107:01', 'Chi-B1501', 'SLA-104:01', 'HLA-A2409', 'Patr-B12:02', 'Mamu-B66', 'SLA-21001', 'Patr-B17:01', 'SLA-21201', 'Mamu-A1602', 'HLA-B9527', 'Patr-C02:05', 'HLA-B5147', 'BoLA-N:05601', 'Mamu-B65', 'SLA-10801', 'Mamu-B40', 'Patr-A15:01', 'Mamu-B55', 'BoLA-301101', 'HLA-A9224', 'BoLA-104201', 'BoLA-106701', 'Mamu-B49', 'SLA-30602', 'Patr-B04:02', 'Mamu-B008:01', 'Patr-C08:01', 'Patr-C05:02', 'BoLA-207101', 'Patr-B30:01', 'Patr-C02:03', 'SLA-10601', 'Patr-C09:02', 'Mamu-A0507', 'HLA-A2465', 'HLA-B9517', 'BoLA-N:03601', 'BoLA-N:01701', 'Mamu-B37', 'SLA-10701', 'BoLA-N:00402', 'HLA-B0808', 'BoLA-N:01401', 'Mamu-A0509', 'Mamu-B47', 'SLA-30401', 'BoLA-N:03501', 'BoLA-102801', 'HLA-A9220', 'BoLA-206901', 'Mamu-A1305', 'SLA-30601', 'BoLA-503901', 'BoLA-N:01501', 'BoLA-406301', 'BoLA-N:01001', 'HLA-B5405', 'SLA-304:01', 'HLA-A9211', 'SLA-30304', 'Mamu-B63', 'HLA-A9217', 'Patr-A04:02', 'BoLA-507201', 'HLA-B9521', 'HLA-A3021', 'Mamu-B001:01', 'BoLA-201201', 'Mamu-B05', 'Mamu-B46', 'HLA-B9505', 'HLA-B9520', 'BoLA-N:02101', 'H2-Db', 'Patr-A06:01', 'Patr-C09:05', 'SLA-20201', 'BoLA-601502', 'Patr-C04:01', 'BoLA-N:00401', 'Mamu-B36', 'H2-Lq', 'Mamu-B5002', 'SLA-30502', 'BoLA-N:01502', 'Patr-A11:01', 'Mamu-B28', 'Patr-A15:02', 'Patr-B19:01', 'BoLA-302702', 'BoLA-N:04901', 'HLA-A9214', 'BoLA-205701', 'BoLA-N:04101', 'BoLA-N:04401', 'BoLA-306501', 'Patr-B11:01', 'Patr-A17:02', 'BoLA-N:01601', 'BoLA-104901', 'Patr-B07:02', 'Mamu-A2601', 'BoLA-204601', 'HLA-A9221', 'Mamu-B67', 'Mamu-B39', 'BoLA-204402', 'Patr-A10:01', 'Mamu-B53', 'Patr-B13:01', 'Patr-B08:01', 'H2-Dd', 'BoLA-N:04801', 'Patr-C13:01', 'Mamu-B:01704', 'HLA-A9223', 'BoLA-306602', 'Patr-B23:03', 'Mamu-B04', 'HLA-A9206', 'BoLA-301701', 'Mamu-B6601', 'Patr-A04:01', 'Mamu-B70', 'BoLA-N:00301', 'Patr-C03:01', 'HLA-B9515', 'HLA-B9510', 'HLA-A9209', 'BoLA-N:00601', 'Mamu-A24', 'Patr-B06:01', 'Mamu-A07', 'BoLA-N:04701', 'Patr-B25:01', 'BoLA-200802', 'Patr-B23:02', 'Patr-B14:01', 'Patr-C10:01', 'Mamu-B69', 'BoLA-202601', 'Mamu-A1002:01', 'Patr-C16:01', 'BoLA-205501', 'HLA-A9218', 'SLA-20302', 'Patr-B11:02', 'Mamu-B6002', 'BoLA-N:00501', 'Mamu-A06', 'Patr-B22:02', 'HLA-B9501', 'Patr-B28:01', 'SLA-30503', 'Patr-B02:01', 'HLA-B9503', 'HLA-B9525', 'Mamu-B41', 'BoLA-300201', 'BoLA-302701', 'BoLA-601402', 'Patr-B01:02', 'BoLA-N:02401', 'H2-Kd', 'Patr-C02:01', 'Mamu-A0703', 'HLA-A9202', 'BoLA-N:02702', 'BoLA-204301', 'Patr-B01:01', 'BoLA-305101', 'Patr-B18:01', 'BoLA-102901', 'Mamu-B27', 'HLA-C0409', 'Patr-A14:01', 'HLA-B9529', 'HLA-A9219', 'HLA-A9203', 'BoLA-202603', 'HLA-B9504', 'HLA-B9532', 'Mamu-B61', 'Patr-A09:01', 'BoLA-N:02001', 'Patr-A07:01', 'Mamu-A19', 'Mamu-A01', 'Patr-C15:01', 'BoLA-402401', 'Mamu-A1306', 'BoLA-N:05401', 'HLA-B3938', 'Mamu-B07', 'BoLA-N:00602', 'HLA-B9523', 'H2-Kq', 'Mamu-B004:01', 'Mamu-B5802', 'Mamu-A2201', 'BoLA-N:00201', 'BoLA-203001', 'BoLA-200602', 'SLA-11201', 'Patr-B16:01', 'Mamu-A02', 'HLA-A9226', 'BoLA-103101', 'BoLA-206201', 'Mamu-A20102', 'SLA-107:02', 'Chi-B1201', 'BoLA-T2C', 'Patr-B23:01', 'Mamu-A0506', 'BoLA-500301', 'HLA-A9205', 'SLA-30501', 'Patr-A23:01', 'BoLA-604001', 'BoLA-205601', 'BoLA-206001', 'Patr-B03:02', 'BoLA-N:01302', 'HLA-A9212', 'BoLA-N:03801', 'HLA-B9508', 'Patr-C06:01', 'SLA-11101', 'BoLA-106101', 'Patr-B22:01', 'Mamu-B20', 'Mamu-A25', 'Mamu-B1001', 'Patr-C05:01', 'BoLA-N:02501', 'Patr-B27:01', 'BoLA-300402', 'Mamu-B003:01', 'BoLA-103102', 'BoLA-N:03701', 'BoLA-N:04201', 'Patr-A03:02', 'Patr-A16:01', 'SLA-20102', 'BoLA-205401', 'BoLA-N:04501', 'HLA-A9210', 'Mamu-A23', 'Patr-B07:01', 'HLA-A9204', 'BoLA-300403', 'Mamu-B64', 'BoLA-N:01301', 'SLA-10401', 'Patr-B24:02', 'SLA-11301', 'SLA-20701', 'Mamu-A1011:01', 'BoLA-N:01801', 'BoLA-N:00101', 'Patr-C02:06', 'Mamu-A0505', 'BoLA-305201', 'Eqca-1600101', 'SLA-20502', 'Mamu-B19', 'Mamu-A70103', 'Mamu-B24', 'HLA-C0301', 'BoLA-201602', 'Patr-C12:01', 'Mamu-B3002', 'SLA-30402', 'Patr-B05:02', 'Patr-A08:03', 'Patr-B29:01', 'Eqca-100101', 'HLA-B9516', 'Mamu-B38', 'HLA-A9201', 'H2-Kb', 'Mamu-B22', 'Patr-C03:03', 'SLA-10202', 'Mamu-B71', 'Mamu-A28', 'BoLA-N:03401', 'Mamu-A0511', 'SLA-205:02', 'BoLA-N:03901', 'Mamu-B065:02', 'SLA-10201', 'BoLA-305901', 'BoLA-200601', 'BoLA-N:01201', 'Mamu-B03901', 'Patr-C09:01', 'Patr-A02:01', 'Patr-A05:01', 'HLA-C0502', 'BoLA-100902', 'BoLA-305002', 'Patr-B03:01', 'BoLA-204501', 'Patr-B08:02', 'BoLA-301703', 'BoLA-N:01101', 'BoLA-601501', 'BoLA-N:00901', 'BoLA-306801', 'BoLA-604101', 'BoLA-506401', 'HLA-A9207', 'HLA-B9512', 'Mamu-B17', 'Patr-B26:01', 'Patr-A24:01', 'BoLA-101901', 'BoLA-N:01901', 'Mamu-B08', 'BoLA-300101', 'HLA-B9528', 'Mamu-B44', 'HLA-B9506', 'HLA-B9519', 'Patr-C02:02', 'Patr-C13:02', 'BoLA-202501', 'BoLA-N:05101', 'BoLA-N:00902', 'H-2-Qa2', 'Patr-B10:01', 'BoLA-102001', 'HLA-B0806', 'HLA-B9507', 'BoLA-N:01602', 'Patr-B36:01', 'BoLA-N:02201', 'BoLA-402402', 'BoLA-305801', 'Mamu-B45', 'HLA-C1407', 'BoLA-301001', 'HLA-A9216', 'Patr-B21:01', 'Mamu-A0510', 'Patr-A03:01', 'Patr-C03:02', 'Patr-A08:02', 'HLA-B3573', 'SLA-20401', 'SLA-20101', 'BoLA-204701', 'BoLA-N:02402', 'SLA-10702', 'BoLA-102301', 'SLA-30303', 'Mamu-B48', 'SLA-20402', 'HLA-A0303', 'BoLA-N:02701', 'HLA-A9215', 'Patr-A17:01', 'HLA-B9502', 'Mamu-A21', 'SLA-30302', 'HLA-B9524', 'BoLA-N:02601', 'BoLA-N:05301', 'Patr-C07:01', 'BoLA-305001', 'SLA-204:01', 'HLA-B9513', 'HLA-A0283', 'Patr-A13:01', 'Mamu-B52', 'Mamu-A03', 'Patr-B37:01', 'BoLA-301702', 'Patr-B05:01', 'SLA-20301', 'BoLA-N:01702', 'BoLA-N:05201', 'BoLA-306601', 'Patr-B02:03', 'BoLA-200501', 'BoLA-300102']
#     print(len(a))

# def create_netmhcpan_allele_file():
#     OUTPUT_FILE = DATA_DIR + '/netmhcpan_compatible_alleles.txt'
#     MHC_PSEUDO_FILE = DATA_DIR + '/MHC_pseudo.dat'
#     UNKNOWN_MHC_FILE = DATA_DIR + '/unknown_mhc_alleles.txt'

#     with open(UNKNOWN_MHC_FILE, 'r') as f:
#         unknown_alleles = [_.strip() for _ in f.readlines()]
    
    
#     with open(MHC_PSEUDO_FILE, 'r') as f:
#         # read allele file from NetMHCpan-4.1 in GitLab
#         content_sec = f.readlines()
        
#         # File Format: left column (allele) right column (seq)
#         content_sec = [_.strip().split(' ') for _ in content_sec]

#         pseudo_alleles = []
#         for _ in content_sec:
#             allele = _[0]
#             if '\t' in allele:    
#                 allele = allele.split('\t')[0]
                
#             pseudo_alleles.append(allele)


def compare_allelenames():
    MHC_NAMES_FILE = DATA_DIR + '/MHC_allele_names.txt'
    MHC_PSEUDO_FILE = DATA_DIR + '/MHC_pseudo.dat'
    UNKNOWN_ALLELES_FILE = DATA_DIR + '/unknown_mhc_alleles.txt'

    '''
    Let's see after excluding the unknown alleles from the 'MHC_pseudo.dat' file, if the
    remaining valid alleles matches the 'MHC_allele_names.txt' file from DTU.

    If so, let's use that as the official allele file for netmhcpan-4.1
    '''

    with open(MHC_PSEUDO_FILE, 'r') as f:
        # read allele file from NetMHCpan-4.1 in GitLab
        content_sec = f.readlines()
        
        # File Format: left column (allele) right column (seq)
        content_sec = [_.strip().split(' ') for _ in content_sec]

        pseudo_alleles = []
        for _ in content_sec:
            allele = _[0]
            if '\t' in allele:    
                allele = allele.split('\t')[0]
                
            pseudo_alleles.append(allele)
        pseudo_alleles = list(set(pseudo_alleles))

    with open(UNKNOWN_ALLELES_FILE, 'r') as f:
        unknown_alleles = [_.strip() for _ in f.readlines()]

    # print(pseudo_alleles[:10])
    # print(len(pseudo_alleles))
    # print(len(set(pseudo_alleles)))
    pseudo_alleles_dict = dict(zip(pseudo_alleles, [False]*len(pseudo_alleles)))

    print(len(pseudo_alleles))
    for unknown_allele in unknown_alleles:
        if unknown_allele in pseudo_alleles:
            pseudo_alleles.remove(unknown_allele)
    
    print(len(pseudo_alleles)) 
    #12502
    # len(MHC_allele_names.txt) = 11101

    netmhcpan_alleles_file = DATA_DIR + '/netmhcpan_4_1_alleles.txt'

    with open(netmhcpan_alleles_file, 'w') as f:
        for _ in pseudo_alleles:
            f.write(f'{_}\n')


def test_alleles():
    NETMHCPAN_ALLELES_FILE = DATA_DIR + '/netmhcpan_4_1_alleles.txt'

    with open(NETMHCPAN_ALLELES_FILE, 'r') as f:
        # Read allele file from the DTU
        content_first = f.readlines()
        netmhcpan_alleles = [_.strip() for _ in content_first]
    

    av = AlleleValidator()
    ac = Allele_Autocomplete()

    # unknown alleles
    unknown_alleles = []

    counter = 0
    for toolname in netmhcpan_alleles:
        counter += 1
        iedb_label = av.convert_methodlabel_to_iedblabel(toolname, 'netmhcpan')
        if not iedb_label:
            # iedb_label = av.convert_synonym_to_iedblabel(toolname)
            potential_names = ac.get_allele_suggestions(query=toolname)
            try:
                iedb_label = potential_names[0]['label']
                # Define the curl command
                curl_command = [
                    'curl',
                    '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={iedb_label}&length=9',
                    'http://localhost/tools_api/mhci/'
                ]

                # Run the curl command
                result = subprocess.run(curl_command, capture_output=True, text=True)

                # Run the curl command
                result = subprocess.run(curl_command, capture_output=True, text=True)

                # Print the output
                # print(result.stdout)
                # print(type(result.stdout))
                
                if 'Invalid allele name' in result.stdout:
                    sys.exit()

            except:
                unknown_alleles.append(toolname)

        print(f'{counter}/{len(netmhcpan_alleles)}')
    print('----')
    print(unknown_alleles)

def update_unknown_mhc():
    UNKNOWN_ALLELES_FILE = DATA_DIR + '/unknown_mhc_alleles.txt'

    extra_unknnown_alleles = ['HLA-C03:284', 'Patr-C0303', 'Mamu-A1:11301', 'BoLA-T5', 'SLA-1-YC', 'SLA-2:LWH.AA', 'SLA-2:HB01', 'SLA-1-HB02', 'SLA-1-HB01', 'Mamu-A2:0520', 'BoLA-D18.4', 'SLA-3-TPK', 'SLA-3-HB01', 'SLA-3-YDL', 'SLA-6:0102', 'BoLA-T2c', 'SLA-3-YC', 'SLA-1-YDL01', 'BoLA-AW10', 'SLA-3-YDY01', 'SLA-3-YDY02', 'BoLA-HD6', 'BoLA-gb1.7', 'SLA-1-HB04', 'SLA-2:YC.AA', 'SLA-2:YDL.AA', 'SLA-2-YDL02', 'SLA-3-YTH', 'Mamu-AG:02012', 'HLA-C03:01', 'SLA-2:TPK.AA', 'SLA-3-LWH', 'SLA-3-CDY', 'BoLA-JSP.1', 'SLA-2:CDY.AA', 'BoLA-T7', 'SLA-1-YTH', 'BoLA-amani.1', 'Mamu-A2:0540', 'Mamu-A2:0547', 'SLA-1-CHANGDA', 'SLA-1-LWH', 'BoLA-T2a', 'BoLA-T2b', 'SLA-1-TPK', 'SLA-1-HB03', 'SLA-2:YDY.AA', 'SLA-2:YTH.AA']

    with open(UNKNOWN_ALLELES_FILE, 'a') as f:
        for _ in extra_unknnown_alleles:
            f.write(f'{_}\n')


def update_tools_mapping_file():
    pass


def extra_analysis():
    MHC_NAMES_FILE = DATA_DIR + '/MHC_allele_names.txt'
    MHC_PSEUDO_FILE = DATA_DIR + '/MHC_pseudo.dat'
    UNKNOWN_ALLELES_FILE = DATA_DIR + '/unknown_mhc_alleles.txt'

    with open(MHC_NAMES_FILE, 'r') as f:
        # Read allele file from the DTU
        content_first = f.readlines()
        mhc_names_alleles = [_.strip() for _ in content_first]
        
    with open(MHC_PSEUDO_FILE, 'r') as f:
        # read allele file from NetMHCpan-4.1 in GitLab
        content_sec = f.readlines()
        
        # File Format: left column (allele) right column (seq)
        content_sec = [_.strip().split(' ') for _ in content_sec]

        pseudo_alleles = []
        for _ in content_sec:
            allele = _[0]
            if '\t' in allele:    
                allele = allele.split('\t')[0]
                
            pseudo_alleles.append(allele)
        pseudo_alleles = list(set(pseudo_alleles))

    print(len(mhc_names_alleles)) ## MHC_allele_names.txt -- 11101
    print(len(pseudo_alleles)) ## MHC_pseudo.dat -- 12997

    with open(UNKNOWN_ALLELES_FILE, 'r') as f:
        unknown_alleles = [_.strip() for _ in f.readlines()]

    for _ in unknown_alleles:
        pseudo_alleles.remove(_)
        # Still leaves with 12454

    for _ in mhc_names_alleles:
        pseudo_alleles.remove(_)

    print(pseudo_alleles)
    print(len(pseudo_alleles))
    
def double_check_invalid_alleles():
    UNKNOWN_ALLELES_FILE = DATA_DIR + '/unknown_mhc_alleles.txt'

    with open(UNKNOWN_ALLELES_FILE, 'r') as f:
        unknown_alleles = [_.strip() for _ in f.readlines()]
    

    av = AlleleValidator()
    ac = Allele_Autocomplete()

    # unknown alleles
    unknown_alleles = []

    counter = 0
    for toolname in unknown_alleles:
        counter += 1
        iedb_label = av.convert_methodlabel_to_iedblabel(toolname, 'netmhcpan')
        if not iedb_label:
            # iedb_label = av.convert_synonym_to_iedblabel(toolname)
            potential_names = ac.get_allele_suggestions(query=toolname)
            try:
                iedb_label = potential_names[0]['label']
                # Define the curl command
                curl_command = [
                    'curl',
                    '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={iedb_label}&length=9',
                    'http://localhost/tools_api/mhci/'
                ]

                # Run the curl command
                result = subprocess.run(curl_command, capture_output=True, text=True)

                # Run the curl command
                result = subprocess.run(curl_command, capture_output=True, text=True)

                # Print the output
                # print(result.stdout)
                # print(type(result.stdout))
                
                if 'Invalid allele name' in result.stdout:
                    sys.exit()

            except:
                unknown_alleles.append(toolname)

        print(f'{counter}/{len(unknown_alleles)}')
    print('----')
    print(unknown_alleles)
    print(len(unknown_alleles))




def compare_allele_files2():
    MHC_NAMES_FILE = DATA_DIR + '/MHC_allele_names.txt'
    MHC_PSEUDO_FILE = DATA_DIR + '/MHC_pseudo.dat'

    with open(MHC_NAMES_FILE, 'r') as f:
        # Read allele file from the DTU
        content_first = f.readlines()
        mhc_names_alleles = [_.strip() for _ in content_first]
        
    with open(MHC_PSEUDO_FILE, 'r') as f:
        # read allele file from NetMHCpan-4.1 in GitLab
        content_sec = f.readlines()
        
        # File Format: left column (allele) right column (seq)
        content_sec = [_.strip().split(' ') for _ in content_sec]

        pseudo_alleles = []
        for _ in content_sec:
            allele = _[0]
            if '\t' in allele:    
                allele = allele.split('\t')[0]
                
            pseudo_alleles.append(allele)

    '''
    There are about 2000 more alleles in the MHC_pseudo.dat file (13101)
    than the MHC_allele_names.txt (11101). Note that the MHC_allele_names.txt
    is from DTU, and that's probably the accurate one.

    Let's mark wich alleles exist both files, and which ones don't.
    
    HOW?
    ----
    Take the one with larger amount of alleles, and turn it into a dictionary.
    We will then set key: allele name, value: T/F (depending on if it exists in
    both files).
    '''
    # Handle duplicates:
    # This removes 104 alleles, leaving 12997 unique alleles
    pseudo_alleles = set(pseudo_alleles) 
    pseudo_alleles_dict= dict(zip(pseudo_alleles, [False]*len(pseudo_alleles)))
    
    for allele in mhc_names_alleles:
        if allele in pseudo_alleles_dict:
            pseudo_alleles_dict[allele] = True
    
    counter = 0
    extra_alleles = []
    for k, v in pseudo_alleles_dict.items():
        if not v:
            counter += 1
            extra_alleles.append(k)

    # This shows that there are 1896 alleles that exist only in the pseudo file.
    print(counter)
    
    av = AlleleValidator()
    ac = Allele_Autocomplete()

    # unknown alleles
    unknown_alleles = []
    
    counter = 0
    for toolname in extra_alleles:
        counter += 1
        iedb_label = av.convert_methodlabel_to_iedblabel(toolname, 'netmhcpan')
        if not iedb_label:
            # iedb_label = av.convert_synonym_to_iedblabel(toolname)
            potential_names = ac.get_allele_suggestions(query=toolname)
            try:
                iedb_label = potential_names[0]['label']
            except:
                print(f'{toolname} has no known iedb label.')
                unknown_alleles.append(toolname)
                print(f'{counter}/{len(extra_alleles)}')
                continue

        print(toolname, iedb_label)

        # Define the curl command
        curl_command = [
            'curl',
            '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={iedb_label}&length=9',
            'http://localhost/tools_api/mhci/'
        ]

        # Run the curl command
        result = subprocess.run(curl_command, capture_output=True, text=True)

        # Print the output
        # print(result.stdout)
        # print(type(result.stdout))

        if 'Invalid allele name' in result.stdout:
            print("!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!")
            unknown_alleles.append(toolname)
            # sys.exit()
        
        print(f'{counter}/{len(extra_alleles)}')
    
    print('----')
    print(unknown_alleles)

    UNKNOWN_ALLELE_FILE = DATA_DIR + '/unknown_mhc_alleles2.txt'

    with open(UNKNOWN_ALLELE_FILE, 'w') as f:
        for _ in unknown_alleles:
            f.write(f'{_}\n')

    # for toolname in extra_alleles:
    #     iedb_label = av.convert_methodlabel_to_iedblabel(toolname, 'netmhcpan')
    #     if not iedb_label:
    #         # iedb_label = av.convert_synonym_to_iedblabel(toolname)
    #         potential_names = ac.get_allele_suggestions(query=toolname)
    #         try:
    #             iedb_label = potential_names[0]['label']
    #             # Define the curl command
    #             curl_command = [
    #                 'curl',
    #                 '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={iedb_label}&length=9',
    #                 'http://localhost/tools_api/mhci/'
    #             ]

    #             # Run the curl command
    #             result = subprocess.run(curl_command, capture_output=True, text=True)

    #             # Run the curl command
    #             result = subprocess.run(curl_command, capture_output=True, text=True)

    #             # Print the output
    #             print(result.stdout)
    #             print(type(result.stdout))
                
    #             if 'Invalid allele name' in result.stdout:
    #                 sys.exit()

    #         except:
    #             unknown_alleles.append(toolname)

    # print('----')
    # print(unknown_alleles)

    # UNKNOWN_ALLELE_FILE = DATA_DIR + '/unknown_mhc_alleles2.txt'

    # with open(UNKNOWN_ALLELE_FILE, 'w') as f:
    #     for _ in unknown_alleles:
    #         f.write(f'{_}\n')


def test1():
    alleles = ['BoLA-506401', 'Mamu-A2201', 'BoLA-204301']
    # alleles = ['BoLA-204301']

    av = AlleleValidator()
    ac = Allele_Autocomplete()

    for toolname in alleles:
        iedb_label = av.convert_methodlabel_to_iedblabel(toolname, 'netmhcpan')
        if not iedb_label:
            # iedb_label = av.convert_synonym_to_iedblabel(toolname)
            potential_names = ac.get_allele_suggestions(query=toolname)
            print(potential_names)
            try:
                iedb_label = potential_names[0]['label']
            except:
                print(f'{toolname} has no known iedb label.')
                continue

        print(toolname, iedb_label)
        mro = av.convert_iedblabel_to_mroid(iedb_label, 'netmhcpan')
        print(mro)
        actual_iedb_label = av.convert_mroid_to_methodlabel(mro, 'netmhcpan')
        print(actual_iedb_label)

        # Define the curl command
        curl_command = [
            'curl',
            '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={actual_iedb_label}&length=9',
            'http://localhost/tools_api/mhci/'
        ]

        print(curl_command)

        # Run the curl command
        result = subprocess.run(curl_command, capture_output=True, text=True)
        print("-----RESULT-------")
        print(result)


def analyze_model_list_file():
    MODEL_LIST_FILE = DATA_DIR + '/model_list.txt'

    with open(MODEL_LIST_FILE, 'r') as f:
        model_alleles = []
        for _ in f.readlines():
            _ = _.strip().split('\t')[0]

            '''
            Each allele has this suffix '-XX' where X is a digit. We need to remove that.
            This will divide the allele into 3 sections, but we only need to discard
            the last portion (the suffix '-XX')
            '''
            _ = _.split('-')
            _ = '-'.join(_[:-1])
            model_alleles.append(_)

        '''
        Because we removed all the suffixes, this will have a lot of duplicates.
        Remove duplicates.
        '''
        model_alleles = list(set(model_alleles))
    
    print(len(model_alleles)) # 3582 alleles

    unknown_alleles = []
    counter = 0
    for toolname in model_alleles:
        counter += 1
        # Define the curl command
        curl_command = [
            'curl',
            '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={toolname}&length=9',
            'http://localhost/tools_api/mhci/'
        ]

        print(curl_command)

        # Run the curl command
        result = subprocess.run(curl_command, capture_output=True, text=True)
        print("-----RESULT-------")
        print(result)
        
        if 'Invalid allele name' in result.stdout:
            print("!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!")
            unknown_alleles.append(toolname)
    
        print(f'{counter}/{len(model_alleles)}')

    UNKNOWN_ALLELE_FILE = DATA_DIR + '/unknown_mhc_alleles_from_model_list.txt'

    with open(UNKNOWN_ALLELE_FILE, 'w') as f:
        for _ in unknown_alleles:
            f.write(f'{_}\n')
   


def double_check_unknown_model_list_alleles():
    UNKNOWN_ALLELE_FILE = DATA_DIR + '/unknown_mhc_alleles_from_model_list.txt'

    with open(UNKNOWN_ALLELE_FILE, 'r') as f:
        unknown_alleles = [_.strip() for _ in f.readlines()]
    
    av = AlleleValidator()
    ac = Allele_Autocomplete()

    counter = 0
    truly_unknown_alleles = []
    for unknown_toolname in unknown_alleles:
        counter += 1
        iedb_label = av.convert_methodlabel_to_iedblabel(unknown_toolname, 'netmhcpan')
        if not iedb_label:
            # iedb_label = av.convert_synonym_to_iedblabel(toolname)
            potential_names = ac.get_allele_suggestions(query=unknown_toolname)
            print(potential_names)
            try:
                iedb_label = potential_names[0]['label']
            except:
                print(f'{unknown_toolname} has no known iedb label.')
                truly_unknown_alleles.append(unknown_toolname)
                continue

        print(unknown_toolname, iedb_label)
        mro = av.convert_iedblabel_to_mroid(iedb_label, 'netmhcpan')
        actual_iedb_label = av.convert_mroid_to_methodlabel(mro, 'netmhcpan')
        print(actual_iedb_label, mro)

        # Define the curl command
        curl_command = [
            'curl',
            '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={actual_iedb_label}&length=9',
            'http://localhost/tools_api/mhci/'
        ]

        print(curl_command)

        # Run the curl command
        result = subprocess.run(curl_command, capture_output=True, text=True)
        print("-----RESULT-------")
        print(result)

        if 'Invalid allele name' in result.stdout:
            print("!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!")
            truly_unknown_alleles.append(unknown_toolname)
    
        print(f'{counter}/{len(unknown_alleles)}')

    UNKNOWN_ALLELE_FILE = DATA_DIR + '/unknown_mhc_alleles_from_model_list_truly.txt'

    with open(UNKNOWN_ALLELE_FILE, 'w') as f:
        for _ in truly_unknown_alleles:
            f.write(f'{_}\n')
   

def triple_check_unknown_model_list_alleles():
    UNKNOWN_ALLELE_FILE = DATA_DIR + '/unknown_mhc_alleles_from_model_list_truly.txt'

    with open(UNKNOWN_ALLELE_FILE, 'r') as f:
        unknown_alleles = [_.strip() for _ in f.readlines()]
    
    av = AlleleValidator()
    ac = Allele_Autocomplete()

    counter = 0
    truly_unknown_alleles = []
    for unknown_toolname in unknown_alleles:
        counter += 1
        iedb_label = av.convert_methodlabel_to_iedblabel(unknown_toolname, 'netmhcpan')
        if not iedb_label:
            # iedb_label = av.convert_synonym_to_iedblabel(toolname)
            potential_names = ac.get_allele_suggestions(query=unknown_toolname)
            print(potential_names)
            try:
                iedb_label = potential_names[0]['label']
            except:
                print(f'{unknown_toolname} has no known iedb label.')
                truly_unknown_alleles.append(unknown_toolname)
                continue

        # print(unknown_toolname, iedb_label)
        # mro = av.convert_iedblabel_to_mroid(iedb_label, 'netmhcpan')
        # actual_iedb_label = av.convert_mroid_to_methodlabel(mro, 'netmhcpan')
        # print(actual_iedb_label, mro)

        # Define the curl command
        curl_command = [
            'curl',
            '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={iedb_label}&length=9',
            'http://localhost/tools_api/mhci/'
        ]

        print(curl_command)

        # Run the curl command
        result = subprocess.run(curl_command, capture_output=True, text=True)
        print("-----RESULT-------")
        print(result)

        if 'Invalid allele name' in result.stdout:
            print("!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!_!")
            truly_unknown_alleles.append(unknown_toolname)
    
        print(f'{counter}/{len(unknown_alleles)}')

    UNKNOWN_ALLELE_FILE = DATA_DIR + '/unknown_mhc_alleles_from_model_list_FINAL.txt'

    with open(UNKNOWN_ALLELE_FILE, 'w') as f:
        for _ in truly_unknown_alleles:
            if _ == 'Gogo-B0101':
                continue

            f.write(f'{_}\n')
            print(_)


def create_valid_alleles_from_model_list_file():
    UNKNOWN_ALLELE_FILE = DATA_DIR + '/unknown_mhc_alleles_from_model_list_FINAL.txt'
    MODEL_LIST_FILE = DATA_DIR + '/model_list.txt'

    with open(MODEL_LIST_FILE, 'r') as f:
        model_alleles = []
        for _ in f.readlines():
            _ = _.strip().split('\t')[0]

            '''
            Each allele has this suffix '-XX' where X is a digit. We need to remove that.
            This will divide the allele into 3 sections, but we only need to discard
            the last portion (the suffix '-XX')
            '''
            _ = _.split('-')
            _ = '-'.join(_[:-1])
            model_alleles.append(_)

        '''
        Because we removed all the suffixes, this will have a lot of duplicates.
        Remove duplicates.
        '''
        model_alleles = list(set(model_alleles))
    
    # print(model_alleles)
    print(len(model_alleles)) # 3582 alleles

    with open(UNKNOWN_ALLELE_FILE, 'r') as f:
        unknown_alleles = [_.strip() for _ in f.readlines()]

    # Remove these unknown alleles from the model_list
    for _ in unknown_alleles:
        model_alleles.remove(_)

    VALID_MHC_LIST_FILE = DATA_DIR + '/model_list_valid.txt'

    with open(VALID_MHC_LIST_FILE, 'w') as f:
        for _ in model_alleles:
            f.write(f'{_}\n')
    

def check_existence_of_valid_alleles_from_model_list():
    PDIR = str(Path(__file__).parent.parent)
    MODEL_LIST_FILE = DATA_DIR + '/model_list_valid.txt'
    TOOLS_MAPPING_FILE = PDIR + '/tools-mapping.tsv'

    tm_df = pd.read_csv(TOOLS_MAPPING_FILE, sep='\t')

    # Filter the DF so that there's only 'netmhcpan' alleles
    tm_df = tm_df[tm_df['Tool']=='netmhcpan']
    print(tm_df.head())
    print(len(tm_df))

    # with open()

    

def read_mhc_allele_names(allele_file):
    with open(allele_file, 'r') as f:
        # Read allele file from the DTU
        content_first = f.readlines()
        mhc_names_alleles = [_.strip() for _ in content_first]
    
    return list(set(mhc_names_alleles))

def read_mhc_pseudo(allele_file):
    with open(allele_file, 'r') as f:
        # read allele file from NetMHCpan-4.1 in GitLab
        content_sec = f.readlines()
        
        # File Format: left column (allele) right column (seq)
        content_sec = [_.strip().split(' ') for _ in content_sec]

        pseudo_alleles = []
        for _ in content_sec:
            allele = _[0]
            if '\t' in allele:    
                allele = allele.split('\t')[0]
                
            pseudo_alleles.append(allele)

    return list(set(pseudo_alleles))

def read_allelenames(allele_file):
    with open(allele_file, 'r') as f:
        # (tool_label iedb_label)
        content = [row.split(' ')[0] for row in f.readlines()]
    
    return list(set(content))

def read_model_list(allele_file):
    with open(allele_file, 'r') as f:
        model_alleles = []
        for _ in f.readlines():
            _ = _.strip().split('\t')[0]

            '''
            Each allele has this suffix '-XX' where X is a digit. We need to remove that.
            This will divide the allele into 3 sections, but we only need to discard
            the last portion (the suffix '-XX')
            '''
            _ = _.split('-')
            _ = '-'.join(_[:-1])
            model_alleles.append(_)

    '''
    Because we removed all the suffixes, this will have a lot of duplicates.
    Remove duplicates.
    '''
    return list(set(model_alleles))

def read_unknown_alleles_file(allele_file):
    with open(allele_file, 'r') as f:
        unknown_alleles = [_.strip() for _ in f.readlines()]
    
    return unknown_alleles

def run_curl_command_with(allele):
    # Define the curl command
    curl_command = [
        'curl',
        '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={allele}&length=9',
        'http://localhost/tools_api/mhci/'
    ]

    # Run the curl command
    result = subprocess.run(curl_command, capture_output=True, text=True)
    print(result)

    if 'Invalid allele name' in result.stdout:
        result = None

    return result

def compile_all_alleles():
    MHC_NAMES_FILE = DATA_DIR + '/MHC_allele_names.txt'
    MHC_PSEUDO_FILE = DATA_DIR + '/MHC_pseudo.dat'
    ALLELENAMES_FILE = DATA_DIR + '/allelenames'
    MODEL_LIST_FILE = DATA_DIR + '/model_list.txt'
    UNKNOWN_ALLELES_FILE = DATA_DIR + '/unknown_alleles.txt'

    mhc_names_list = read_mhc_allele_names(MHC_NAMES_FILE)
    mhc_pseudo_list = read_mhc_pseudo(MHC_PSEUDO_FILE)
    allelenames_list = read_allelenames(ALLELENAMES_FILE)
    model_list_list = read_model_list(MODEL_LIST_FILE)
    unknown_alleles_list = read_unknown_alleles_file(UNKNOWN_ALLELES_FILE)

    allele_files_list = [
        ('MHC_allele_names.txt', mhc_names_list), 
        ('MHC_pseudo.dat', mhc_pseudo_list), 
        ('allelenames', allelenames_list), 
        ('model_list.txt', model_list_list)]
    netmhcpan_list = []

    '''
    Compile all the alleles into a list
    '''
    for each_allele_list in allele_files_list:
        print(f'Iterating alleles from {each_allele_list[0]}')
        allele_list = each_allele_list[1]

        for allele in allele_list:
            if allele not in netmhcpan_list:
                netmhcpan_list.append(allele)
    
    '''
    Remove unknown alleles
    '''
    for unknown_allele in unknown_alleles_list:
        netmhcpan_list.remove(unknown_allele)
    
    '''
    Run each allele against the netmhcpan tool, and see if they work.
    If it works, they are the true 'tool name'. If it doesn't,
    use allele-validator or allele-autocomplete to find the 'iedb label' and
    try to run it against the netmhcpan tool. If it works, then save the
    'iedb label' as the 'tool name'. All the ones that didn't work should be
    saved as a synonym to that allele.
    '''
    av = AlleleValidator()
    ac = Allele_Autocomplete()

    netmhcpan_dict = {
        'Tool Label': [],
        'IEDB Label': [],
        'Synonyms': [],
    }
    # Store alleles that curl command can't find.
    unavailable_alleles = []

    # suspicious alleles -- must be verified manually
    suspicious_alleles = []

    counter = 1
    for toolname in netmhcpan_list:
        print(f'{counter}/{len(netmhcpan_list)}')
        counter += 1

        # Because toolname didn't work, try getting the corresponding iedb_label,
        # then run the iedb_label against the curl command.
        iedb_label = av.convert_methodlabel_to_iedblabel(toolname, 'netmhcpan')
        if not iedb_label:
            potential_names = ac.get_allele_suggestions(query=toolname)
            try:
                iedb_label = potential_names[0]['label']
            except:
                suspicious_alleles.append(toolname)
                # raise ValueError(f'{toolname} does not exist in the allele-autocomplete.')

        if run_curl_command_with(toolname):
            # Add these when toolname does work with the curl command
            netmhcpan_dict['Tool Label'].append(toolname)
            netmhcpan_dict['IEDB Label'].append(iedb_label)
            netmhcpan_dict['Synonyms'].append(iedb_label)
            continue
        
        if run_curl_command_with(iedb_label):
            # Add these when iedb_label does work with the curl command
            netmhcpan_dict['Tool Label'].append(iedb_label)
            netmhcpan_dict['IEDB Label'].append(iedb_label)
            netmhcpan_dict['Synonyms'].append(toolname)
            continue
        
        
        print('something is weird....')
        print(toolname, iedb_label)
        unavailable_alleles.append((toolname, iedb_label))
    
    
    df = pd.DataFrame(netmhcpan_dict)
    df.to_csv('netmhcpan_4_1_alleles.tsv', sep='\t', index=False)
    print("Finished.")
    print('=======================')
    print(unavailable_alleles)
    # unavailable_alleles = [
    #     ('Mamu-A1:00401', 'Mamu-A1*004:01'), ('Mamu-B:04608', 'Mamu-B*046:08'), ('Mamu-A5:3003', 'Mamu-A5*30:03'), ('Mamu-B:06001', 'Mamu-B*060:01'), ('Patr-B1601', 'Patr-B*16:01'), ('Mamu-B:04503', 'Mamu-B*045:03'), ('HLA-B07:44', 'HLA-B*07:44N'), ('Mamu-A1:11301', 'Mamu-A1*008:01'), ('Mamu-B:01902', 'Mamu-B*019:02'), ('Patr-A0101', 'Patr-A*01:01'), ('Patr-B2202', 'Patr-B*22:02'), ('Mamu-A1:02602', 'Mamu-A1*026:02'), ('Mamu-A1:05603', 'Mamu-A1*056:03'), ('HLA-C04:338', 'HLA-C*04:338Q'), ('Mamu-A1:03202', 'Mamu-A1*032:02'), ('Patr-B1901', 'Patr-B*19:01'), ('Mamu-B:05602', 'Mamu-B*056:02'), ('Patr-B2701', 'Patr-B*27:01'), ('Mamu-A1:02502', 'Mamu-A1*025:02'), ('Mamu-A2:0525', 'Mamu-A2*05:25'), ('Mamu-A2:0531', 'Mamu-A2*05:31'), ('Patr-B3601', 'Patr-B*36:01'), ('Mamu-A1:05001', 'Mamu-A1*050:01'), ('Mamu-B:04602', 'Mamu-B*046:02'), ('Mamu-A2:0534', 'Mamu-A2*05:34'), ('Mamu-A2:0510', 'Mamu-A2*05:10'), ('Mamu-A1:10502', 'Mamu-A1*105:02'), ('Mamu-B:04901', 'Mamu-B*049:01'), ('Mamu-A1:00104', 'Mamu-A1*001:04'), ('Patr-A0802', 'Patr-A*08:02'), ('Patr-A0902', 'Patr-A*09:02'), ('Mamu-A1:00303', 'Mamu-A1*003:03'), ('Mamu-A1:01807', 'Mamu-A1*018:07'), ('Patr-A0404', 'Patr-A*04:04'), ('SLA-3:0302', 'SLA-3*03:02'), ('Mamu-B:06402', 'Mamu-B*064:02'), ('SLA-6:0105', 'SLA-6*05:01'), ('SLA-2:0101', 'SLA-2*01:01'), ('HLA-C03:01', None), ('Mamu-B:08901', 'Mamu-B*089:01'), ('Mamu-A1:03203', 'Mamu-A1*032:03'), ('Patr-C1201', 'Patr-C*12:01'), ('SLA-3:0303', 'SLA-3*03:03'), ('Mamu-A1:00601', 'Mamu-A1*006:01'), ('Patr-C0201', 'Patr-C*02:01'), ('Mamu-B:04403', 'Mamu-B*044:03'), ('Mamu-A5:3002', 'Mamu-A5*30:02'), ('Mamu-B:06503', 'Mamu-B*065:03'), ('Mamu-A1:06001', 'Mamu-A1*060:01'), ('Mamu-B:09401', 'Mamu-B*094:01'), ('Mamu-A1:01804', 'Mamu-A1*018:04'), ('Mamu-B:05302', 'Mamu-B*053:02'), ('Mamu-A2:0517', 'Mamu-A2*05:17'), ('Mamu-B:08801', 'Mamu-B*088:01'), ('Mamu-A1:01103', 'Mamu-A1*011:03'), ('Patr-A0201', 'Patr-A*02:01'), ('Mamu-A1:01808', 'Mamu-A1*018:08'), ('Mamu-B:07202', 'Mamu-B*072:02'), ('Mamu-A2:0519', 'Mamu-A2*05:19'), ('Mamu-B:04604', 'Mamu-B*046:04'), ('Mamu-B:08501', 'Mamu-B*085:01'), ('Mamu-A4:0202', 'Mamu-A4*02:02'), ('Mamu-A1:00306', 'Mamu-A1*003:06'), ('Patr-B0702', 'Patr-B*07:02'), ('Mamu-A1:11001', 'Mamu-A1*110:01'), ('Patr-B2101', 'Patr-B*21:01'), ('Mamu-A1:05602', 'Mamu-A1*056:02'), ('Mamu-B:07401', 'Mamu-B*074:01'), ('Mamu-B:09601', 'Mamu-B*063:03'), ('Mamu-B:00301', 'Mamu-B*003:01'), ('Mamu-B:00101', 'Mamu-B*001:01'), ('Patr-B0601', 'Patr-B*06:01'), ('HLA-C07:432', 'HLA-C*07:432Q'), ('Mamu-B:06804', 'Mamu-B*068:04'), ('Mamu-B:04802', 'Mamu-B*048:02'), ('Mamu-A1:00310', 'Mamu-A1*003:10'), ('SLA-1:0202', 'SLA-1*02:02'), ('Patr-C0401', 'Patr-C*04:01'), ('Mamu-B:02401', 'Mamu-B*024:01'), ('Mamu-A2:0546', 'Mamu-A2*05:46'), ('Mamu-B:00703', 'Mamu-B*007:03'), ('Mamu-B:01501', 'Mamu-B*015:01'), ('Patr-B1102', 'Patr-B*11:02'), ('Mamu-B:02301', 'Mamu-B*023:01'), ('Mamu-A3:1303', 'Mamu-A3*13:03'), ('SLA-2:1002', 'SLA-2*10:02'), ('Patr-B1602', 'Patr-B*16:02'), ('Mamu-A3:1305', 'Mamu-A3*13:05'), ('Patr-A0401', 'Patr-A*04:01'), ('Patr-C0601', 'Patr-C*06:01'), ('Mamu-A7:0102', 'Mamu-A7*01:02'), ('Mamu-A1:10801', 'Mamu-A1*108:01'), ('HLA-A01:281', 'HLA-A*01:281Q'), ('Mamu-B:05201', 'Mamu-B*052:01'), ('Mamu-A2:05030', 'Mamu-A2*05:30'), ('Mamu-B:07901', 'Mamu-B*079:01'), ('Mamu-A2:0102', 'Mamu-A2*01:02'), ('Mamu-B:03802', 'Mamu-B*038:02'), ('Mamu-A6:0102', 'Mamu-A6*01:02'), ('SLA-3:0503', 'SLA-3*05:03'), ('Mamu-A1:04001', 'Mamu-A1*040:01'), ('Mamu-A1:10901', 'Mamu-A1*109:01'), ('Mamu-A1:05501', 'Mamu-A1*055:01'), ('Mamu-A1:01801', 'Mamu-A1*018:01'), ('SLA-3:0701', 'SLA-3*07:01'), ('Mamu-A2:2402', 'Mamu-A2*24:02'), ('Mamu-B:03101', 'Mamu-B*031:01'), ('Mamu-B:01703', 'Mamu-B*017:03'), ('Mamu-B:06501', 'Mamu-B*065:01'), ('SLA-3:0301', 'SLA-3*03:01'), ('Patr-A1301', 'Patr-A*13:01'), ('Patr-C0204', 'Patr-C*02:04'), ('Mamu-A1:02806', 'Mamu-A1*028:06'), ('Mamu-B:08202', 'Mamu-B*082:02'), ('SLA-2:0402', 'SLA-2*04:02'), ('Mamu-A1:05701', 'Mamu-A1*057:01'), ('Mamu-B:03701', 'Mamu-B*037:01'), ('Mamu-B:01001', 'Mamu-B*010:01'), ('Mamu-A2:05040', 'Mamu-A2*05:40'), ('Mamu-A2:0524', 'Mamu-A2*05:24'), ('Mamu-B:00602', 'Mamu-B*006:02'), ('Mamu-A4:0301', 'Mamu-A4*03:01'), ('Mamu-A5:3005', 'Mamu-A5*30:05'), ('Mamu-A1:00302', 'Mamu-A1*003:02'), ('Mamu-B:07601', 'Mamu-B*076:01'), ('Mamu-B:01101', 'Mamu-B*011:01'), ('Mamu-B:03501', 'Mamu-B*035:01'), ('SLA-1:0702', 'SLA-1*07:02'), ('Mamu-A1:10501', 'Mamu-A1*105:01'), ('Mamu-A1:02201', 'Mamu-A1*022:01'), ('Mamu-A6:0101', 'Mamu-A6*01:01'), ('Mamu-A1:00602', 'Mamu-A1*006:02'), ('Patr-B2402', 'Patr-B*24:02'), ('Mamu-A1:00308', 'Mamu-A1*003:08'), ('Mamu-B:01801', 'Mamu-B*018:01'), ('Mamu-A2:0527', 'Mamu-A2*05:27'), ('Mamu-B:02902', 'Mamu-B*029:02'), ('Mamu-A1:07401', 'Mamu-A1*074:01'), ('SLA-3:0101', 'SLA-3*01:01'), ('Mamu-B:03004', 'Mamu-B*030:04'), ('Patr-B0201', 'Patr-B*02:01'), ('Patr-B2303', 'Patr-B*22:03'), ('Patr-B0102', 'Patr-B*01:02'), ('Mamu-B:01702', 'Mamu-B*017:02'), ('Mamu-A1:04101', 'Mamu-A1*041:01'), ('Mamu-B:04701', 'Mamu-B*047:01'), ('Mamu-B:04609', 'Mamu-B*046:09'), ('Mamu-B:09201', 'Mamu-B*092:01'), ('Patr-C1001', 'Patr-C*10:01'), ('Patr-C1601', 'Patr-C*16:01'), ('Patr-B2601', 'Patr-B*26:01'), ('Mamu-A2:2403', 'Mamu-A2*24:03'), ('Mamu-A1:02804', 'Mamu-A1*028:04'), ('Mamu-B:02701', 'Mamu-B*027:01'), ('SLA-1:0201', 'SLA-1*02:01'), ('Mamu-A1:02603', 'Mamu-A1*026:03'), ('Patr-A1101', 'Patr-A*11:01'), ('Mamu-A1:01906', 'Mamu-A1*019:06'), ('Mamu-B:01901', 'Mamu-B*019:01'), ('Mamu-B:06502', 'Mamu-B*065:02'), ('Mamu-B:07902', 'Mamu-B*079:02'), ('Patr-A0901', 'Patr-A*09:01'), ('Mamu-A1:00201', 'Mamu-A1*002:01'), ('Mamu-B:02803', 'Mamu-B*028:03'), ('Mamu-A2:0512', 'Mamu-A2*05:12'), ('Patr-B0701', 'Patr-B*07:01'), ('Mamu-B:06801', 'Mamu-B*068:01'), ('HLA-C03:23', 'HLA-C*03:23N'), ('SLA-2:0102', 'SLA-2*01:02'), ('Mamu-A4:1406', 'Mamu-A4*14:06'), ('Mamu-A1:05302', 'Mamu-A1*053:02'), ('Mamu-B:08502', 'Mamu-B*085:02'), ('Mamu-B:00701', 'Mamu-B*007:01'), ('Mamu-A2:0505', 'Mamu-A2*05:05'), ('Mamu-B:09501', 'Mamu-B*095:01'), ('SLA-6:0103', 'SLA-6*03:01'), ('Patr-A1201', 'Patr-A*12:01'), ('HLA-C12:139', 'HLA-C*12:139Q'), ('Mamu-A1:00402', 'Mamu-A1*004:02'), ('Patr-B1401', 'Patr-B*14:01'), ('Mamu-B:00502', 'Mamu-B*005:02'), ('SLA-1:1201', 'SLA-1*12:01'), ('HLA-A01:159', 'HLA-A*01:159Q'), ('Patr-A0602', 'Patr-A*06:02'), ('Patr-B2001', 'Patr-B*20:01'), ('Patr-B3501', 'Patr-B*35:01'), ('Mamu-B:04404', 'Mamu-B*044:04'), ('Mamu-A2:0507', 'Mamu-A2*05:07'), ('Mamu-A1:02203', 'Mamu-A1*022:03'), ('Mamu-B:03102', 'Mamu-B*031:02'), ('Patr-B1202', 'Patr-B*12:02'), ('Mamu-B:06901', 'Mamu-B*069:01'), ('Mamu-B:01401', 'Mamu-B*014:01'), ('HLA-B18:106', 'HLA-B*18:106Q'), ('Mamu-A1:08101', 'Mamu-A1*081:01'), ('Mamu-B:04201', 'Mamu-B*042:01'), ('Mamu-A1:04201', 'Mamu-A1*042:01'), ('Mamu-B:08102', 'Mamu-B*081:02'), ('SLA-3:0602', 'SLA-3*06:02'), ('Mamu-A5:3004', 'Mamu-A5*30:04'), ('Mamu-A2:0523', 'Mamu-A2*05:23'), ('Mamu-B:05105', 'Mamu-B*051:05'), ('Mamu-A4:0101', 'Mamu-A4*01:01'), ('Mamu-A1:02805', 'Mamu-A1*028:05'), ('Mamu-B:04601', 'Mamu-B*046:01'), ('Mamu-A2:0537', 'Mamu-A2*05:37'), ('Mamu-B:00201', 'Mamu-B*002:01'), ('Mamu-B:04501', 'Mamu-B*045:01'), ('Mamu-A1:10504', 'Mamu-A1*105:04'), ('Patr-B0401', 'Patr-B*04:01'), ('Mamu-B:10001', 'Mamu-B*100:01'), ('SLA-2:1101', 'SLA-2*11:01'), ('Mamu-A1:02801', 'Mamu-A1*028:01'), ('Mamu-B:04705', 'Mamu-B*047:05'), ('Mamu-B:04301', 'Mamu-B*043:01'), ('Mamu-A1:05301', 'Mamu-A1*053:01'), ('Mamu-A1:11101', 'Mamu-A1*111:01'), ('SLA-2:0302', 'SLA-2*03:02'), ('Mamu-A2:0521', 'Mamu-A2*05:21'), ('SLA-1:1101', 'SLA-1*11:01'), ('Mamu-B:04605', 'Mamu-B*046:05'), ('Mamu-A1:06101', 'Mamu-A1*061:01'), ('Mamu-A1:00701', 'Mamu-A1*007:01'), ('Mamu-B:08001', 'Mamu-B*080:01'), ('Mamu-A1:01802', 'Mamu-A1*018:02'), ('Mamu-A2:0103', 'Mamu-A2*01:03'), ('Mamu-B:02802', 'Mamu-B*028:02'), ('Mamu-A1:02501', 'Mamu-A1*025:01'), ('Mamu-A2:0529', 'Mamu-A2*05:29'), ('Mamu-B:02101', 'Mamu-B*021:01'), ('Patr-A1401', 'Patr-A*14:01'), ('Mamu-B:02801', 'Mamu-B*028:01'), ('Mamu-A1:10503', 'Mamu-A1*105:03'), ('Mamu-A1:00105', 'Mamu-A1*001:05'), ('Patr-B3001', 'Patr-B*30:01'), ('Mamu-A2:05020', 'Mamu-A2*05:20'), ('Mamu-A1:01102', 'Mamu-A1*011:02'), ('Mamu-A3:1313', 'Mamu-A3*13:13'), ('Mamu-B:09801', 'Mamu-B*098:01'), ('Patr-B1801', 'Patr-B*18:01'), ('Mamu-A1:00705', 'Mamu-A1*007:05'), ('HLA-C07:713', 'HLA-C*07:713Q'), ('Mamu-B:03601', 'Mamu-B*036:01'), ('Mamu-B:04607', 'Mamu-B*046:07'), ('Patr-A0701', 'Patr-A*07:01'), ('Patr-B0301', 'Patr-B*03:01'), ('Mamu-B:07702', 'Mamu-B*077:02'), ('HLA-B27:185', 'HLA-B*27:185Q'), ('Mamu-B:05901', 'Mamu-B*013:03'), ('Mamu-A2:0533', 'Mamu-A2*05:33'), ('SLA-1:0701', 'SLA-1*07:01'), ('Mamu-A4:0302', 'Mamu-A4*03:02'), ('Patr-A1703', 'Patr-A*17:03'), ('Mamu-A1:01907', 'Mamu-A1*019:07'), ('Mamu-A1:05901', 'Mamu-A1*059:01'), ('Mamu-A1:09101', 'Mamu-A1*091:01'), ('Mamu-B:04702', 'Mamu-B*047:02'), ('Mamu-A1:01803', 'Mamu-A1*018:03'), ('Patr-B0802', 'Patr-B*08:02'), ('Mamu-A2:0509', 'Mamu-A2*05:09'), ('Patr-A1502', 'Patr-A*15:02'), ('Mamu-A1:06301', 'Mamu-A1*063:01'), ('Mamu-B:04001', 'Mamu-B*015:04'), ('Mamu-A1:00103', 'Mamu-A1*001:03'), ('Mamu-B:03301', 'Mamu-B*033:01'), ('Mamu-B:06701', 'Mamu-B*067:01'), ('Mamu-B:04617', 'Mamu-B*046:17'), ('Mamu-A1:05201', 'Mamu-A1*052:01'), ('Mamu-B:07701', 'Mamu-B*077:01'), ('Mamu-B:06903', 'Mamu-B*069:03'), ('Mamu-B:01502', 'Mamu-B*015:02'), ('Patr-B1703', 'Patr-B*17:03'), ('Patr-A0801', 'Patr-A*08:01'), ('Mamu-A2:0518', 'Mamu-A2*05:18'), ('Mamu-A2:0522', 'Mamu-A2*05:22'), ('Mamu-A6:0103', 'Mamu-A6*01:03'), ('Mamu-A1:08501', 'Mamu-A1*085:01'), ('Mamu-B:03901', 'Mamu-B*039:01'), ('Mamu-A3:1308', 'Mamu-A3*13:08'), ('Mamu-A7:0103', 'Mamu-A7*01:03'), ('Mamu-B:00704', 'Mamu-B*007:04'), ('Mamu-A1:07403', 'Mamu-A1*074:03'), ('Mamu-B:08601', 'Mamu-B*086:01'), ('Patr-C0801', 'Patr-C*08:01'), ('Mamu-A1:01903', 'Mamu-A1*019:03'), ('Patr-A1501', 'Patr-A*15:01'), ('Mamu-B:09301', 'Mamu-B*093:01'), ('Mamu-B:05301', 'Mamu-B*053:01'), ('Mamu-A2:0514', 'Mamu-A2*05:14'), ('Mamu-A2:0536', 'Mamu-A2*05:36'), ('SLA-2:0202', 'SLA-2*02:02'), ('Mamu-B:08101', 'Mamu-B*081:01'), ('Mamu-B:05601', 'Mamu-B*056:01'), ('Patr-A1701', 'Patr-A*17:01'), ('Mamu-A1:05601', 'Mamu-A1*056:01'), ('Mamu-B:01601', 'Mamu-B*016:01'), ('Mamu-A3:1312', 'Mamu-A3*13:12'), ('Mamu-A1:01806', 'Mamu-A1*018:06'), ('Patr-A0301', 'Patr-A*03:01'), ('Mamu-A1:02601', 'Mamu-A1*026:01'), ('Mamu-A1:04501', 'Mamu-A1*045:01'), ('HLA-A24:329', 'HLA-A*24:329Q'), ('Gogo-B0101', 'Gogo-B*01:01'), ('Mamu-B:07301', 'Mamu-B*073:01'), ('Mamu-B:06101', 'Mamu-B*061:01'), ('Mamu-B:09701', 'Mamu-B*097:01'), ('Mamu-B:00102', 'Mamu-B*001:02'), ('Mamu-B:06102', 'Mamu-B*061:02'), ('Patr-A1001', 'Patr-A*10:01'), ('SLA-6:0102', None), ('Mamu-A1:01002', 'Mamu-A1*010:02'), ('HLA-A03:260', 'HLA-C*03:284N'), ('Patr-A0601', 'Patr-A*06:01'), ('Patr-B3701', 'Patr-B*37:01'), ('Patr-B2901', 'Patr-B*29:01'), ('Mamu-B:04616', 'Mamu-B*046:16'), ('Mamu-A2:05150', 'Mamu-A2*05:15'), ('Patr-A1601', 'Patr-A*16:01'), ('Mamu-A1:00703', 'Mamu-A1*007:03'), ('Mamu-B:02102', 'Mamu-B*021:02'), ('Patr-C1101', 'Patr-C*11:01'), ('Mamu-B:06301', 'Mamu-B*063:01'), ('Mamu-A1:02301', 'Mamu-A1*023:01'), ('SLA-2:0601', 'SLA-2*06:01'), ('Mamu-A1:04902', 'Mamu-A1*049:02'), ('Mamu-A1:05402', 'Mamu-A1*054:02'), ('Mamu-B:05802', 'Mamu-B*058:02'), ('Mamu-B:04704', 'Mamu-B*047:04'), ('Patr-C0904', 'Patr-C*09:04'), ('Mamu-A2:0539', 'Mamu-A2*05:39'), ('Patr-B2201', 'Patr-B*22:01'), ('Patr-C0301', 'Patr-C*03:01'), ('Mamu-A1:00403', 'Mamu-A1*004:03'), ('Mamu-A1:00702', 'Mamu-A1*007:02'), ('Patr-B2801', 'Patr-B*28:01'), ('SLA-1:0401', 'SLA-1*04:01'), ('Mamu-B:02901', 'Mamu-B*029:01'), ('Mamu-B:00601', 'Mamu-B*006:01'), ('Mamu-B:05501', 'Mamu-B*055:01'), ('Mamu-B:06902', 'Mamu-B*069:02'), ('Mamu-B:04610', 'Mamu-B*046:10'), ('Mamu-B:00401', 'Mamu-B*004:01'), ('Mamu-B:04612', 'Mamu-B*046:12'), ('Mamu-B:05702', 'Mamu-B*057:02'), ('Mamu-B:08701', 'Mamu-B*087:01'), ('Mamu-B:06702', 'Mamu-B*067:02'), ('Mamu-A1:02302', 'Mamu-A1*023:02'), ('Mamu-B:05701', 'Mamu-B*057:01'), ('Mamu-B:08603', 'Mamu-B*086:03'), ('Mamu-B:06601', 'Mamu-B*066:01'), ('Patr-B0203', 'Patr-B*02:03'), ('Patr-C0304', 'Patr-C*03:04'), ('Mamu-A2:05160', 'Mamu-A2*05:16'), ('Mamu-B:02602', 'Mamu-B*057:06'), ('Mamu-B:03001', 'Mamu-B*030:01'), ('Patr-B1001', 'Patr-B*10:01'), ('Patr-B1702', 'Patr-B*17:02'), ('Mamu-A4:14030', 'Mamu-A4*14:03'), ('Mamu-B:05103', 'Mamu-B*051:03'), ('Mamu-B:05101', 'Mamu-B*051:01'), ('Mamu-B:03801', 'Mamu-B*038:01'), ('Mamu-A2:0538', 'Mamu-A2*05:38'), ('Mamu-A2:0520', 'Mamu-A2*05:20'), ('Mamu-B:04002', 'Mamu-B*015:05'), ('Mamu-B:01701', 'Mamu-B*017:01'), ('Mamu-B:05002', 'Mamu-B*050:02'), ('Patr-C1302', 'Patr-C*13:02'), ('Mamu-B:04402', 'Mamu-B*044:02'), ('Mamu-A1:07402', 'Mamu-A1*074:02'), ('Mamu-A6:0105', 'Mamu-A6*01:05'), ('Mamu-B:08602', 'Mamu-B*086:02'), ('Mamu-A3:1306', 'Mamu-A3*13:06'), ('Mamu-B:07903', 'Mamu-B*079:03'), ('SLA-2:0701', 'SLA-2*07:01'), ('Mamu-B:00302', 'Mamu-B*003:02'), ('Mamu-A7:0201', 'Mamu-A7*02:01'), ('SLA-6:0104', 'SLA-6*04:01'), ('SLA-2:1001', 'SLA-2*10:01'), ('Mamu-A1:01104', 'Mamu-A1*011:04'), ('Mamu-A1:01201', 'Mamu-A1*012:01'), ('SLA-2:1201', 'SLA-2*12:01'), ('Mamu-B:10101', 'Mamu-B*101:01'), ('SLA-6:0101', 'SLA-6*01:01'), ('Patr-B2401', 'Patr-B*24:01'), ('Mamu-A2:05320', 'Mamu-A2*05:32'), ('SLA-1:0501', 'SLA-1*05:01'), ('Mamu-B:04603', 'Mamu-B*046:03'), ('Mamu-B:06401', 'Mamu-B*064:01'), ('Mamu-B:02703', 'Mamu-B*027:03'), ('Mamu-A1:02202', 'Mamu-A1*022:02'), ('Mamu-A1:00102', 'Mamu-A1*001:02'), ('Mamu-A1:02802', 'Mamu-A1*028:02'), ('Mamu-B:07501', 'Mamu-B*075:01'), ('Mamu-A1:04102', 'Mamu-A1*041:02'), ('Mamu-B:02201', 'Mamu-B*022:01'), ('Mamu-A1:11201', 'Mamu-A1*112:01'), ('Mamu-A1:10701', 'Mamu-A1*107:01'), ('Patr-A0302', 'Patr-A*03:02'), ('Mamu-B:07801', 'Mamu-B*078:01'), ('Patr-C0901', 'Patr-C*09:01'), ('Mamu-A3:1310', 'Mamu-A3*13:10'), ('Mamu-A1:00301', 'Mamu-A1*003:01'), ('Mamu-A1:04903', 'Mamu-A1*049:03'), ('HLA-C04:61', 'HLA-C*04:61N'), ('Mamu-B:07002', 'Mamu-B*070:02'), ('Mamu-A2:0526', 'Mamu-A2*05:26'), ('Mamu-B:06103', 'Mamu-B*061:03'), ('Mamu-B:01903', 'Mamu-B*019:03'), ('Mamu-A3:1311', 'Mamu-A3*13:11'), ('Mamu-B:03602', 'Mamu-B*036:02'), ('Mamu-A4:1407', 'Mamu-A4*14:07'), ('Mamu-B:09101', 'Mamu-B*091:01'), ('SLA-2:0201', 'SLA-2*02:01'), ('Mamu-B:08301', 'Mamu-B*083:01'), ('Mamu-A2:0543', 'Mamu-A2*05:43'), ('Mamu-B:09001', 'Mamu-B*090:01'), ('Patr-C0902', 'Patr-C*09:02'), ('Mamu-B:06003', 'Mamu-B*060:03'), ('Mamu-A1:04801', 'Mamu-A1*048:01'), ('Mamu-A1:00101', 'Mamu-A1*001:01'), ('Mamu-B:03201', 'Mamu-B*032:01'), ('Mamu-A1:06601', 'Mamu-A1*066:01'), ('Mamu-B:00501', 'Mamu-B*005:01'), ('Mamu-A2:0101', 'Mamu-A2*01:01'), ('Mamu-A2:0535', 'Mamu-A2*05:35'), ('Mamu-A1:01905', 'Mamu-A1*019:05'), ('Mamu-A2:0513', 'Mamu-A2*05:13'), ('Mamu-A1:05401', 'Mamu-A1*054:01'), ('Mamu-A3:1301', 'Mamu-A3*13:01'), ('Mamu-A1:01101', 'Mamu-A1*011:01'), ('Mamu-A4:1404', 'Mamu-A4*14:04'), ('Mamu-A3:1307', 'Mamu-A3*13:07'), ('Mamu-B:05401', 'Mamu-B*054:01'), ('Patr-B2501', 'Patr-B*25:01'), ('Mamu-A2:0506', 'Mamu-A2*05:06'), ('Mamu-A1:01902', 'Mamu-A1*019:02'), ('Mamu-B:03103', 'Mamu-B*031:03'), ('Mamu-A1:01904', 'Mamu-A1*019:04'), ('Patr-A2401', 'Patr-A*24:01'), ('Patr-A0402', 'Patr-A*04:02'), ('Mamu-B:08201', 'Mamu-B*082:01'), ('Mamu-B:04502', 'Mamu-B*045:02'), ('Mamu-B:06803', 'Mamu-B*068:03'), ('Mamu-B:09102', 'Mamu-B*091:02'), ('Mamu-A2:0542', 'Mamu-A2*05:42'), ('HLA-A02:437', 'HLA-A*02:437Q'), ('Mamu-B:00702', 'Mamu-B*007:02'), ('SLA-1:0801', 'SLA-1*08:01'), ('Mamu-B:07602', 'Mamu-B*076:02'), ('HLA-B41:56', 'HLA-B*41:56Q'), ('Mamu-B:04504', 'Mamu-B*045:04'), ('Mamu-A4:0205', 'Mamu-A4*02:05'), ('Mamu-A5:30010', 'Mamu-A5*30:01'), ('Mamu-B:03003', 'Mamu-B*030:03'), ('Mamu-B:01201', 'Mamu-B*012:01'), ('Mamu-A1:01901', 'Mamu-A1*019:01'), ('Mamu-B:03002', 'Mamu-B*030:02'), ('Patr-C0701', 'Patr-C*07:01'), ('Mamu-B:04801', 'Mamu-B*048:01'), ('SLA-1:0601', 'SLA-1*06:01'), ('Patr-A0501', 'Patr-A*05:01'), ('Mamu-A1:01601', 'Mamu-A1*016:01'), ('Patr-A2301', 'Patr-A*22:01'), ('Mamu-A4:1405', 'Mamu-A4*14:05'), ('Mamu-A4:0203', 'Mamu-A4*02:03'), ('Mamu-B:04405', 'Mamu-B*044:05'), ('Mamu-A1:06501', 'Mamu-A1*065:01'), ('Mamu-A1:01805', 'Mamu-A1*018:05'), ('Mamu-B:06802', 'Mamu-B*068:02'), ('SLA-2:0301', 'SLA-2*03:01'), ('Patr-B1101', 'Patr-B*11:01'), ('Patr-C0202', 'Patr-C*02:02'), ('Mamu-A1:04003', 'Mamu-A1*040:03'), ('Patr-B1301', 'Patr-B*13:01'), ('Patr-C0903', 'Patr-C*09:03'), ('SLA-1:1301', 'SLA-1*13:01'), ('Mamu-A1:00305', 'Mamu-A1*003:05'), ('Mamu-A2:0545', 'Mamu-A2*05:45'), ('Mamu-B:06805', 'Mamu-B*068:05'), ('Mamu-A1:04301', 'Mamu-A1*043:01'), ('Mamu-A3:1304', 'Mamu-A3*13:04'), ('Mamu-B:00801', 'Mamu-B*008:01'), ('Mamu-B:04611', 'Mamu-B*046:11'), ('Mamu-B:03005', 'Mamu-B*030:05'), ('Mamu-B:04101', 'Mamu-B*041:01'), ('Patr-C0303', 'Patr-B*03:02'), ('Mamu-B:04401', 'Mamu-B*044:01'), ('Mamu-B:04613', 'Mamu-B*046:13'), ('Mamu-B:07502', 'Mamu-B*075:02'), ('Patr-B2301', 'Patr-B*23:01'), ('Mamu-A1:00801', 'Mamu-A1*008:01'), ('Mamu-A2:0541', 'Mamu-A2*05:41'), ('Mamu-A1:00704', 'Mamu-A1*007:04'), ('Mamu-B:05104', 'Mamu-B*051:04'), ('Mamu-B:01301', 'Mamu-B*013:01'), ('Patr-C1501', 'Patr-C*15:01'), ('Patr-C0203', 'Patr-C*02:03'), ('Patr-A1702', 'Patr-A*17:02'), ('HLA-A02:581', 'HLA-A*02:581Q'), ('Mamu-A7:0101', 'Mamu-A7*01:01'), ('Mamu-A4:1409', 'Mamu-A4*14:09'), ('Mamu-A2:0511', 'Mamu-A2*05:11'), ('Mamu-A1:05101', 'Mamu-A1*051:01'), ('Mamu-A4:1408', 'Mamu-A4*14:08'), ('Mamu-B:07001', 'Mamu-B*070:01'), ('Patr-B2302', 'Patr-B*23:02'), ('HLA-A02:728', 'HLA-A*02:728Q'), ('Patr-C0205', 'Patr-C*02:05'), ('Mamu-B:02103', 'Mamu-B*021:03'), ('Mamu-A2:0547', None), ('SLA-2:0401', 'SLA-2*04:01'), ('SLA-3:0502', 'SLA-3*05:02'), ('HLA-A02:795', 'HLA-A*02:795Q'), ('Mamu-B:05102', 'Mamu-B*051:02'), ('Mamu-B:07101', 'Mamu-B*071:01'), ('Mamu-B:03401', 'Mamu-B*034:01'), ('Mamu-B:07201', 'Mamu-B*072:01'), ('Mamu-B:09901', 'Mamu-B*099:01'), ('Patr-C0501', 'Patr-C*05:01'), ('Patr-A0803', 'Patr-A*08:03'), ('SLA-3:0401', 'SLA-3*04:01'), ('Mamu-A2:2401', 'Mamu-A2*24:01'), ('Mamu-B:08401', 'Mamu-B*084:01'), ('Mamu-A1:05702', 'Mamu-A1*057:02'), ('Patr-C1301', 'Patr-C*13:01'), ('Patr-C0302', 'Patr-C*03:02'), ('Mamu-B:06904', 'Mamu-B*069:04'), ('Mamu-A4:1402', 'Mamu-A4*14:02'), ('Mamu-A1:07301', 'Mamu-A1*073:01'), ('Mamu-A2:0528', 'Mamu-A2*05:28'), ('Mamu-A4:01020', 'Mamu-A4*01:02'), ('Patr-B0501', 'Patr-B*05:01'), ('Mamu-B:06002', 'Mamu-B*060:02'), ('Mamu-B:02501', 'Mamu-B*025:01'), ('Mamu-B:00202', 'Mamu-B*002:02'), ('Patr-B0302', 'Patr-B*03:02'), ('Patr-B1701', 'Patr-B*17:01'), ('Mamu-A1:04002', 'Mamu-A1*040:02'), ('Mamu-A1:02803', 'Mamu-A1*028:03'), ('HLA-C03:284', 'HLA-C*03:284N'), ('Mamu-B:06302', 'Mamu-B*063:02'), ('SLA-1:0101', 'SLA-1*01:01'), ('Mamu-A1:00304', 'Mamu-A1*003:04'), ('Mamu-B:02001', 'Mamu-B*068:06'), ('Mamu-B:04615', 'Mamu-B*046:15'), ('Patr-B0901', 'Patr-B*09:01'), ('Patr-B0101', 'Patr-B*01:01'), ('Patr-B0402', 'Patr-B*04:02'), ('Mamu-A1:03201', 'Mamu-A1*032:01'), ('Patr-C0206', 'Patr-C*02:06'), ('Mamu-B:04614', 'Mamu-B*046:14'), ('Mamu-A1:01001', 'Mamu-A1*010:01'), ('Patr-B0801', 'Patr-B*08:01'), ('Mamu-A1:04904', 'Mamu-A1*049:04'), ('HLA-C07:226', 'HLA-C*07:226Q'), ('Mamu-A2:0540', 'Mamu-A2*05:40'), ('Mamu-A1:10601', 'Mamu-A1*106:01'), ('Mamu-A1:03301', 'Mamu-A1*033:01'), ('Patr-B0502', 'Patr-B*05:02'), ('Mamu-A2:0544', 'Mamu-A2*05:44'), ('Mamu-B:02702', 'Mamu-B*027:02'), ('Mamu-A1:00307', 'Mamu-A1*003:07'), ('Patr-C0905', 'Patr-C*09:05'), ('Mamu-B:04703', 'Mamu-B*047:03'), ('Patr-C0502', 'Patr-C*05:02'), ('Mamu-A5:3006', 'Mamu-A5*30:06'), ('Mamu-B:07402', 'Mamu-B*074:02'), ('Mamu-A1:09201', 'Mamu-A1*092:01'), ('Mamu-A2:0501', 'Mamu-A2*05:01'), ('SLA-3:0304', 'SLA-3*03:04'), ('Patr-A1801', 'Patr-A*18:01'), ('Mamu-A3:1302', 'Mamu-A3*13:02'), ('HLA-A24:378', 'HLA-A*24:378Q'), ('Mamu-B:02601', 'Mamu-B*057:05'), ('Mamu-A1:04901', 'Mamu-A1*049:01'), ('SLA-3:0501', 'SLA-3*05:01'), ('SLA-3:0601', 'SLA-3*06:01'), ('SLA-2:0501', 'SLA-2*05:01'), ('Mamu-A3:1309', 'Mamu-A3*13:09'), ('Mamu-A4:0103', 'Mamu-A4*01:03'), ('SLA-2:0502', 'SLA-2*05:02'), ('Mamu-A6:0104', 'Mamu-A6*01:04'), ('BoLA-106101', 'BoLA-1*061:01'), ('SLA-10401', 'SLA-1*04:01'), ('Patr-A18:01', 'Patr-A*18:01'), ('SLA-20301', 'SLA-2*03:01'), ('BoLA-204401', 'BoLA-2*044:01'), ('Mamu-A0510', None), ('SLA-20402', 'SLA-2*04:02'), ('HLA-B3573', None), ('Patr-A14:01', 'Patr-A*14:01'), ('Patr-A06:01', 'Patr-A*06:01'), ('BoLA-N:02402', 'BoLA-4*024:02'), ('BoLA-102001', 'BoLA-1*020:01'), ('Patr-B06:01', 'Patr-B*06:01'), ('BoLA-200602', 'BoLA-2*006:02'), ('SLA-30302', 'SLA-3*03:02'), ('Patr-B03:02', 'Patr-B*03:02'), ('BoLA-N:02001', 'BoLA-1*020:01'), ('HLA-A9210', None), ('Patr-A04:04', 'Patr-A*04:04'), ('Mamu-A1305', None), ('SLA-304:01', 'SLA-3*04:01'), ('SLA-204:01', 'SLA-2*04:01'), ('HLA-A0283', None), ('BoLA-306801', 'BoLA-3*068:01'), ('BoLA-103102', 'BoLA-1*031:02'), ('BoLA-102901', 'BoLA-1*029:01'), ('HLA-B9521', None), ('Mamu-B37', None), ('Patr-C09:01', 'Patr-C*09:01'), ('Mamu-A1011:01', 'Mamu-A1*011:01'), ('Mamu-A26', 'Mamu-A1*026:01'), ('HLA-B0744', 'HLA-B*07:44N'), ('BoLA-N:01001', 'BoLA-3*010:01'), ('HLA-A3021', None), ('HLA-B0806', None), ('Mamu-A11', 'Mamu-A1*105:01'), ('Mamu-B:01704', 'Mamu-B*017:04'), ('BoLA-N:04801', 'BoLA-2*048:01'), ('Mamu-B49', None), ('Patr-A11:01', 'Patr-A*11:01'), ('BoLA-204402', 'BoLA-2*044:02'), ('HLA-A9224', None), ('BoLA-N:00402', 'BoLA-3*004:02'), ('BoLA-N:00101', 'BoLA-3*001:01'), ('Patr-A01:01', 'Patr-A*01:01'), ('BoLA-104901', 'BoLA-1*049:01'), ('Patr-C02:05', 'Patr-C*02:05'), ('Mamu-B07', 'Mamu-B*070:01'), ('Patr-A15:01', 'Patr-A*15:01'), ('Mamu-B1001', 'Mamu-B*100:01'), ('Patr-B07:01', 'Patr-B*07:01'), ('SLA-20102', 'SLA-2*01:02'), ('HLA-B9504', None), ('BoLA-N:03001', 'BoLA-2*030:01'), ('HLA-B5147', None), ('BoLA-500301', 'BoLA-5*003:01'), ('Patr-B04:01', 'Patr-B*04:01'), ('SLA-30301', 'SLA-3*03:01'), ('Patr-A24:01', 'Patr-A*24:01'), ('Patr-B10:01', 'Patr-B*10:01'), ('Mamu-B43', None), ('BoLA-305801', 'BoLA-3*058:01'), ('Patr-A02:01', 'Patr-A*02:01'), ('Mamu-B57', None), ('BoLA-N:05001', 'BoLA-3*050:01'), ('BoLA-305101', 'BoLA-3*051:01'), ('SLA-20701', 'SLA-2*07:01'), ('Patr-B26:01', 'Patr-B*26:01'), ('HLA-A9222', None), ('HLA-B5405', None), ('Mamu-B21', None), ('BoLA-205501', 'BoLA-2*055:01'), ('Patr-C16:01', 'Patr-C*16:01'), ('H2-Kq', 'H2-Kq'), ('Patr-C02:06', 'Patr-C*02:06'), ('Patr-A03:01', 'Patr-A*03:01'), ('BoLA-N:03901', 'BoLA-5*039:01'), ('Mamu-B3901', 'Mamu-B*039:01'), ('Mamu-B8301', 'Mamu-B*083:01'), ('Mamu-B46', None), ('HLA-B9505', None), ('BoLA-107401', 'BoLA-1*074:01'), ('HLA-A9220', None), ('BoLA-N:00602', 'BoLA-2*006:02'), ('HLA-A2307', None), ('HLA-B9527', None), ('Mamu-A0507', None), ('Mamu-B22', None), ('Patr-B11:02', 'Patr-B*11:02'), ('BoLA-N:02101', 'BoLA-1*021:01'), ('HLA-B9501', None), ('BoLA-103101', 'BoLA-1*031:01'), ('BoLA-N:01601', 'BoLA-2*016:01'), ('HLA-A9214', None), ('SLA-20201', 'SLA-2*02:01'), ('Mamu-B08', 'Mamu-B*080:01'), ('H2-Kk', 'H2-Kk'), ('HLA-B9518', None), ('HLA-A9215', None), ('Mamu-A06', None), ('BoLA-N:03501', 'BoLA-3*035:01'), ('Patr-C07:01', 'Patr-C*07:01'), ('Patr-B05:01', 'Patr-B*05:01'), ('BoLA-102301', 'BoLA-1*023:01'), ('BoLA-N:03401', 'BoLA-6*034:01'), ('BoLA-201201', 'BoLA-2*012:01'), ('BoLA-300403', 'BoLA-3*004:03'), ('BoLA-101901', 'BoLA-1*019:01'), ('HLA-A9202', None), ('BoLA-306602', 'BoLA-3*066:02'), ('Patr-B23:02', 'Patr-B*23:02'), ('Patr-A23:01', 'Patr-A*22:01'), ('Patr-A17:01', 'Patr-A*17:01'), ('Patr-C08:01', 'Patr-C*08:01'), ('BoLA-N:03801', 'BoLA-3*038:01'), ('BoLA-204801', 'BoLA-2*048:01'), ('BoLA-507201', 'BoLA-5*072:01'), ('BoLA-N:02401', 'BoLA-4*024:01'), ('BoLA-N:02301', 'BoLA-1*023:01'), ('HLA-A9201', None), ('Patr-C12:01', 'Patr-C*12:01'), ('HLA-C0409', None), ('Patr-A04:01', 'Patr-A*04:01'), ('Patr-B02:01', 'Patr-B*02:01'), ('HLA-B3938', None), ('HLA-B9507', None), ('BoLA-204501', 'BoLA-2*045:01'), ('Mamu-B44', None), ('HLA-B9528', None), ('HLA-B1823', None), ('Mamu-B8701', 'Mamu-B*087:01'), ('BoLA-104201', 'BoLA-1*042:01'), ('Chi-B1201', None), ('BoLA-N:00401', 'BoLA-3*004:01'), ('BoLA-603401', 'BoLA-6*034:01'), ('HLA-A9218', None), ('BoLA-N:03101', 'BoLA-1*031:01'), ('Mamu-A04', None), ('Patr-A09:01', 'Patr-A*09:01'), ('BoLA-N:04901', 'BoLA-1*049:01'), ('Mamu-B3002', None), ('SLA-107:01', 'SLA-1*07:01'), ('Patr-B29:01', 'Patr-B*29:01'), ('Patr-C02:02', 'Patr-C*02:02'), ('BoLA-N:01802', 'BoLA-2*018:02'), ('BoLA-N:03601', 'BoLA-3*036:01'), ('H2-Qa2', None), ('HLA-A2409', None), ('HLA-B9502', None), ('BoLA-301702', 'BoLA-3*017:02'), ('Mamu-A28', None), ('BoLA-305001', 'BoLA-3*050:01'), ('HLA-B9514', None), ('BoLA-203202', 'BoLA-2*032:02'), ('Mamu-A1001:01', 'Mamu-A1*001:01'), ('BoLA-300402', 'BoLA-3*004:02'), ('Patr-B37:01', 'Patr-B*37:01'), ('Mamu-B70', None), ('BoLA-N:02501', 'BoLA-2*025:01'), ('Mamu-B017:04', 'Mamu-B*017:04'), ('BoLA-207101', 'BoLA-2*071:01'), ('HLA-B9508', None), ('BoLA-601302', 'BoLA-6*013:02'), ('SLA-107:02', 'SLA-1*07:02'), ('BoLA-300401', 'BoLA-3*004:01'), ('Patr-A08:01', 'Patr-A*08:01'), ('BoLA-300103', 'BoLA-3*001:03'), ('BoLA-203001', 'BoLA-2*030:01'), ('BoLA-N:04701', 'BoLA-2*047:01'), ('Mamu-B003:01', 'Mamu-B*003:01'), ('HLA-B9522', None), ('SLA-205:02', 'SLA-2*05:02'), ('BoLA-305901', 'BoLA-3*059:01'), ('HLA-B3565', None), ('Mamu-A0506', None), ('BoLA-206901', 'BoLA-2*069:01'), ('Mamu-A01', 'Mamu-A1*001:01'), ('BoLA-201601', 'BoLA-2*016:01'), ('Patr-C09:04', 'Patr-C*09:04'), ('Mamu-B71', None), ('HLA-A9212', None), ('Mamu-A70103', 'Mamu-A7*01:03'), ('BoLA-N:01501', 'BoLA-6*015:01'), ('Patr-C03:01', 'Patr-C*03:01'), ('H2-Kb', 'H2-Kb'), ('Patr-C06:01', 'Patr-C*06:01'), ('Mamu-B19', None), ('BoLA-303501', 'BoLA-3*035:01'), ('Mamu-B41', None), ('H2-Qa1', 'H2-Qa-1a'), ('Patr-B22:01', 'Patr-B*22:01'), ('BoLA-204701', 'BoLA-2*047:01'), ('Patr-A15:02', 'Patr-A*15:02'), ('Patr-A03:02', 'Patr-A*03:02'), ('BoLA-N:01402', 'BoLA-6*014:02'), ('HLA-A9223', None), ('SLA-20502', 'SLA-2*05:02'), ('Mamu-A0509', None), ('Patr-B24:02', 'Patr-B*24:02'), ('Mamu-B47', None), ('Chi-B0401', None), ('BoLA-202201', 'BoLA-2*022:01'), ('BoLA-305002', 'BoLA-3*050:02'), ('Mamu-B63', None), ('SLA-21201', 'SLA-2*12:01'), ('BoLA-N:01801', 'BoLA-2*018:01'), ('BoLA-601402', 'BoLA-6*014:02'), ('HLA-A9211', None), ('Mamu-A1002:01', 'Mamu-A1*002:01'), ('Patr-B03:01', 'Patr-B*03:01'), ('BoLA-205401', 'BoLA-2*054:01'), ('BoLA-204601', 'BoLA-2*046:01'), ('Mamu-A1602', None), ('Mamu-B27', None), ('HLA-C0301', None), ('Patr-A17:03', 'Patr-A*17:03'), ('BoLA-N:02702', 'BoLA-3*027:02'), ('Patr-C04:01', 'Patr-C*04:01'), ('BoLA-202602', 'BoLA-2*026:02'), ('BoLA-N:00301', 'BoLA-5*003:01'), ('HLA-B0808', None), ('Patr-C09:02', 'Patr-C*09:02'), ('Patr-C11:01', 'Patr-C*11:01'), ('BoLA-N:04301', 'BoLA-2*043:01'), ('BoLA-N:02601', 'BoLA-2*026:01'), ('BoLA-402402', 'BoLA-4*024:02'), ('SLA-21001', 'SLA-2*10:01'), ('HLA-B9519', None), ('BoLA-402401', 'BoLA-4*024:01'), ('HLA-B9529', None), ('BoLA-303701', 'BoLA-3*037:01'), ('BoLA-N:03701', 'BoLA-3*037:01'), ('HLA-B9516', None), ('Patr-B01:01', 'Patr-B*01:01'), ('SLA-30304', 'SLA-3*03:04'), ('BoLA-N:02801', 'BoLA-1*028:01'), ('Patr-B02:03', 'Patr-B*02:03'), ('H2-Dq', 'H2-Dq'), ('Patr-A13:01', 'Patr-A*13:01'), ('SLA-104:01', 'SLA-1*04:01'), ('BoLA-200601', 'BoLA-2*006:01'), ('BoLA-305301', 'BoLA-3*053:01'), ('BoLA-306501', 'BoLA-3*065:01'), ('HLA-B9524', None), ('Patr-B17:01', 'Patr-B*17:01'), ('Patr-A04:02', 'Patr-A*04:02'), ('BoLA-N:02602', 'BoLA-2*026:02'), ('Mamu-B66', 'Mamu-B*066:01'), ('SLA-30601', 'SLA-3*06:01'), ('H2-Db', 'H2-Db'), ('Mamu-B67', None), ('H2-Lq', 'H2-Lq'), ('Patr-B13:01', 'Patr-B*13:01'), ('BoLA-200802', 'BoLA-2*008:02'), ('BoLA-601502', 'BoLA-6*015:02'), ('BoLA-301001', 'BoLA-3*010:01'), ('HLA-B9526', None), ('BoLA-N:01901', 'BoLA-1*019:01'), ('BoLA-N:05201', 'BoLA-3*052:01'), ('Mamu-A20101', 'Mamu-A2*01:01'), ('Mamu-B69', None), ('HLA-A9207', None), ('BoLA-N:00103', 'BoLA-3*001:03'), ('Mamu-B05', 'Mamu-B*050:02'), ('Patr-A08:02', 'Patr-A*08:02'), ('BoLA-N:01602', 'BoLA-2*016:02'), ('BoLA-300102', 'BoLA-3*001:02'), ('BoLA-200801', 'BoLA-2*008:01'), ('Mamu-A07', 'Mamu-A1*007:01'), ('BoLA-206001', 'BoLA-2*060:01'), ('Patr-A12:01', 'Patr-A*12:01'), ('Patr-C15:01', 'Patr-C*15:01'), ('Patr-A17:02', 'Patr-A*17:02'), ('Patr-B18:01', 'Patr-B*18:01'), ('SLA-3:0402', 'SLA-3*04:02'), ('Mamu-B40', None), ('SLA-30402', 'SLA-3*04:02'), ('Mamu-B17', 'Mamu-B*017:01'), ('Patr-B24:01', 'Patr-B*24:01'), ('BoLA-301703', 'BoLA-3*017:03'), ('HLA-B0749', None), ('Patr-C05:01', 'Patr-C*05:01'), ('Patr-C03:04', 'Patr-C*03:04'), ('BoLA-302701', 'BoLA-3*027:01'), ('BoLA-301701', 'BoLA-3*017:01'), ('BoLA-200501', 'BoLA-2*005:01'), ('Mamu-B28', None), ('BoLA-N:00901', 'BoLA-1*009:01'), ('Patr-C09:05', 'Patr-C*09:05'), ('HLA-B9532', None), ('SLA-10501', 'SLA-1*05:01'), ('BoLA-202601', 'BoLA-2*026:01'), ('HLA-A9221', None), ('BoLA-N:04101', 'BoLA-6*041:01'), ('SLA-20501', 'SLA-2*05:01'), ('SLA-11101', 'SLA-1*11:01'), ('BoLA-503901', 'BoLA-5*039:01'), ('BoLA-205701', 'BoLA-2*057:01'), ('Mamu-B64', None), ('Patr-C02:04', 'Patr-C*02:04'), ('HLA-A9204', None), ('SLA-11201', 'SLA-1*12:01'), ('BoLA-601301', 'BoLA-6*013:01'), ('Mamu-B52', 'Mamu-B*052:01'), ('SLA-10701', 'SLA-1*07:01'), ('Patr-B25:01', 'Patr-B*25:01'), ('Mamu-A0511', None), ('Patr-B20:01', 'Patr-B*20:01'), ('BoLA-N:00801', 'BoLA-2*008:01'), ('Patr-A16:01', 'Patr-A*16:01'), ('Patr-A07:01', 'Patr-A*07:01'), ('Mamu-B02', 'Mamu-B*021:01'), ('SLA-10202', 'SLA-1*02:02'), ('HLA-A9226', None), ('BoLA-604001', 'BoLA-6*040:01'), ('BoLA-N:04001', 'BoLA-6*040:01'), ('SLA-10601', 'SLA-1*06:01'), ('SLA-21002', 'SLA-2*10:02'), ('Patr-B36:01', 'Patr-B*36:01'), ('BoLA-102801', 'BoLA-1*028:01'), ('BoLA-302702', 'BoLA-3*027:02'), ('Mamu-A23', None), ('Mamu-B39', 'Mamu-B*039:01'), ('HLA-B9530', None), ('Patr-C13:01', 'Patr-C*13:01'), ('HLA-A9206', None), ('Mamu-A21', None), ('BoLA-N:01302', 'BoLA-6*013:02'), ('BoLA-201602', 'BoLA-2*016:02'), ('Patr-A09:02', 'Patr-A*09:02'), ('HLA-C1407', None), ('Mamu-B55', None), ('Patr-B21:01', 'Patr-B*21:01'), ('SLA-10101', 'SLA-1*01:01'), ('SLA-30602', 'SLA-3*06:02'), ('BoLA-N:00201', 'BoLA-3*002:01'), ('BoLA-N:02701', 'BoLA-3*027:01'), ('BoLA-307301', 'BoLA-3*073:01'), ('HLA-A0303', None), ('BoLA-102101', 'BoLA-1*021:01'), ('Eqca-1600101', 'Eqca-16*001:01'), ('Mamu-A20102', 'Mamu-A2*01:02'), ('HLA-B9515', None), ('Mamu-B45', None), ('SLA-10702', 'SLA-1*07:02'), ('Mamu-B6601', 'Mamu-B*066:01'), ('H2-Kd', 'H2-Kd'), ('HLA-A9208', None), ('Chi-B1501', None), ('Mamu-A0505', None), ('Mamu-B6002', None), ('BoLA-N:02201', 'BoLA-2*022:01'), ('Mamu-B24', None), ('Patr-A05:01', 'Patr-A*05:01'), ('HLA-B9503', None), ('BoLA-300201', 'BoLA-3*002:01'), ('BoLA-202603', 'BoLA-2*026:03'), ('Mamu-B5002', None), ('Mamu-B61', None), ('HLA-B9506', None), ('BoLA-N:05401', 'BoLA-2*054:01'), ('Patr-C02:03', 'Patr-C*02:03'), ('Mamu-B48', None), ('Mamu-A0703', None), ('Mamu-A2601', 'Mamu-A1*026:01'), ('HLA-B9523', None), ('Patr-B16:01', 'Patr-B*16:01'), ('Patr-B23:01', 'Patr-B*23:01'), ('BoLA-N:00102', 'BoLA-3*001:02'), ('Patr-B17:02', 'Patr-B*17:02'), ('SLA-20302', 'SLA-2*03:02'), ('Patr-B16:02', 'Patr-B*16:02'), ('BoLA-N:01502', 'BoLA-6*015:02'), ('BoLA-N:04201', 'BoLA-1*042:01'), ('Patr-B23:03', 'Patr-B*22:03'), ('BoLA-N:00902', 'BoLA-1*009:02'), ('SLA-20601', 'SLA-2*06:01'), ('Patr-B09:01', 'Patr-B*09:01'), ('H2-Ld', 'H2-Ld'), ('H2-Dd', 'H2-Dd'), ('BoLA-506401', 'BoLA-5*064:01'), ('BoLA-N:01101', 'BoLA-3*011:01'), ('BoLA-N:04501', 'BoLA-2*045:01'), ('Patr-C03:02', 'Patr-C*03:02'), ('HLA-A9217', None), ('BoLA-207001', 'BoLA-2*070:01'), ('Patr-A06:02', 'Patr-A*06:02'), ('SLA-30502', 'SLA-3*05:02'), ('Patr-B19:01', 'Patr-B*19:01'), ('HLA-A2465', None), ('HLA-A30:14L', 'HLA-A*30:14L'), ('Mamu-B12', None), ('BoLA-N:00601', 'BoLA-2*006:01'), ('SLA-20401', 'SLA-2*04:01'), ('HLA-B9512', None), ('BoLA-305201', 'BoLA-3*052:01'), ('SLA-30101', 'SLA-3*01:01'), ('Mamu-B01', 'Mamu-B*010:01'), ('Mamu-A2201', 'Mamu-A1*022:01'), ('BoLA-N:02901', 'BoLA-1*029:01'), ('BoLA-204301', 'BoLA-2*043:01'), ('BoLA-N:01301', 'BoLA-6*013:01'), ('Mamu-B38', None), ('Mamu-A1306', None), ('BoLA-100901', 'BoLA-1*009:01'), ('BoLA-100902', 'BoLA-1*009:02'), ('BoLA-300101', 'BoLA-3*001:01'), ('BoLA-202501', 'BoLA-2*025:01'), ('HLA-A9216', None), ('BoLA-N:01201', 'BoLA-2*012:01'), ('Patr-C02:01', 'Patr-C*02:01'), ('BoLA-N:00501', 'BoLA-2*005:01'), ('SLA-30501', 'SLA-3*05:01'), ('Mamu-A02', 'Mamu-A1*002:01'), ('SLA-30503', 'SLA-3*05:03'), ('Mamu-A25', None), ('HLA-C0502', None), ('Patr-B14:01', 'Patr-B*14:01'), ('Patr-A08:03', 'Patr-A*08:03'), ('Mamu-B20', None), ('Patr-B08:02', 'Patr-B*08:02'), ('Patr-B17:03', 'Patr-B*17:03'), ('BoLA-406301', 'BoLA-4*063:01'), ('BoLA-601501', 'BoLA-6*015:01'), ('Mamu-A19', None), ('Patr-C09:03', 'Patr-C*09:03'), ('Mamu-B03', 'Mamu-B*030:01'), ('Mamu-B36', None), ('BoLA-601401', 'BoLA-6*014:01'), ('SLA-20101', 'SLA-2*01:01'), ('SLA-30401', 'SLA-3*04:01'), ('BoLA-N:01701', 'BoLA-3*017:01'), ('Patr-C03:03', 'Patr-B*03:02'), ('Patr-B22:02', 'Patr-B*22:02'), ('HLA-A9205', None), ('BoLA-201802', 'BoLA-2*018:02'), ('BoLA-303801', 'BoLA-3*038:01'), ('Patr-B05:02', 'Patr-B*05:02'), ('Patr-C13:02', 'Patr-C*13:02'), ('SLA-10201', 'SLA-1*02:01'), ('Mamu-B5802', None), ('Mamu-B004:01', 'Mamu-B*004:01'), ('BoLA-N:04601', 'BoLA-2*046:01'), ('HLA-B9525', None), ('HLA-A9203', None), ('Mamu-B53', None), ('Patr-B28:01', 'Patr-B*28:01'), ('BoLA-106701', 'BoLA-1*067:01'), ('Mamu-B065:02', 'Mamu-B*065:02'), ('BoLA-604101', 'BoLA-6*041:01'), ('BoLA-205601', 'BoLA-2*056:01'), ('BoLA-306601', 'BoLA-3*066:01'), ('Patr-B12:02', 'Patr-B*12:02'), ('HLA-B9513', None), ('Mamu-B03901', 'Mamu-B*039:01'), ('Patr-C05:02', 'Patr-C*05:02'), ('SLA-20202', 'SLA-2*02:02'), ('HLA-B9520', None), ('BoLA-206201', 'BoLA-2*062:01'), ('HLA-B9510', None), ('HLA-A9209', None), ('Mamu-A03', None), ('HLA-B9517', None), ('Mamu-B65', None), ('BoLA-N:05101', 'BoLA-3*051:01'), ('BoLA-N:05601', 'BoLA-2*056:01'), ('HLA-A9219', None), ('Patr-B08:01', 'Patr-B*08:01'), ('Eqca-100101', 'Eqca-1*001:01'), ('Mamu-B04', 'Mamu-B*041:01'), ('BoLA-T2C', None), ('Patr-B04:02', 'Patr-B*04:02'), ('Mamu-B001:01', 'Mamu-B*001:01'), ('BoLA-N:05301', 'BoLA-3*053:01'), ('Patr-B07:02', 'Patr-B*07:02'), ('SLA-30303', 'SLA-3*03:03'), ('BoLA-N:04401', 'BoLA-2*044:01'), ('BoLA-201801', 'BoLA-2*018:01'), ('Patr-A10:01', 'Patr-A*10:01'), ('BoLA-303601', 'BoLA-3*036:01'), ('BoLA-N:01401', 'BoLA-6*014:01'), ('Mamu-B008:01', 'Mamu-B*008:01'), ('BoLA-301101', 'BoLA-3*011:01'), ('Patr-B11:01', 'Patr-B*11:01'), ('BoLA-N:00802', 'BoLA-2*008:02'), ('SLA-10801', 'SLA-1*08:01'), ('Patr-B35:01', 'Patr-B*35:01'), ('Patr-C10:01', 'Patr-C*10:01'), ('BoLA-N:01702', 'BoLA-3*017:02'), ('SLA-11301', 'SLA-1*13:01'), ('BoLA-N:05501', 'BoLA-2*055:01'), ('Patr-B01:02', 'Patr-B*01:02'), ('Patr-B27:01', 'Patr-B*27:01'), ('HLA-B9509', None), ('Patr-B30:01', 'Patr-B*30:01'), ('Mamu-A0602', None), ('Mamu-A24', None), ('Mamu-AG*03011', 'Mamu-AG*03:01'), ('Mamu-AG*01', 'Mamu-AG*01:01'), ('Mamu-AG*02011', 'Mamu-AG*02:01'), ('Mamu-AG*02012', 'Mamu-AG*02:01'), ('Mamu-AG*0302', 'Mamu-AG*03:02'), ('Mamu-AG*0202', 'Mamu-AG*02:02')
    #     ]
    print('=======================')
    print(suspicious_alleles)
    
    a = ['SLA-3-CDY', 'SLA-3-YDL', 'SLA-1-HB04', 'SLA-1-HB01', 'SLA-3-YDY02', 'SLA-1-TPK', 'SLA-1-YDL01', 'Mamu-A2:0547', 'SLA-3-YDY01', 'SLA-2:HB01', 'SLA-1-YTH', 'SLA-2:YC.AA', 'SLA-1-CHANGDA', 'SLA-1-HB02', 'SLA-2-YDL02', 'BoLA-T2c', 'SLA-1-LWH', 'SLA-3-LWH', 'SLA-2:YDY.AA', 'SLA-2:LWH.AA', 'SLA-2:TPK.AA', 'BoLA-T5', 'SLA-2:YTH.AA', 'SLA-3-YTH', 'SLA-1-HB03', 'BoLA-T7', 'SLA-6:0102', 'BoLA-gb1.7', 'SLA-3-YC', 'SLA-1-YC', 'SLA-2:CDY.AA', 'BoLA-amani.1', 'HLA-C03:01', 'SLA-3-HB01', 'SLA-2:YDL.AA', 'SLA-3-TPK', 'Mamu-B55', 'Mamu-A0506', 'HLA-A9216', 'HLA-B9522', 'Mamu-B48', 'Mamu-A28', 'HLA-C0301', 'H2-Qa2', 'HLA-B9510', 'Mamu-A0507', 'HLA-B9508', 'HLA-A9221', 'HLA-B9515', 'Mamu-B41', 'HLA-A2465', 'HLA-A9218', 'HLA-A9208', 'HLA-B9530', 'HLA-B3573', 'HLA-A2307', 'Mamu-B64', 'Mamu-B37', 'Mamu-B6002', 'H-2-Qa2', 'Mamu-B5002', 'HLA-A9205', 'HLA-B3565', 'HLA-B9528', 'Mamu-A04', 'HLA-A9223', 'Mamu-B63', 'Mamu-B49', 'HLA-A9224', 'HLA-A9201', 'Mamu-B3002', 'HLA-B9506', 'HLA-B9514', 'HLA-A9219', 'Mamu-A21', 'Mamu-B53', 'HLA-B5405', 'HLA-A9222', 'HLA-B0806', 'HLA-B0808', 'HLA-A9214', 'Mamu-B43', 'HLA-B9507', 'HLA-B9519', 'HLA-A9206', 'HLA-C0409', 'HLA-B9517', 'HLA-B9504', 'Chi-B0401', 'Mamu-B67', 'HLA-A9220', 'Mamu-A03', 'HLA-B9525', 'HLA-B1823', 'HLA-B9503', 'Mamu-B61', 'HLA-B9513', 'Mamu-B20', 'HLA-A9202', 'HLA-B9523', 'HLA-A9207', 'Mamu-B71', 'Mamu-A1305', 'Chi-B1201', 'HLA-A9209', 'Mamu-B24', 'Mamu-B44', 'HLA-B9532', 'HLA-A3021', 'HLA-A0283', 'HLA-B0749', 'Mamu-B70', 'HLA-B9509', 'Mamu-B28', 'Mamu-A0510', 'HLA-A9203', 'Mamu-A0602', 'HLA-B9521', 'Mamu-B45', 'HLA-A9212', 'Mamu-B36', 'Mamu-B65', 'Mamu-B46', 'HLA-B9502', 'Mamu-B22', 'HLA-B5147', 'Mamu-B19', 'HLA-B9526', 'Mamu-A06', 'Mamu-B57', 'Mamu-B12', 'Mamu-B5802', 'Mamu-B40', 'Mamu-A25', 'HLA-B9520', 'HLA-C1407', 'Mamu-B38', 'HLA-B9501', 'Mamu-B69', 'Mamu-A0511', 'HLA-C0502', 'BoLA-T2C', 'HLA-A9226', 'HLA-B9518', 'HLA-A2409', 'Mamu-A19', 'Mamu-A23', 'HLA-A9204', 'Mamu-A0505', 'HLA-B9524', 'Mamu-A0509', 'Mamu-A24', 'Mamu-A0703', 'Mamu-B27', 'HLA-A0303', 'Chi-B1501', 'Mamu-A1602', 'HLA-B9505', 'Mamu-B21', 'HLA-B9529', 'HLA-B3938', 'HLA-A9215', 'HLA-A9210', 'HLA-B9516', 'HLA-B9512', 'HLA-A9211', 'Mamu-A1306', 'HLA-B9527', 'Mamu-B47', 'HLA-A9217', 'HLA-C*03:01', 'SLA-6*0102', 'Mamu-A2*0547']
    b = [None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None, None]
    
    print(len(a), len(b))
    

def handle_rest_of_the_alleles():
    # Allele-Autocomplete can't find these alleles.
    unknown_alleles = ['SLA-3-CDY', 'SLA-3-YDL', 'SLA-1-HB04', 'SLA-1-HB01', 'SLA-3-YDY02', 'SLA-1-TPK', 'SLA-1-YDL01', 'Mamu-A2:0547', 'SLA-3-YDY01', 'SLA-2:HB01', 'SLA-1-YTH', 'SLA-2:YC.AA', 'SLA-1-CHANGDA', 'SLA-1-HB02', 'SLA-2-YDL02', 'BoLA-T2c', 'SLA-1-LWH', 'SLA-3-LWH', 'SLA-2:YDY.AA', 'SLA-2:LWH.AA', 'SLA-2:TPK.AA', 'BoLA-T5', 'SLA-2:YTH.AA', 'SLA-3-YTH', 'SLA-1-HB03', 'BoLA-T7', 'SLA-6:0102', 'BoLA-gb1.7', 'SLA-3-YC', 'SLA-1-YC', 'SLA-2:CDY.AA', 'BoLA-amani.1', 'HLA-C03:01', 'SLA-3-HB01', 'SLA-2:YDL.AA', 'SLA-3-TPK', 'Mamu-B55', 'Mamu-A0506', 'HLA-A9216', 'HLA-B9522', 'Mamu-B48', 'Mamu-A28', 'HLA-C0301', 'H2-Qa2', 'HLA-B9510', 'Mamu-A0507', 'HLA-B9508', 'HLA-A9221', 'HLA-B9515', 'Mamu-B41', 'HLA-A2465', 'HLA-A9218', 'HLA-A9208', 'HLA-B9530', 'HLA-B3573', 'HLA-A2307', 'Mamu-B64', 'Mamu-B37', 'Mamu-B6002', 'H-2-Qa2', 'Mamu-B5002', 'HLA-A9205', 'HLA-B3565', 'HLA-B9528', 'Mamu-A04', 'HLA-A9223', 'Mamu-B63', 'Mamu-B49', 'HLA-A9224', 'HLA-A9201', 'Mamu-B3002', 'HLA-B9506', 'HLA-B9514', 'HLA-A9219', 'Mamu-A21', 'Mamu-B53', 'HLA-B5405', 'HLA-A9222', 'HLA-B0806', 'HLA-B0808', 'HLA-A9214', 'Mamu-B43', 'HLA-B9507', 'HLA-B9519', 'HLA-A9206', 'HLA-C0409', 'HLA-B9517', 'HLA-B9504', 'Chi-B0401', 'Mamu-B67', 'HLA-A9220', 'Mamu-A03', 'HLA-B9525', 'HLA-B1823', 'HLA-B9503', 'Mamu-B61', 'HLA-B9513', 'Mamu-B20', 'HLA-A9202', 'HLA-B9523', 'HLA-A9207', 'Mamu-B71', 'Mamu-A1305', 'Chi-B1201', 'HLA-A9209', 'Mamu-B24', 'Mamu-B44', 'HLA-B9532', 'HLA-A3021', 'HLA-A0283', 'HLA-B0749', 'Mamu-B70', 'HLA-B9509', 'Mamu-B28', 'Mamu-A0510', 'HLA-A9203', 'Mamu-A0602', 'HLA-B9521', 'Mamu-B45', 'HLA-A9212', 'Mamu-B36', 'Mamu-B65', 'Mamu-B46', 'HLA-B9502', 'Mamu-B22', 'HLA-B5147', 'Mamu-B19', 'HLA-B9526', 'Mamu-A06', 'Mamu-B57', 'Mamu-B12', 'Mamu-B5802', 'Mamu-B40', 'Mamu-A25', 'HLA-B9520', 'HLA-C1407', 'Mamu-B38', 'HLA-B9501', 'Mamu-B69', 'Mamu-A0511', 'HLA-C0502', 'BoLA-T2C', 'HLA-A9226', 'HLA-B9518', 'HLA-A2409', 'Mamu-A19', 'Mamu-A23', 'HLA-A9204', 'Mamu-A0505', 'HLA-B9524', 'Mamu-A0509', 'Mamu-A24', 'Mamu-A0703', 'Mamu-B27', 'HLA-A0303', 'Chi-B1501', 'Mamu-A1602', 'HLA-B9505', 'Mamu-B21', 'HLA-B9529', 'HLA-B3938', 'HLA-A9215', 'HLA-A9210', 'HLA-B9516', 'HLA-B9512', 'HLA-A9211', 'Mamu-A1306', 'HLA-B9527', 'Mamu-B47', 'HLA-A9217', 'HLA-C*03:01', 'SLA-6*0102', 'Mamu-A2*0547']
    
    # Not working for curl commands.
    unavailable_alleles = [
        ('Mamu-A1:00401', 'Mamu-A1*004:01'), ('Mamu-B:04608', 'Mamu-B*046:08'), ('Mamu-A5:3003', 'Mamu-A5*30:03'), ('Mamu-B:06001', 'Mamu-B*060:01'), ('Patr-B1601', 'Patr-B*16:01'), ('Mamu-B:04503', 'Mamu-B*045:03'), ('HLA-B07:44', 'HLA-B*07:44N'), ('Mamu-A1:11301', 'Mamu-A1*008:01'), ('Mamu-B:01902', 'Mamu-B*019:02'), ('Patr-A0101', 'Patr-A*01:01'), ('Patr-B2202', 'Patr-B*22:02'), ('Mamu-A1:02602', 'Mamu-A1*026:02'), ('Mamu-A1:05603', 'Mamu-A1*056:03'), ('HLA-C04:338', 'HLA-C*04:338Q'), ('Mamu-A1:03202', 'Mamu-A1*032:02'), ('Patr-B1901', 'Patr-B*19:01'), ('Mamu-B:05602', 'Mamu-B*056:02'), ('Patr-B2701', 'Patr-B*27:01'), ('Mamu-A1:02502', 'Mamu-A1*025:02'), ('Mamu-A2:0525', 'Mamu-A2*05:25'), ('Mamu-A2:0531', 'Mamu-A2*05:31'), ('Patr-B3601', 'Patr-B*36:01'), ('Mamu-A1:05001', 'Mamu-A1*050:01'), ('Mamu-B:04602', 'Mamu-B*046:02'), ('Mamu-A2:0534', 'Mamu-A2*05:34'), ('Mamu-A2:0510', 'Mamu-A2*05:10'), ('Mamu-A1:10502', 'Mamu-A1*105:02'), ('Mamu-B:04901', 'Mamu-B*049:01'), ('Mamu-A1:00104', 'Mamu-A1*001:04'), ('Patr-A0802', 'Patr-A*08:02'), ('Patr-A0902', 'Patr-A*09:02'), ('Mamu-A1:00303', 'Mamu-A1*003:03'), ('Mamu-A1:01807', 'Mamu-A1*018:07'), ('Patr-A0404', 'Patr-A*04:04'), ('SLA-3:0302', 'SLA-3*03:02'), ('Mamu-B:06402', 'Mamu-B*064:02'), ('SLA-6:0105', 'SLA-6*05:01'), ('SLA-2:0101', 'SLA-2*01:01'), ('HLA-C03:01', None), ('Mamu-B:08901', 'Mamu-B*089:01'), ('Mamu-A1:03203', 'Mamu-A1*032:03'), ('Patr-C1201', 'Patr-C*12:01'), ('SLA-3:0303', 'SLA-3*03:03'), ('Mamu-A1:00601', 'Mamu-A1*006:01'), ('Patr-C0201', 'Patr-C*02:01'), ('Mamu-B:04403', 'Mamu-B*044:03'), ('Mamu-A5:3002', 'Mamu-A5*30:02'), ('Mamu-B:06503', 'Mamu-B*065:03'), ('Mamu-A1:06001', 'Mamu-A1*060:01'), ('Mamu-B:09401', 'Mamu-B*094:01'), ('Mamu-A1:01804', 'Mamu-A1*018:04'), ('Mamu-B:05302', 'Mamu-B*053:02'), ('Mamu-A2:0517', 'Mamu-A2*05:17'), ('Mamu-B:08801', 'Mamu-B*088:01'), ('Mamu-A1:01103', 'Mamu-A1*011:03'), ('Patr-A0201', 'Patr-A*02:01'), ('Mamu-A1:01808', 'Mamu-A1*018:08'), ('Mamu-B:07202', 'Mamu-B*072:02'), ('Mamu-A2:0519', 'Mamu-A2*05:19'), ('Mamu-B:04604', 'Mamu-B*046:04'), ('Mamu-B:08501', 'Mamu-B*085:01'), ('Mamu-A4:0202', 'Mamu-A4*02:02'), ('Mamu-A1:00306', 'Mamu-A1*003:06'), ('Patr-B0702', 'Patr-B*07:02'), ('Mamu-A1:11001', 'Mamu-A1*110:01'), ('Patr-B2101', 'Patr-B*21:01'), ('Mamu-A1:05602', 'Mamu-A1*056:02'), ('Mamu-B:07401', 'Mamu-B*074:01'), ('Mamu-B:09601', 'Mamu-B*063:03'), ('Mamu-B:00301', 'Mamu-B*003:01'), ('Mamu-B:00101', 'Mamu-B*001:01'), ('Patr-B0601', 'Patr-B*06:01'), ('HLA-C07:432', 'HLA-C*07:432Q'), ('Mamu-B:06804', 'Mamu-B*068:04'), ('Mamu-B:04802', 'Mamu-B*048:02'), ('Mamu-A1:00310', 'Mamu-A1*003:10'), ('SLA-1:0202', 'SLA-1*02:02'), ('Patr-C0401', 'Patr-C*04:01'), ('Mamu-B:02401', 'Mamu-B*024:01'), ('Mamu-A2:0546', 'Mamu-A2*05:46'), ('Mamu-B:00703', 'Mamu-B*007:03'), ('Mamu-B:01501', 'Mamu-B*015:01'), ('Patr-B1102', 'Patr-B*11:02'), ('Mamu-B:02301', 'Mamu-B*023:01'), ('Mamu-A3:1303', 'Mamu-A3*13:03'), ('SLA-2:1002', 'SLA-2*10:02'), ('Patr-B1602', 'Patr-B*16:02'), ('Mamu-A3:1305', 'Mamu-A3*13:05'), ('Patr-A0401', 'Patr-A*04:01'), ('Patr-C0601', 'Patr-C*06:01'), ('Mamu-A7:0102', 'Mamu-A7*01:02'), ('Mamu-A1:10801', 'Mamu-A1*108:01'), ('HLA-A01:281', 'HLA-A*01:281Q'), ('Mamu-B:05201', 'Mamu-B*052:01'), ('Mamu-A2:05030', 'Mamu-A2*05:30'), ('Mamu-B:07901', 'Mamu-B*079:01'), ('Mamu-A2:0102', 'Mamu-A2*01:02'), ('Mamu-B:03802', 'Mamu-B*038:02'), ('Mamu-A6:0102', 'Mamu-A6*01:02'), ('SLA-3:0503', 'SLA-3*05:03'), ('Mamu-A1:04001', 'Mamu-A1*040:01'), ('Mamu-A1:10901', 'Mamu-A1*109:01'), ('Mamu-A1:05501', 'Mamu-A1*055:01'), ('Mamu-A1:01801', 'Mamu-A1*018:01'), ('SLA-3:0701', 'SLA-3*07:01'), ('Mamu-A2:2402', 'Mamu-A2*24:02'), ('Mamu-B:03101', 'Mamu-B*031:01'), ('Mamu-B:01703', 'Mamu-B*017:03'), ('Mamu-B:06501', 'Mamu-B*065:01'), ('SLA-3:0301', 'SLA-3*03:01'), ('Patr-A1301', 'Patr-A*13:01'), ('Patr-C0204', 'Patr-C*02:04'), ('Mamu-A1:02806', 'Mamu-A1*028:06'), ('Mamu-B:08202', 'Mamu-B*082:02'), ('SLA-2:0402', 'SLA-2*04:02'), ('Mamu-A1:05701', 'Mamu-A1*057:01'), ('Mamu-B:03701', 'Mamu-B*037:01'), ('Mamu-B:01001', 'Mamu-B*010:01'), ('Mamu-A2:05040', 'Mamu-A2*05:40'), ('Mamu-A2:0524', 'Mamu-A2*05:24'), ('Mamu-B:00602', 'Mamu-B*006:02'), ('Mamu-A4:0301', 'Mamu-A4*03:01'), ('Mamu-A5:3005', 'Mamu-A5*30:05'), ('Mamu-A1:00302', 'Mamu-A1*003:02'), ('Mamu-B:07601', 'Mamu-B*076:01'), ('Mamu-B:01101', 'Mamu-B*011:01'), ('Mamu-B:03501', 'Mamu-B*035:01'), ('SLA-1:0702', 'SLA-1*07:02'), ('Mamu-A1:10501', 'Mamu-A1*105:01'), ('Mamu-A1:02201', 'Mamu-A1*022:01'), ('Mamu-A6:0101', 'Mamu-A6*01:01'), ('Mamu-A1:00602', 'Mamu-A1*006:02'), ('Patr-B2402', 'Patr-B*24:02'), ('Mamu-A1:00308', 'Mamu-A1*003:08'), ('Mamu-B:01801', 'Mamu-B*018:01'), ('Mamu-A2:0527', 'Mamu-A2*05:27'), ('Mamu-B:02902', 'Mamu-B*029:02'), ('Mamu-A1:07401', 'Mamu-A1*074:01'), ('SLA-3:0101', 'SLA-3*01:01'), ('Mamu-B:03004', 'Mamu-B*030:04'), ('Patr-B0201', 'Patr-B*02:01'), ('Patr-B2303', 'Patr-B*22:03'), ('Patr-B0102', 'Patr-B*01:02'), ('Mamu-B:01702', 'Mamu-B*017:02'), ('Mamu-A1:04101', 'Mamu-A1*041:01'), ('Mamu-B:04701', 'Mamu-B*047:01'), ('Mamu-B:04609', 'Mamu-B*046:09'), ('Mamu-B:09201', 'Mamu-B*092:01'), ('Patr-C1001', 'Patr-C*10:01'), ('Patr-C1601', 'Patr-C*16:01'), ('Patr-B2601', 'Patr-B*26:01'), ('Mamu-A2:2403', 'Mamu-A2*24:03'), ('Mamu-A1:02804', 'Mamu-A1*028:04'), ('Mamu-B:02701', 'Mamu-B*027:01'), ('SLA-1:0201', 'SLA-1*02:01'), ('Mamu-A1:02603', 'Mamu-A1*026:03'), ('Patr-A1101', 'Patr-A*11:01'), ('Mamu-A1:01906', 'Mamu-A1*019:06'), ('Mamu-B:01901', 'Mamu-B*019:01'), ('Mamu-B:06502', 'Mamu-B*065:02'), ('Mamu-B:07902', 'Mamu-B*079:02'), ('Patr-A0901', 'Patr-A*09:01'), ('Mamu-A1:00201', 'Mamu-A1*002:01'), ('Mamu-B:02803', 'Mamu-B*028:03'), ('Mamu-A2:0512', 'Mamu-A2*05:12'), ('Patr-B0701', 'Patr-B*07:01'), ('Mamu-B:06801', 'Mamu-B*068:01'), ('HLA-C03:23', 'HLA-C*03:23N'), ('SLA-2:0102', 'SLA-2*01:02'), ('Mamu-A4:1406', 'Mamu-A4*14:06'), ('Mamu-A1:05302', 'Mamu-A1*053:02'), ('Mamu-B:08502', 'Mamu-B*085:02'), ('Mamu-B:00701', 'Mamu-B*007:01'), ('Mamu-A2:0505', 'Mamu-A2*05:05'), ('Mamu-B:09501', 'Mamu-B*095:01'), ('SLA-6:0103', 'SLA-6*03:01'), ('Patr-A1201', 'Patr-A*12:01'), ('HLA-C12:139', 'HLA-C*12:139Q'), ('Mamu-A1:00402', 'Mamu-A1*004:02'), ('Patr-B1401', 'Patr-B*14:01'), ('Mamu-B:00502', 'Mamu-B*005:02'), ('SLA-1:1201', 'SLA-1*12:01'), ('HLA-A01:159', 'HLA-A*01:159Q'), ('Patr-A0602', 'Patr-A*06:02'), ('Patr-B2001', 'Patr-B*20:01'), ('Patr-B3501', 'Patr-B*35:01'), ('Mamu-B:04404', 'Mamu-B*044:04'), ('Mamu-A2:0507', 'Mamu-A2*05:07'), ('Mamu-A1:02203', 'Mamu-A1*022:03'), ('Mamu-B:03102', 'Mamu-B*031:02'), ('Patr-B1202', 'Patr-B*12:02'), ('Mamu-B:06901', 'Mamu-B*069:01'), ('Mamu-B:01401', 'Mamu-B*014:01'), ('HLA-B18:106', 'HLA-B*18:106Q'), ('Mamu-A1:08101', 'Mamu-A1*081:01'), ('Mamu-B:04201', 'Mamu-B*042:01'), ('Mamu-A1:04201', 'Mamu-A1*042:01'), ('Mamu-B:08102', 'Mamu-B*081:02'), ('SLA-3:0602', 'SLA-3*06:02'), ('Mamu-A5:3004', 'Mamu-A5*30:04'), ('Mamu-A2:0523', 'Mamu-A2*05:23'), ('Mamu-B:05105', 'Mamu-B*051:05'), ('Mamu-A4:0101', 'Mamu-A4*01:01'), ('Mamu-A1:02805', 'Mamu-A1*028:05'), ('Mamu-B:04601', 'Mamu-B*046:01'), ('Mamu-A2:0537', 'Mamu-A2*05:37'), ('Mamu-B:00201', 'Mamu-B*002:01'), ('Mamu-B:04501', 'Mamu-B*045:01'), ('Mamu-A1:10504', 'Mamu-A1*105:04'), ('Patr-B0401', 'Patr-B*04:01'), ('Mamu-B:10001', 'Mamu-B*100:01'), ('SLA-2:1101', 'SLA-2*11:01'), ('Mamu-A1:02801', 'Mamu-A1*028:01'), ('Mamu-B:04705', 'Mamu-B*047:05'), ('Mamu-B:04301', 'Mamu-B*043:01'), ('Mamu-A1:05301', 'Mamu-A1*053:01'), ('Mamu-A1:11101', 'Mamu-A1*111:01'), ('SLA-2:0302', 'SLA-2*03:02'), ('Mamu-A2:0521', 'Mamu-A2*05:21'), ('SLA-1:1101', 'SLA-1*11:01'), ('Mamu-B:04605', 'Mamu-B*046:05'), ('Mamu-A1:06101', 'Mamu-A1*061:01'), ('Mamu-A1:00701', 'Mamu-A1*007:01'), ('Mamu-B:08001', 'Mamu-B*080:01'), ('Mamu-A1:01802', 'Mamu-A1*018:02'), ('Mamu-A2:0103', 'Mamu-A2*01:03'), ('Mamu-B:02802', 'Mamu-B*028:02'), ('Mamu-A1:02501', 'Mamu-A1*025:01'), ('Mamu-A2:0529', 'Mamu-A2*05:29'), ('Mamu-B:02101', 'Mamu-B*021:01'), ('Patr-A1401', 'Patr-A*14:01'), ('Mamu-B:02801', 'Mamu-B*028:01'), ('Mamu-A1:10503', 'Mamu-A1*105:03'), ('Mamu-A1:00105', 'Mamu-A1*001:05'), ('Patr-B3001', 'Patr-B*30:01'), ('Mamu-A2:05020', 'Mamu-A2*05:20'), ('Mamu-A1:01102', 'Mamu-A1*011:02'), ('Mamu-A3:1313', 'Mamu-A3*13:13'), ('Mamu-B:09801', 'Mamu-B*098:01'), ('Patr-B1801', 'Patr-B*18:01'), ('Mamu-A1:00705', 'Mamu-A1*007:05'), ('HLA-C07:713', 'HLA-C*07:713Q'), ('Mamu-B:03601', 'Mamu-B*036:01'), ('Mamu-B:04607', 'Mamu-B*046:07'), ('Patr-A0701', 'Patr-A*07:01'), ('Patr-B0301', 'Patr-B*03:01'), ('Mamu-B:07702', 'Mamu-B*077:02'), ('HLA-B27:185', 'HLA-B*27:185Q'), ('Mamu-B:05901', 'Mamu-B*013:03'), ('Mamu-A2:0533', 'Mamu-A2*05:33'), ('SLA-1:0701', 'SLA-1*07:01'), ('Mamu-A4:0302', 'Mamu-A4*03:02'), ('Patr-A1703', 'Patr-A*17:03'), ('Mamu-A1:01907', 'Mamu-A1*019:07'), ('Mamu-A1:05901', 'Mamu-A1*059:01'), ('Mamu-A1:09101', 'Mamu-A1*091:01'), ('Mamu-B:04702', 'Mamu-B*047:02'), ('Mamu-A1:01803', 'Mamu-A1*018:03'), ('Patr-B0802', 'Patr-B*08:02'), ('Mamu-A2:0509', 'Mamu-A2*05:09'), ('Patr-A1502', 'Patr-A*15:02'), ('Mamu-A1:06301', 'Mamu-A1*063:01'), ('Mamu-B:04001', 'Mamu-B*015:04'), ('Mamu-A1:00103', 'Mamu-A1*001:03'), ('Mamu-B:03301', 'Mamu-B*033:01'), ('Mamu-B:06701', 'Mamu-B*067:01'), ('Mamu-B:04617', 'Mamu-B*046:17'), ('Mamu-A1:05201', 'Mamu-A1*052:01'), ('Mamu-B:07701', 'Mamu-B*077:01'), ('Mamu-B:06903', 'Mamu-B*069:03'), ('Mamu-B:01502', 'Mamu-B*015:02'), ('Patr-B1703', 'Patr-B*17:03'), ('Patr-A0801', 'Patr-A*08:01'), ('Mamu-A2:0518', 'Mamu-A2*05:18'), ('Mamu-A2:0522', 'Mamu-A2*05:22'), ('Mamu-A6:0103', 'Mamu-A6*01:03'), ('Mamu-A1:08501', 'Mamu-A1*085:01'), ('Mamu-B:03901', 'Mamu-B*039:01'), ('Mamu-A3:1308', 'Mamu-A3*13:08'), ('Mamu-A7:0103', 'Mamu-A7*01:03'), ('Mamu-B:00704', 'Mamu-B*007:04'), ('Mamu-A1:07403', 'Mamu-A1*074:03'), ('Mamu-B:08601', 'Mamu-B*086:01'), ('Patr-C0801', 'Patr-C*08:01'), ('Mamu-A1:01903', 'Mamu-A1*019:03'), ('Patr-A1501', 'Patr-A*15:01'), ('Mamu-B:09301', 'Mamu-B*093:01'), ('Mamu-B:05301', 'Mamu-B*053:01'), ('Mamu-A2:0514', 'Mamu-A2*05:14'), ('Mamu-A2:0536', 'Mamu-A2*05:36'), ('SLA-2:0202', 'SLA-2*02:02'), ('Mamu-B:08101', 'Mamu-B*081:01'), ('Mamu-B:05601', 'Mamu-B*056:01'), ('Patr-A1701', 'Patr-A*17:01'), ('Mamu-A1:05601', 'Mamu-A1*056:01'), ('Mamu-B:01601', 'Mamu-B*016:01'), ('Mamu-A3:1312', 'Mamu-A3*13:12'), ('Mamu-A1:01806', 'Mamu-A1*018:06'), ('Patr-A0301', 'Patr-A*03:01'), ('Mamu-A1:02601', 'Mamu-A1*026:01'), ('Mamu-A1:04501', 'Mamu-A1*045:01'), ('HLA-A24:329', 'HLA-A*24:329Q'), ('Gogo-B0101', 'Gogo-B*01:01'), ('Mamu-B:07301', 'Mamu-B*073:01'), ('Mamu-B:06101', 'Mamu-B*061:01'), ('Mamu-B:09701', 'Mamu-B*097:01'), ('Mamu-B:00102', 'Mamu-B*001:02'), ('Mamu-B:06102', 'Mamu-B*061:02'), ('Patr-A1001', 'Patr-A*10:01'), ('SLA-6:0102', None), ('Mamu-A1:01002', 'Mamu-A1*010:02'), ('HLA-A03:260', 'HLA-C*03:284N'), ('Patr-A0601', 'Patr-A*06:01'), ('Patr-B3701', 'Patr-B*37:01'), ('Patr-B2901', 'Patr-B*29:01'), ('Mamu-B:04616', 'Mamu-B*046:16'), ('Mamu-A2:05150', 'Mamu-A2*05:15'), ('Patr-A1601', 'Patr-A*16:01'), ('Mamu-A1:00703', 'Mamu-A1*007:03'), ('Mamu-B:02102', 'Mamu-B*021:02'), ('Patr-C1101', 'Patr-C*11:01'), ('Mamu-B:06301', 'Mamu-B*063:01'), ('Mamu-A1:02301', 'Mamu-A1*023:01'), ('SLA-2:0601', 'SLA-2*06:01'), ('Mamu-A1:04902', 'Mamu-A1*049:02'), ('Mamu-A1:05402', 'Mamu-A1*054:02'), ('Mamu-B:05802', 'Mamu-B*058:02'), ('Mamu-B:04704', 'Mamu-B*047:04'), ('Patr-C0904', 'Patr-C*09:04'), ('Mamu-A2:0539', 'Mamu-A2*05:39'), ('Patr-B2201', 'Patr-B*22:01'), ('Patr-C0301', 'Patr-C*03:01'), ('Mamu-A1:00403', 'Mamu-A1*004:03'), ('Mamu-A1:00702', 'Mamu-A1*007:02'), ('Patr-B2801', 'Patr-B*28:01'), ('SLA-1:0401', 'SLA-1*04:01'), ('Mamu-B:02901', 'Mamu-B*029:01'), ('Mamu-B:00601', 'Mamu-B*006:01'), ('Mamu-B:05501', 'Mamu-B*055:01'), ('Mamu-B:06902', 'Mamu-B*069:02'), ('Mamu-B:04610', 'Mamu-B*046:10'), ('Mamu-B:00401', 'Mamu-B*004:01'), ('Mamu-B:04612', 'Mamu-B*046:12'), ('Mamu-B:05702', 'Mamu-B*057:02'), ('Mamu-B:08701', 'Mamu-B*087:01'), ('Mamu-B:06702', 'Mamu-B*067:02'), ('Mamu-A1:02302', 'Mamu-A1*023:02'), ('Mamu-B:05701', 'Mamu-B*057:01'), ('Mamu-B:08603', 'Mamu-B*086:03'), ('Mamu-B:06601', 'Mamu-B*066:01'), ('Patr-B0203', 'Patr-B*02:03'), ('Patr-C0304', 'Patr-C*03:04'), ('Mamu-A2:05160', 'Mamu-A2*05:16'), ('Mamu-B:02602', 'Mamu-B*057:06'), ('Mamu-B:03001', 'Mamu-B*030:01'), ('Patr-B1001', 'Patr-B*10:01'), ('Patr-B1702', 'Patr-B*17:02'), ('Mamu-A4:14030', 'Mamu-A4*14:03'), ('Mamu-B:05103', 'Mamu-B*051:03'), ('Mamu-B:05101', 'Mamu-B*051:01'), ('Mamu-B:03801', 'Mamu-B*038:01'), ('Mamu-A2:0538', 'Mamu-A2*05:38'), ('Mamu-A2:0520', 'Mamu-A2*05:20'), ('Mamu-B:04002', 'Mamu-B*015:05'), ('Mamu-B:01701', 'Mamu-B*017:01'), ('Mamu-B:05002', 'Mamu-B*050:02'), ('Patr-C1302', 'Patr-C*13:02'), ('Mamu-B:04402', 'Mamu-B*044:02'), ('Mamu-A1:07402', 'Mamu-A1*074:02'), ('Mamu-A6:0105', 'Mamu-A6*01:05'), ('Mamu-B:08602', 'Mamu-B*086:02'), ('Mamu-A3:1306', 'Mamu-A3*13:06'), ('Mamu-B:07903', 'Mamu-B*079:03'), ('SLA-2:0701', 'SLA-2*07:01'), ('Mamu-B:00302', 'Mamu-B*003:02'), ('Mamu-A7:0201', 'Mamu-A7*02:01'), ('SLA-6:0104', 'SLA-6*04:01'), ('SLA-2:1001', 'SLA-2*10:01'), ('Mamu-A1:01104', 'Mamu-A1*011:04'), ('Mamu-A1:01201', 'Mamu-A1*012:01'), ('SLA-2:1201', 'SLA-2*12:01'), ('Mamu-B:10101', 'Mamu-B*101:01'), ('SLA-6:0101', 'SLA-6*01:01'), ('Patr-B2401', 'Patr-B*24:01'), ('Mamu-A2:05320', 'Mamu-A2*05:32'), ('SLA-1:0501', 'SLA-1*05:01'), ('Mamu-B:04603', 'Mamu-B*046:03'), ('Mamu-B:06401', 'Mamu-B*064:01'), ('Mamu-B:02703', 'Mamu-B*027:03'), ('Mamu-A1:02202', 'Mamu-A1*022:02'), ('Mamu-A1:00102', 'Mamu-A1*001:02'), ('Mamu-A1:02802', 'Mamu-A1*028:02'), ('Mamu-B:07501', 'Mamu-B*075:01'), ('Mamu-A1:04102', 'Mamu-A1*041:02'), ('Mamu-B:02201', 'Mamu-B*022:01'), ('Mamu-A1:11201', 'Mamu-A1*112:01'), ('Mamu-A1:10701', 'Mamu-A1*107:01'), ('Patr-A0302', 'Patr-A*03:02'), ('Mamu-B:07801', 'Mamu-B*078:01'), ('Patr-C0901', 'Patr-C*09:01'), ('Mamu-A3:1310', 'Mamu-A3*13:10'), ('Mamu-A1:00301', 'Mamu-A1*003:01'), ('Mamu-A1:04903', 'Mamu-A1*049:03'), ('HLA-C04:61', 'HLA-C*04:61N'), ('Mamu-B:07002', 'Mamu-B*070:02'), ('Mamu-A2:0526', 'Mamu-A2*05:26'), ('Mamu-B:06103', 'Mamu-B*061:03'), ('Mamu-B:01903', 'Mamu-B*019:03'), ('Mamu-A3:1311', 'Mamu-A3*13:11'), ('Mamu-B:03602', 'Mamu-B*036:02'), ('Mamu-A4:1407', 'Mamu-A4*14:07'), ('Mamu-B:09101', 'Mamu-B*091:01'), ('SLA-2:0201', 'SLA-2*02:01'), ('Mamu-B:08301', 'Mamu-B*083:01'), ('Mamu-A2:0543', 'Mamu-A2*05:43'), ('Mamu-B:09001', 'Mamu-B*090:01'), ('Patr-C0902', 'Patr-C*09:02'), ('Mamu-B:06003', 'Mamu-B*060:03'), ('Mamu-A1:04801', 'Mamu-A1*048:01'), ('Mamu-A1:00101', 'Mamu-A1*001:01'), ('Mamu-B:03201', 'Mamu-B*032:01'), ('Mamu-A1:06601', 'Mamu-A1*066:01'), ('Mamu-B:00501', 'Mamu-B*005:01'), ('Mamu-A2:0101', 'Mamu-A2*01:01'), ('Mamu-A2:0535', 'Mamu-A2*05:35'), ('Mamu-A1:01905', 'Mamu-A1*019:05'), ('Mamu-A2:0513', 'Mamu-A2*05:13'), ('Mamu-A1:05401', 'Mamu-A1*054:01'), ('Mamu-A3:1301', 'Mamu-A3*13:01'), ('Mamu-A1:01101', 'Mamu-A1*011:01'), ('Mamu-A4:1404', 'Mamu-A4*14:04'), ('Mamu-A3:1307', 'Mamu-A3*13:07'), ('Mamu-B:05401', 'Mamu-B*054:01'), ('Patr-B2501', 'Patr-B*25:01'), ('Mamu-A2:0506', 'Mamu-A2*05:06'), ('Mamu-A1:01902', 'Mamu-A1*019:02'), ('Mamu-B:03103', 'Mamu-B*031:03'), ('Mamu-A1:01904', 'Mamu-A1*019:04'), ('Patr-A2401', 'Patr-A*24:01'), ('Patr-A0402', 'Patr-A*04:02'), ('Mamu-B:08201', 'Mamu-B*082:01'), ('Mamu-B:04502', 'Mamu-B*045:02'), ('Mamu-B:06803', 'Mamu-B*068:03'), ('Mamu-B:09102', 'Mamu-B*091:02'), ('Mamu-A2:0542', 'Mamu-A2*05:42'), ('HLA-A02:437', 'HLA-A*02:437Q'), ('Mamu-B:00702', 'Mamu-B*007:02'), ('SLA-1:0801', 'SLA-1*08:01'), ('Mamu-B:07602', 'Mamu-B*076:02'), ('HLA-B41:56', 'HLA-B*41:56Q'), ('Mamu-B:04504', 'Mamu-B*045:04'), ('Mamu-A4:0205', 'Mamu-A4*02:05'), ('Mamu-A5:30010', 'Mamu-A5*30:01'), ('Mamu-B:03003', 'Mamu-B*030:03'), ('Mamu-B:01201', 'Mamu-B*012:01'), ('Mamu-A1:01901', 'Mamu-A1*019:01'), ('Mamu-B:03002', 'Mamu-B*030:02'), ('Patr-C0701', 'Patr-C*07:01'), ('Mamu-B:04801', 'Mamu-B*048:01'), ('SLA-1:0601', 'SLA-1*06:01'), ('Patr-A0501', 'Patr-A*05:01'), ('Mamu-A1:01601', 'Mamu-A1*016:01'), ('Patr-A2301', 'Patr-A*22:01'), ('Mamu-A4:1405', 'Mamu-A4*14:05'), ('Mamu-A4:0203', 'Mamu-A4*02:03'), ('Mamu-B:04405', 'Mamu-B*044:05'), ('Mamu-A1:06501', 'Mamu-A1*065:01'), ('Mamu-A1:01805', 'Mamu-A1*018:05'), ('Mamu-B:06802', 'Mamu-B*068:02'), ('SLA-2:0301', 'SLA-2*03:01'), ('Patr-B1101', 'Patr-B*11:01'), ('Patr-C0202', 'Patr-C*02:02'), ('Mamu-A1:04003', 'Mamu-A1*040:03'), ('Patr-B1301', 'Patr-B*13:01'), ('Patr-C0903', 'Patr-C*09:03'), ('SLA-1:1301', 'SLA-1*13:01'), ('Mamu-A1:00305', 'Mamu-A1*003:05'), ('Mamu-A2:0545', 'Mamu-A2*05:45'), ('Mamu-B:06805', 'Mamu-B*068:05'), ('Mamu-A1:04301', 'Mamu-A1*043:01'), ('Mamu-A3:1304', 'Mamu-A3*13:04'), ('Mamu-B:00801', 'Mamu-B*008:01'), ('Mamu-B:04611', 'Mamu-B*046:11'), ('Mamu-B:03005', 'Mamu-B*030:05'), ('Mamu-B:04101', 'Mamu-B*041:01'), ('Patr-C0303', 'Patr-B*03:02'), ('Mamu-B:04401', 'Mamu-B*044:01'), ('Mamu-B:04613', 'Mamu-B*046:13'), ('Mamu-B:07502', 'Mamu-B*075:02'), ('Patr-B2301', 'Patr-B*23:01'), ('Mamu-A1:00801', 'Mamu-A1*008:01'), ('Mamu-A2:0541', 'Mamu-A2*05:41'), ('Mamu-A1:00704', 'Mamu-A1*007:04'), ('Mamu-B:05104', 'Mamu-B*051:04'), ('Mamu-B:01301', 'Mamu-B*013:01'), ('Patr-C1501', 'Patr-C*15:01'), ('Patr-C0203', 'Patr-C*02:03'), ('Patr-A1702', 'Patr-A*17:02'), ('HLA-A02:581', 'HLA-A*02:581Q'), ('Mamu-A7:0101', 'Mamu-A7*01:01'), ('Mamu-A4:1409', 'Mamu-A4*14:09'), ('Mamu-A2:0511', 'Mamu-A2*05:11'), ('Mamu-A1:05101', 'Mamu-A1*051:01'), ('Mamu-A4:1408', 'Mamu-A4*14:08'), ('Mamu-B:07001', 'Mamu-B*070:01'), ('Patr-B2302', 'Patr-B*23:02'), ('HLA-A02:728', 'HLA-A*02:728Q'), ('Patr-C0205', 'Patr-C*02:05'), ('Mamu-B:02103', 'Mamu-B*021:03'), ('Mamu-A2:0547', None), ('SLA-2:0401', 'SLA-2*04:01'), ('SLA-3:0502', 'SLA-3*05:02'), ('HLA-A02:795', 'HLA-A*02:795Q'), ('Mamu-B:05102', 'Mamu-B*051:02'), ('Mamu-B:07101', 'Mamu-B*071:01'), ('Mamu-B:03401', 'Mamu-B*034:01'), ('Mamu-B:07201', 'Mamu-B*072:01'), ('Mamu-B:09901', 'Mamu-B*099:01'), ('Patr-C0501', 'Patr-C*05:01'), ('Patr-A0803', 'Patr-A*08:03'), ('SLA-3:0401', 'SLA-3*04:01'), ('Mamu-A2:2401', 'Mamu-A2*24:01'), ('Mamu-B:08401', 'Mamu-B*084:01'), ('Mamu-A1:05702', 'Mamu-A1*057:02'), ('Patr-C1301', 'Patr-C*13:01'), ('Patr-C0302', 'Patr-C*03:02'), ('Mamu-B:06904', 'Mamu-B*069:04'), ('Mamu-A4:1402', 'Mamu-A4*14:02'), ('Mamu-A1:07301', 'Mamu-A1*073:01'), ('Mamu-A2:0528', 'Mamu-A2*05:28'), ('Mamu-A4:01020', 'Mamu-A4*01:02'), ('Patr-B0501', 'Patr-B*05:01'), ('Mamu-B:06002', 'Mamu-B*060:02'), ('Mamu-B:02501', 'Mamu-B*025:01'), ('Mamu-B:00202', 'Mamu-B*002:02'), ('Patr-B0302', 'Patr-B*03:02'), ('Patr-B1701', 'Patr-B*17:01'), ('Mamu-A1:04002', 'Mamu-A1*040:02'), ('Mamu-A1:02803', 'Mamu-A1*028:03'), ('HLA-C03:284', 'HLA-C*03:284N'), ('Mamu-B:06302', 'Mamu-B*063:02'), ('SLA-1:0101', 'SLA-1*01:01'), ('Mamu-A1:00304', 'Mamu-A1*003:04'), ('Mamu-B:02001', 'Mamu-B*068:06'), ('Mamu-B:04615', 'Mamu-B*046:15'), ('Patr-B0901', 'Patr-B*09:01'), ('Patr-B0101', 'Patr-B*01:01'), ('Patr-B0402', 'Patr-B*04:02'), ('Mamu-A1:03201', 'Mamu-A1*032:01'), ('Patr-C0206', 'Patr-C*02:06'), ('Mamu-B:04614', 'Mamu-B*046:14'), ('Mamu-A1:01001', 'Mamu-A1*010:01'), ('Patr-B0801', 'Patr-B*08:01'), ('Mamu-A1:04904', 'Mamu-A1*049:04'), ('HLA-C07:226', 'HLA-C*07:226Q'), ('Mamu-A2:0540', 'Mamu-A2*05:40'), ('Mamu-A1:10601', 'Mamu-A1*106:01'), ('Mamu-A1:03301', 'Mamu-A1*033:01'), ('Patr-B0502', 'Patr-B*05:02'), ('Mamu-A2:0544', 'Mamu-A2*05:44'), ('Mamu-B:02702', 'Mamu-B*027:02'), ('Mamu-A1:00307', 'Mamu-A1*003:07'), ('Patr-C0905', 'Patr-C*09:05'), ('Mamu-B:04703', 'Mamu-B*047:03'), ('Patr-C0502', 'Patr-C*05:02'), ('Mamu-A5:3006', 'Mamu-A5*30:06'), ('Mamu-B:07402', 'Mamu-B*074:02'), ('Mamu-A1:09201', 'Mamu-A1*092:01'), ('Mamu-A2:0501', 'Mamu-A2*05:01'), ('SLA-3:0304', 'SLA-3*03:04'), ('Patr-A1801', 'Patr-A*18:01'), ('Mamu-A3:1302', 'Mamu-A3*13:02'), ('HLA-A24:378', 'HLA-A*24:378Q'), ('Mamu-B:02601', 'Mamu-B*057:05'), ('Mamu-A1:04901', 'Mamu-A1*049:01'), ('SLA-3:0501', 'SLA-3*05:01'), ('SLA-3:0601', 'SLA-3*06:01'), ('SLA-2:0501', 'SLA-2*05:01'), ('Mamu-A3:1309', 'Mamu-A3*13:09'), ('Mamu-A4:0103', 'Mamu-A4*01:03'), ('SLA-2:0502', 'SLA-2*05:02'), ('Mamu-A6:0104', 'Mamu-A6*01:04'), ('BoLA-106101', 'BoLA-1*061:01'), ('SLA-10401', 'SLA-1*04:01'), ('Patr-A18:01', 'Patr-A*18:01'), ('SLA-20301', 'SLA-2*03:01'), ('BoLA-204401', 'BoLA-2*044:01'), ('Mamu-A0510', None), ('SLA-20402', 'SLA-2*04:02'), ('HLA-B3573', None), ('Patr-A14:01', 'Patr-A*14:01'), ('Patr-A06:01', 'Patr-A*06:01'), ('BoLA-N:02402', 'BoLA-4*024:02'), ('BoLA-102001', 'BoLA-1*020:01'), ('Patr-B06:01', 'Patr-B*06:01'), ('BoLA-200602', 'BoLA-2*006:02'), ('SLA-30302', 'SLA-3*03:02'), ('Patr-B03:02', 'Patr-B*03:02'), ('BoLA-N:02001', 'BoLA-1*020:01'), ('HLA-A9210', None), ('Patr-A04:04', 'Patr-A*04:04'), ('Mamu-A1305', None), ('SLA-304:01', 'SLA-3*04:01'), ('SLA-204:01', 'SLA-2*04:01'), ('HLA-A0283', None), ('BoLA-306801', 'BoLA-3*068:01'), ('BoLA-103102', 'BoLA-1*031:02'), ('BoLA-102901', 'BoLA-1*029:01'), ('HLA-B9521', None), ('Mamu-B37', None), ('Patr-C09:01', 'Patr-C*09:01'), ('Mamu-A1011:01', 'Mamu-A1*011:01'), ('Mamu-A26', 'Mamu-A1*026:01'), ('HLA-B0744', 'HLA-B*07:44N'), ('BoLA-N:01001', 'BoLA-3*010:01'), ('HLA-A3021', None), ('HLA-B0806', None), ('Mamu-A11', 'Mamu-A1*105:01'), ('Mamu-B:01704', 'Mamu-B*017:04'), ('BoLA-N:04801', 'BoLA-2*048:01'), ('Mamu-B49', None), ('Patr-A11:01', 'Patr-A*11:01'), ('BoLA-204402', 'BoLA-2*044:02'), ('HLA-A9224', None), ('BoLA-N:00402', 'BoLA-3*004:02'), ('BoLA-N:00101', 'BoLA-3*001:01'), ('Patr-A01:01', 'Patr-A*01:01'), ('BoLA-104901', 'BoLA-1*049:01'), ('Patr-C02:05', 'Patr-C*02:05'), ('Mamu-B07', 'Mamu-B*070:01'), ('Patr-A15:01', 'Patr-A*15:01'), ('Mamu-B1001', 'Mamu-B*100:01'), ('Patr-B07:01', 'Patr-B*07:01'), ('SLA-20102', 'SLA-2*01:02'), ('HLA-B9504', None), ('BoLA-N:03001', 'BoLA-2*030:01'), ('HLA-B5147', None), ('BoLA-500301', 'BoLA-5*003:01'), ('Patr-B04:01', 'Patr-B*04:01'), ('SLA-30301', 'SLA-3*03:01'), ('Patr-A24:01', 'Patr-A*24:01'), ('Patr-B10:01', 'Patr-B*10:01'), ('Mamu-B43', None), ('BoLA-305801', 'BoLA-3*058:01'), ('Patr-A02:01', 'Patr-A*02:01'), ('Mamu-B57', None), ('BoLA-N:05001', 'BoLA-3*050:01'), ('BoLA-305101', 'BoLA-3*051:01'), ('SLA-20701', 'SLA-2*07:01'), ('Patr-B26:01', 'Patr-B*26:01'), ('HLA-A9222', None), ('HLA-B5405', None), ('Mamu-B21', None), ('BoLA-205501', 'BoLA-2*055:01'), ('Patr-C16:01', 'Patr-C*16:01'), ('H2-Kq', 'H2-Kq'), ('Patr-C02:06', 'Patr-C*02:06'), ('Patr-A03:01', 'Patr-A*03:01'), ('BoLA-N:03901', 'BoLA-5*039:01'), ('Mamu-B3901', 'Mamu-B*039:01'), ('Mamu-B8301', 'Mamu-B*083:01'), ('Mamu-B46', None), ('HLA-B9505', None), ('BoLA-107401', 'BoLA-1*074:01'), ('HLA-A9220', None), ('BoLA-N:00602', 'BoLA-2*006:02'), ('HLA-A2307', None), ('HLA-B9527', None), ('Mamu-A0507', None), ('Mamu-B22', None), ('Patr-B11:02', 'Patr-B*11:02'), ('BoLA-N:02101', 'BoLA-1*021:01'), ('HLA-B9501', None), ('BoLA-103101', 'BoLA-1*031:01'), ('BoLA-N:01601', 'BoLA-2*016:01'), ('HLA-A9214', None), ('SLA-20201', 'SLA-2*02:01'), ('Mamu-B08', 'Mamu-B*080:01'), ('H2-Kk', 'H2-Kk'), ('HLA-B9518', None), ('HLA-A9215', None), ('Mamu-A06', None), ('BoLA-N:03501', 'BoLA-3*035:01'), ('Patr-C07:01', 'Patr-C*07:01'), ('Patr-B05:01', 'Patr-B*05:01'), ('BoLA-102301', 'BoLA-1*023:01'), ('BoLA-N:03401', 'BoLA-6*034:01'), ('BoLA-201201', 'BoLA-2*012:01'), ('BoLA-300403', 'BoLA-3*004:03'), ('BoLA-101901', 'BoLA-1*019:01'), ('HLA-A9202', None), ('BoLA-306602', 'BoLA-3*066:02'), ('Patr-B23:02', 'Patr-B*23:02'), ('Patr-A23:01', 'Patr-A*22:01'), ('Patr-A17:01', 'Patr-A*17:01'), ('Patr-C08:01', 'Patr-C*08:01'), ('BoLA-N:03801', 'BoLA-3*038:01'), ('BoLA-204801', 'BoLA-2*048:01'), ('BoLA-507201', 'BoLA-5*072:01'), ('BoLA-N:02401', 'BoLA-4*024:01'), ('BoLA-N:02301', 'BoLA-1*023:01'), ('HLA-A9201', None), ('Patr-C12:01', 'Patr-C*12:01'), ('HLA-C0409', None), ('Patr-A04:01', 'Patr-A*04:01'), ('Patr-B02:01', 'Patr-B*02:01'), ('HLA-B3938', None), ('HLA-B9507', None), ('BoLA-204501', 'BoLA-2*045:01'), ('Mamu-B44', None), ('HLA-B9528', None), ('HLA-B1823', None), ('Mamu-B8701', 'Mamu-B*087:01'), ('BoLA-104201', 'BoLA-1*042:01'), ('Chi-B1201', None), ('BoLA-N:00401', 'BoLA-3*004:01'), ('BoLA-603401', 'BoLA-6*034:01'), ('HLA-A9218', None), ('BoLA-N:03101', 'BoLA-1*031:01'), ('Mamu-A04', None), ('Patr-A09:01', 'Patr-A*09:01'), ('BoLA-N:04901', 'BoLA-1*049:01'), ('Mamu-B3002', None), ('SLA-107:01', 'SLA-1*07:01'), ('Patr-B29:01', 'Patr-B*29:01'), ('Patr-C02:02', 'Patr-C*02:02'), ('BoLA-N:01802', 'BoLA-2*018:02'), ('BoLA-N:03601', 'BoLA-3*036:01'), ('H2-Qa2', None), ('HLA-A2409', None), ('HLA-B9502', None), ('BoLA-301702', 'BoLA-3*017:02'), ('Mamu-A28', None), ('BoLA-305001', 'BoLA-3*050:01'), ('HLA-B9514', None), ('BoLA-203202', 'BoLA-2*032:02'), ('Mamu-A1001:01', 'Mamu-A1*001:01'), ('BoLA-300402', 'BoLA-3*004:02'), ('Patr-B37:01', 'Patr-B*37:01'), ('Mamu-B70', None), ('BoLA-N:02501', 'BoLA-2*025:01'), ('Mamu-B017:04', 'Mamu-B*017:04'), ('BoLA-207101', 'BoLA-2*071:01'), ('HLA-B9508', None), ('BoLA-601302', 'BoLA-6*013:02'), ('SLA-107:02', 'SLA-1*07:02'), ('BoLA-300401', 'BoLA-3*004:01'), ('Patr-A08:01', 'Patr-A*08:01'), ('BoLA-300103', 'BoLA-3*001:03'), ('BoLA-203001', 'BoLA-2*030:01'), ('BoLA-N:04701', 'BoLA-2*047:01'), ('Mamu-B003:01', 'Mamu-B*003:01'), ('HLA-B9522', None), ('SLA-205:02', 'SLA-2*05:02'), ('BoLA-305901', 'BoLA-3*059:01'), ('HLA-B3565', None), ('Mamu-A0506', None), ('BoLA-206901', 'BoLA-2*069:01'), ('Mamu-A01', 'Mamu-A1*001:01'), ('BoLA-201601', 'BoLA-2*016:01'), ('Patr-C09:04', 'Patr-C*09:04'), ('Mamu-B71', None), ('HLA-A9212', None), ('Mamu-A70103', 'Mamu-A7*01:03'), ('BoLA-N:01501', 'BoLA-6*015:01'), ('Patr-C03:01', 'Patr-C*03:01'), ('H2-Kb', 'H2-Kb'), ('Patr-C06:01', 'Patr-C*06:01'), ('Mamu-B19', None), ('BoLA-303501', 'BoLA-3*035:01'), ('Mamu-B41', None), ('H2-Qa1', 'H2-Qa-1a'), ('Patr-B22:01', 'Patr-B*22:01'), ('BoLA-204701', 'BoLA-2*047:01'), ('Patr-A15:02', 'Patr-A*15:02'), ('Patr-A03:02', 'Patr-A*03:02'), ('BoLA-N:01402', 'BoLA-6*014:02'), ('HLA-A9223', None), ('SLA-20502', 'SLA-2*05:02'), ('Mamu-A0509', None), ('Patr-B24:02', 'Patr-B*24:02'), ('Mamu-B47', None), ('Chi-B0401', None), ('BoLA-202201', 'BoLA-2*022:01'), ('BoLA-305002', 'BoLA-3*050:02'), ('Mamu-B63', None), ('SLA-21201', 'SLA-2*12:01'), ('BoLA-N:01801', 'BoLA-2*018:01'), ('BoLA-601402', 'BoLA-6*014:02'), ('HLA-A9211', None), ('Mamu-A1002:01', 'Mamu-A1*002:01'), ('Patr-B03:01', 'Patr-B*03:01'), ('BoLA-205401', 'BoLA-2*054:01'), ('BoLA-204601', 'BoLA-2*046:01'), ('Mamu-A1602', None), ('Mamu-B27', None), ('HLA-C0301', None), ('Patr-A17:03', 'Patr-A*17:03'), ('BoLA-N:02702', 'BoLA-3*027:02'), ('Patr-C04:01', 'Patr-C*04:01'), ('BoLA-202602', 'BoLA-2*026:02'), ('BoLA-N:00301', 'BoLA-5*003:01'), ('HLA-B0808', None), ('Patr-C09:02', 'Patr-C*09:02'), ('Patr-C11:01', 'Patr-C*11:01'), ('BoLA-N:04301', 'BoLA-2*043:01'), ('BoLA-N:02601', 'BoLA-2*026:01'), ('BoLA-402402', 'BoLA-4*024:02'), ('SLA-21001', 'SLA-2*10:01'), ('HLA-B9519', None), ('BoLA-402401', 'BoLA-4*024:01'), ('HLA-B9529', None), ('BoLA-303701', 'BoLA-3*037:01'), ('BoLA-N:03701', 'BoLA-3*037:01'), ('HLA-B9516', None), ('Patr-B01:01', 'Patr-B*01:01'), ('SLA-30304', 'SLA-3*03:04'), ('BoLA-N:02801', 'BoLA-1*028:01'), ('Patr-B02:03', 'Patr-B*02:03'), ('H2-Dq', 'H2-Dq'), ('Patr-A13:01', 'Patr-A*13:01'), ('SLA-104:01', 'SLA-1*04:01'), ('BoLA-200601', 'BoLA-2*006:01'), ('BoLA-305301', 'BoLA-3*053:01'), ('BoLA-306501', 'BoLA-3*065:01'), ('HLA-B9524', None), ('Patr-B17:01', 'Patr-B*17:01'), ('Patr-A04:02', 'Patr-A*04:02'), ('BoLA-N:02602', 'BoLA-2*026:02'), ('Mamu-B66', 'Mamu-B*066:01'), ('SLA-30601', 'SLA-3*06:01'), ('H2-Db', 'H2-Db'), ('Mamu-B67', None), ('H2-Lq', 'H2-Lq'), ('Patr-B13:01', 'Patr-B*13:01'), ('BoLA-200802', 'BoLA-2*008:02'), ('BoLA-601502', 'BoLA-6*015:02'), ('BoLA-301001', 'BoLA-3*010:01'), ('HLA-B9526', None), ('BoLA-N:01901', 'BoLA-1*019:01'), ('BoLA-N:05201', 'BoLA-3*052:01'), ('Mamu-A20101', 'Mamu-A2*01:01'), ('Mamu-B69', None), ('HLA-A9207', None), ('BoLA-N:00103', 'BoLA-3*001:03'), ('Mamu-B05', 'Mamu-B*050:02'), ('Patr-A08:02', 'Patr-A*08:02'), ('BoLA-N:01602', 'BoLA-2*016:02'), ('BoLA-300102', 'BoLA-3*001:02'), ('BoLA-200801', 'BoLA-2*008:01'), ('Mamu-A07', 'Mamu-A1*007:01'), ('BoLA-206001', 'BoLA-2*060:01'), ('Patr-A12:01', 'Patr-A*12:01'), ('Patr-C15:01', 'Patr-C*15:01'), ('Patr-A17:02', 'Patr-A*17:02'), ('Patr-B18:01', 'Patr-B*18:01'), ('SLA-3:0402', 'SLA-3*04:02'), ('Mamu-B40', None), ('SLA-30402', 'SLA-3*04:02'), ('Mamu-B17', 'Mamu-B*017:01'), ('Patr-B24:01', 'Patr-B*24:01'), ('BoLA-301703', 'BoLA-3*017:03'), ('HLA-B0749', None), ('Patr-C05:01', 'Patr-C*05:01'), ('Patr-C03:04', 'Patr-C*03:04'), ('BoLA-302701', 'BoLA-3*027:01'), ('BoLA-301701', 'BoLA-3*017:01'), ('BoLA-200501', 'BoLA-2*005:01'), ('Mamu-B28', None), ('BoLA-N:00901', 'BoLA-1*009:01'), ('Patr-C09:05', 'Patr-C*09:05'), ('HLA-B9532', None), ('SLA-10501', 'SLA-1*05:01'), ('BoLA-202601', 'BoLA-2*026:01'), ('HLA-A9221', None), ('BoLA-N:04101', 'BoLA-6*041:01'), ('SLA-20501', 'SLA-2*05:01'), ('SLA-11101', 'SLA-1*11:01'), ('BoLA-503901', 'BoLA-5*039:01'), ('BoLA-205701', 'BoLA-2*057:01'), ('Mamu-B64', None), ('Patr-C02:04', 'Patr-C*02:04'), ('HLA-A9204', None), ('SLA-11201', 'SLA-1*12:01'), ('BoLA-601301', 'BoLA-6*013:01'), ('Mamu-B52', 'Mamu-B*052:01'), ('SLA-10701', 'SLA-1*07:01'), ('Patr-B25:01', 'Patr-B*25:01'), ('Mamu-A0511', None), ('Patr-B20:01', 'Patr-B*20:01'), ('BoLA-N:00801', 'BoLA-2*008:01'), ('Patr-A16:01', 'Patr-A*16:01'), ('Patr-A07:01', 'Patr-A*07:01'), ('Mamu-B02', 'Mamu-B*021:01'), ('SLA-10202', 'SLA-1*02:02'), ('HLA-A9226', None), ('BoLA-604001', 'BoLA-6*040:01'), ('BoLA-N:04001', 'BoLA-6*040:01'), ('SLA-10601', 'SLA-1*06:01'), ('SLA-21002', 'SLA-2*10:02'), ('Patr-B36:01', 'Patr-B*36:01'), ('BoLA-102801', 'BoLA-1*028:01'), ('BoLA-302702', 'BoLA-3*027:02'), ('Mamu-A23', None), ('Mamu-B39', 'Mamu-B*039:01'), ('HLA-B9530', None), ('Patr-C13:01', 'Patr-C*13:01'), ('HLA-A9206', None), ('Mamu-A21', None), ('BoLA-N:01302', 'BoLA-6*013:02'), ('BoLA-201602', 'BoLA-2*016:02'), ('Patr-A09:02', 'Patr-A*09:02'), ('HLA-C1407', None), ('Mamu-B55', None), ('Patr-B21:01', 'Patr-B*21:01'), ('SLA-10101', 'SLA-1*01:01'), ('SLA-30602', 'SLA-3*06:02'), ('BoLA-N:00201', 'BoLA-3*002:01'), ('BoLA-N:02701', 'BoLA-3*027:01'), ('BoLA-307301', 'BoLA-3*073:01'), ('HLA-A0303', None), ('BoLA-102101', 'BoLA-1*021:01'), ('Eqca-1600101', 'Eqca-16*001:01'), ('Mamu-A20102', 'Mamu-A2*01:02'), ('HLA-B9515', None), ('Mamu-B45', None), ('SLA-10702', 'SLA-1*07:02'), ('Mamu-B6601', 'Mamu-B*066:01'), ('H2-Kd', 'H2-Kd'), ('HLA-A9208', None), ('Chi-B1501', None), ('Mamu-A0505', None), ('Mamu-B6002', None), ('BoLA-N:02201', 'BoLA-2*022:01'), ('Mamu-B24', None), ('Patr-A05:01', 'Patr-A*05:01'), ('HLA-B9503', None), ('BoLA-300201', 'BoLA-3*002:01'), ('BoLA-202603', 'BoLA-2*026:03'), ('Mamu-B5002', None), ('Mamu-B61', None), ('HLA-B9506', None), ('BoLA-N:05401', 'BoLA-2*054:01'), ('Patr-C02:03', 'Patr-C*02:03'), ('Mamu-B48', None), ('Mamu-A0703', None), ('Mamu-A2601', 'Mamu-A1*026:01'), ('HLA-B9523', None), ('Patr-B16:01', 'Patr-B*16:01'), ('Patr-B23:01', 'Patr-B*23:01'), ('BoLA-N:00102', 'BoLA-3*001:02'), ('Patr-B17:02', 'Patr-B*17:02'), ('SLA-20302', 'SLA-2*03:02'), ('Patr-B16:02', 'Patr-B*16:02'), ('BoLA-N:01502', 'BoLA-6*015:02'), ('BoLA-N:04201', 'BoLA-1*042:01'), ('Patr-B23:03', 'Patr-B*22:03'), ('BoLA-N:00902', 'BoLA-1*009:02'), ('SLA-20601', 'SLA-2*06:01'), ('Patr-B09:01', 'Patr-B*09:01'), ('H2-Ld', 'H2-Ld'), ('H2-Dd', 'H2-Dd'), ('BoLA-506401', 'BoLA-5*064:01'), ('BoLA-N:01101', 'BoLA-3*011:01'), ('BoLA-N:04501', 'BoLA-2*045:01'), ('Patr-C03:02', 'Patr-C*03:02'), ('HLA-A9217', None), ('BoLA-207001', 'BoLA-2*070:01'), ('Patr-A06:02', 'Patr-A*06:02'), ('SLA-30502', 'SLA-3*05:02'), ('Patr-B19:01', 'Patr-B*19:01'), ('HLA-A2465', None), ('HLA-A30:14L', 'HLA-A*30:14L'), ('Mamu-B12', None), ('BoLA-N:00601', 'BoLA-2*006:01'), ('SLA-20401', 'SLA-2*04:01'), ('HLA-B9512', None), ('BoLA-305201', 'BoLA-3*052:01'), ('SLA-30101', 'SLA-3*01:01'), ('Mamu-B01', 'Mamu-B*010:01'), ('Mamu-A2201', 'Mamu-A1*022:01'), ('BoLA-N:02901', 'BoLA-1*029:01'), ('BoLA-204301', 'BoLA-2*043:01'), ('BoLA-N:01301', 'BoLA-6*013:01'), ('Mamu-B38', None), ('Mamu-A1306', None), ('BoLA-100901', 'BoLA-1*009:01'), ('BoLA-100902', 'BoLA-1*009:02'), ('BoLA-300101', 'BoLA-3*001:01'), ('BoLA-202501', 'BoLA-2*025:01'), ('HLA-A9216', None), ('BoLA-N:01201', 'BoLA-2*012:01'), ('Patr-C02:01', 'Patr-C*02:01'), ('BoLA-N:00501', 'BoLA-2*005:01'), ('SLA-30501', 'SLA-3*05:01'), ('Mamu-A02', 'Mamu-A1*002:01'), ('SLA-30503', 'SLA-3*05:03'), ('Mamu-A25', None), ('HLA-C0502', None), ('Patr-B14:01', 'Patr-B*14:01'), ('Patr-A08:03', 'Patr-A*08:03'), ('Mamu-B20', None), ('Patr-B08:02', 'Patr-B*08:02'), ('Patr-B17:03', 'Patr-B*17:03'), ('BoLA-406301', 'BoLA-4*063:01'), ('BoLA-601501', 'BoLA-6*015:01'), ('Mamu-A19', None), ('Patr-C09:03', 'Patr-C*09:03'), ('Mamu-B03', 'Mamu-B*030:01'), ('Mamu-B36', None), ('BoLA-601401', 'BoLA-6*014:01'), ('SLA-20101', 'SLA-2*01:01'), ('SLA-30401', 'SLA-3*04:01'), ('BoLA-N:01701', 'BoLA-3*017:01'), ('Patr-C03:03', 'Patr-B*03:02'), ('Patr-B22:02', 'Patr-B*22:02'), ('HLA-A9205', None), ('BoLA-201802', 'BoLA-2*018:02'), ('BoLA-303801', 'BoLA-3*038:01'), ('Patr-B05:02', 'Patr-B*05:02'), ('Patr-C13:02', 'Patr-C*13:02'), ('SLA-10201', 'SLA-1*02:01'), ('Mamu-B5802', None), ('Mamu-B004:01', 'Mamu-B*004:01'), ('BoLA-N:04601', 'BoLA-2*046:01'), ('HLA-B9525', None), ('HLA-A9203', None), ('Mamu-B53', None), ('Patr-B28:01', 'Patr-B*28:01'), ('BoLA-106701', 'BoLA-1*067:01'), ('Mamu-B065:02', 'Mamu-B*065:02'), ('BoLA-604101', 'BoLA-6*041:01'), ('BoLA-205601', 'BoLA-2*056:01'), ('BoLA-306601', 'BoLA-3*066:01'), ('Patr-B12:02', 'Patr-B*12:02'), ('HLA-B9513', None), ('Mamu-B03901', 'Mamu-B*039:01'), ('Patr-C05:02', 'Patr-C*05:02'), ('SLA-20202', 'SLA-2*02:02'), ('HLA-B9520', None), ('BoLA-206201', 'BoLA-2*062:01'), ('HLA-B9510', None), ('HLA-A9209', None), ('Mamu-A03', None), ('HLA-B9517', None), ('Mamu-B65', None), ('BoLA-N:05101', 'BoLA-3*051:01'), ('BoLA-N:05601', 'BoLA-2*056:01'), ('HLA-A9219', None), ('Patr-B08:01', 'Patr-B*08:01'), ('Eqca-100101', 'Eqca-1*001:01'), ('Mamu-B04', 'Mamu-B*041:01'), ('BoLA-T2C', None), ('Patr-B04:02', 'Patr-B*04:02'), ('Mamu-B001:01', 'Mamu-B*001:01'), ('BoLA-N:05301', 'BoLA-3*053:01'), ('Patr-B07:02', 'Patr-B*07:02'), ('SLA-30303', 'SLA-3*03:03'), ('BoLA-N:04401', 'BoLA-2*044:01'), ('BoLA-201801', 'BoLA-2*018:01'), ('Patr-A10:01', 'Patr-A*10:01'), ('BoLA-303601', 'BoLA-3*036:01'), ('BoLA-N:01401', 'BoLA-6*014:01'), ('Mamu-B008:01', 'Mamu-B*008:01'), ('BoLA-301101', 'BoLA-3*011:01'), ('Patr-B11:01', 'Patr-B*11:01'), ('BoLA-N:00802', 'BoLA-2*008:02'), ('SLA-10801', 'SLA-1*08:01'), ('Patr-B35:01', 'Patr-B*35:01'), ('Patr-C10:01', 'Patr-C*10:01'), ('BoLA-N:01702', 'BoLA-3*017:02'), ('SLA-11301', 'SLA-1*13:01'), ('BoLA-N:05501', 'BoLA-2*055:01'), ('Patr-B01:02', 'Patr-B*01:02'), ('Patr-B27:01', 'Patr-B*27:01'), ('HLA-B9509', None), ('Patr-B30:01', 'Patr-B*30:01'), ('Mamu-A0602', None), ('Mamu-A24', None), ('Mamu-AG*03011', 'Mamu-AG*03:01'), ('Mamu-AG*01', 'Mamu-AG*01:01'), ('Mamu-AG*02011', 'Mamu-AG*02:01'), ('Mamu-AG*02012', 'Mamu-AG*02:01'), ('Mamu-AG*0302', 'Mamu-AG*03:02'), ('Mamu-AG*0202', 'Mamu-AG*02:02')
    ]

    av = AlleleValidator()
    ac = Allele_Autocomplete()

    netmhcpan_dict = {
        'Tool Label': [],
        'IEDB Label': [],
        'Synonyms': [],
    }
    # Store alleles that curl command can't find.
    unavailable_alleles_2 = []

    # suspicious alleles -- must be verified manually
    suspicious_alleles_2 = []

    counter = 1
    for toolname, iedb_label in unavailable_alleles:
        print(f'{counter}/{len(unavailable_alleles)}')
        counter += 1

        synonyms = ac.get_allele_suggestions(query=toolname)
        if 0 < len(synonyms):
            # Grab all the synonyms and IEDB Label, and verify which one of these work.
            synonyms = ac.get_allele_suggestions(query=toolname)[0]['synonyms'].split(',')
        else:
            unavailable_alleles_2.append((toolname, iedb_label))
            continue
        synonyms.insert(0, iedb_label)
        synonyms = [_.strip() for _ in synonyms]
        synonyms = [x for x in synonyms if x is not None]
        print(synonyms)
        
        working_allele_found = False
        for allele in synonyms:
            if run_curl_command_with(allele):
                # Add these when toolname does work with the curl command
                netmhcpan_dict['Tool Label'].append(allele)
                netmhcpan_dict['IEDB Label'].append(iedb_label)
                netmhcpan_dict['Synonyms'].append(', '.join(synonyms))
                working_allele_found = True
                break
        
        if not working_allele_found:
            unavailable_alleles_2.append((toolname, iedb_label))
        
    
    df = pd.DataFrame(netmhcpan_dict)
    print(df.head())
    print(len(df))

    # This file checks with the unavailable_alleles gotten from the creation of
    # netmhcpan_4_1_allele.tsv file, it will see if any of the synonyms can
    # potentially be matched and work with the netmhcpan tool.
    df.to_csv('unknown_alleles_in_autocomplete.tsv', sep='\t', index=False)


    for _ in unavailable_alleles_2:
        print(_)

def test1():
    MASTER_ALLELE_FILE = DATA_DIR + '/netmhcpan_4_1_alleles.tsv'
    master_df = pd.read_csv(MASTER_ALLELE_FILE, sep='\t', index_col=False)
    print(master_df.head())

    print(list(set(master_df['Tool Label'].to_list())))
    print(len(list(set(master_df['Tool Label'].to_list())))) # unique alleles 11083

    fresh_netmhcpan_dict = {
        'Tool Label': [],
        'IEDB Label': [],
        'Synonyms': [],
    }

    for idx, row in master_df.iterrows():
        toollabel = row['Tool Label']
        iedblabel = row['IEDB Label']
        synonym = row['Synonyms'] # This will always be 1

        # Check if the value is NaN
        if pd.isna(synonym):
            synonym = None
        
        if toollabel not in fresh_netmhcpan_dict['Tool Label']:
            fresh_netmhcpan_dict['Tool Label'].append(toollabel)
            fresh_netmhcpan_dict['IEDB Label'].append(iedblabel)
            fresh_netmhcpan_dict['Synonyms'].append(synonym)
       
    # Should have unique tool label
    fresh_master_df = pd.DataFrame(fresh_netmhcpan_dict)
    fresh_master_df = fresh_master_df.fillna('')

    for idx, row in master_df.iterrows():
        tlabel = row['Tool Label']
        iedblabel = row['IEDB Label']
        synonym = row['Synonyms'] # This will always be 1
        if pd.isna(synonym):
            continue

        indices = fresh_master_df.index[fresh_master_df['Tool Label'] == tlabel].tolist()
        print(tlabel, indices)

        
        orig_syns = fresh_master_df.at[indices[0], 'Synonyms']
        orig_syns = orig_syns.split(',')
        orig_syns.append(iedblabel)    
        orig_syns.append(synonym)
        orig_syns = list(set(orig_syns))
        # If there's a synonym that matches the tool-label, remove it.
        if tlabel in orig_syns:
            orig_syns.remove(tlabel)
        print(orig_syns)
        
        syns_str = ','.join(orig_syns)
        fresh_master_df.at[indices[0], 'Synonyms'] = syns_str
        
    for idx, row in fresh_master_df.iterrows():
        syns = row['Synonyms'].split(',')
        syns = list(set(syns))
        row['Synonyms'] = ','.join(syns)

    # the 'netmhcpan_4_1_alleles.tsv' had duplicates.
    # This new file will have the same data but with unique ones, and merged all the synonyms.
    FRESH_MASTER_ALLELE_FILE = DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    fresh_master_df.to_csv(FRESH_MASTER_ALLELE_FILE, sep='\t', index=False)



def add_verfied_standalone_alleles():
    # Tool Label should be unique in this file
    MASTER_ALLELES_FILE=DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    VFYD_CLI_ALLELES=DATA_DIR + '/verified_standalone_alleles.tsv'

    master_alleles_df = pd.read_csv(MASTER_ALLELES_FILE, sep='\t', index_col=False)
    cli_alleles = pd.read_csv(VFYD_CLI_ALLELES, sep='\t', index_col=False)

    unseen_tlabels = []
    for idx, row in cli_alleles.iterrows():
        tlabel = row['Tool Label']
        synonym = row['Synonyms']
        indices = master_alleles_df.index[master_alleles_df['Tool Label'] == tlabel].tolist()

        if not indices:
            unseen_tlabels.append(tlabel)
            continue
        
        master_synonyms = master_alleles_df.at[indices[0], 'Synonyms']

        if pd.isna(master_synonyms):
            master_alleles_df.at[indices[0], 'Synonyms'] = synonym
            continue
        
        master_synonyms = master_synonyms.split(',')
        
        if synonym not in master_synonyms:
            master_synonyms.append(synonym)

        master_synonyms = ','.join(master_synonyms)

        master_alleles_df.at[indices[0], 'Synonyms'] = master_synonyms

    FRESH_MASTER_ALLELE_FILE = DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    master_alleles_df.to_csv(FRESH_MASTER_ALLELE_FILE, sep='\t', index=False)

    print("-----")
    print(unseen_tlabels)
    print(len(unseen_tlabels))


def identify_mroid_for_unknown_alleles():
    PDIR = str(Path(__file__).parent.parent)
    UNKNOWN_CLI_ALLELES=DATA_DIR + '/unknown_standalone_alleles.tsv'
    TOOLS_MAPPING_FILE = PDIR + '/Tools_MRO_mapping.xlsx'
    original_tm_df = pd.read_excel(TOOLS_MAPPING_FILE, engine='openpyxl', index_col=False)
    unknown_cli_df = pd.read_csv(UNKNOWN_CLI_ALLELES, sep='\t', index_col=False)


    unseen_tlabels = []
    mro_ids = []
    for idx, row in unknown_cli_df.iterrows():
        cli_tlabel = row['NetMHCpan_Standalone_Allele']

        indices = original_tm_df.index[original_tm_df['term'] == cli_tlabel].tolist()

        if not indices:
            unseen_tlabels.append(cli_tlabel)
            mro_ids.append(None)
            continue

        mroid = original_tm_df.at[indices[0], 'MRO ID']
        mro_ids.append(mroid)
    
    unknown_cli_df['MRO ID'] = mro_ids

    UNKNOWN_ALLELES_WITH_MROID = DATA_DIR + '/unknown_standalone_alleles_with_mroid.tsv'
    unknown_cli_df.to_csv(UNKNOWN_ALLELES_WITH_MROID, sep='\t', index=False)

    print('-----')
    print(unseen_tlabels)
    print(len(unseen_tlabels))

def find_corr_labels_using_mroid():
    PDIR = str(Path(__file__).parent.parent)
    MASTER_ALLELES_FILE=DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    UNKNOWN_CLI_ALLELES=DATA_DIR + '/unknown_standalone_alleles_with_mroid.tsv'
    TOOLS_MAPPING_FILE = PDIR + '/tools-mapping.tsv'

    unknown_cli_df = pd.read_csv(UNKNOWN_CLI_ALLELES, sep='\t', index_col=False)
    tm_df = pd.read_csv(TOOLS_MAPPING_FILE, sep='\t', index_col=False)
    master_alleles_df = pd.read_csv(MASTER_ALLELES_FILE, sep='\t', index_col=False)

    
    unseen_tlabels = []
    mro_ids = []
    for idx, row in unknown_cli_df.iterrows():
        mroid = row['MRO ID']

        corr_tlabel_idices = tm_df.index[tm_df['MRO ID'] == mroid].tolist()

        if not corr_tlabel_idices:
            # MRO ID is not searchable in the tools-mapping file.
            unseen_tlabels.append((row['Tool Label'], mroid))
            continue
        
        # find corresponding 'Tool Label' from the tools-mapping
        corr_tlabel = tm_df.at[corr_tlabel_idices[0], 'Tool Label']

        # check existence of this corresponding tool label from the master alleles file,
        # and if so, add the original unkonwn tool label as the synonym.
        corr_tlabel_idices_in_master = master_alleles_df.index[master_alleles_df['Tool Label'] == corr_tlabel].tolist()

        if not corr_tlabel_idices_in_master:
            # the corresponding tool label from the tools-mapping not found in the master sheet.
            unseen_tlabels.append((row['Tool Label'], row['NetMHCpan_Standalone_Allele'], mroid))

            # Expecting 'Gogo-B0101' to be here, as the master sheet doesn't have this allele at all.
            # NOTE: Because it the 'unknown_standalone_allels_with_mroid' was based off of ptyhon's difflib,
            # Seems like it's not 100% accurate. However, I have spotted 4 alleles that were incorrect.
            # The rest of the allles seem to be matching well.
            #
            # I will manually take care of those 4 alleles in the bottom.
            continue
        
        corr_master_synonyms = master_alleles_df.at[corr_tlabel_idices_in_master[0], 'Synonyms']

        unknown_tlabel = row['Tool Label']
        unknown_cli_tlabel = row['NetMHCpan_Standalone_Allele']
        
        if pd.isna(corr_master_synonyms):    
            unknown_labels = [unknown_tlabel, unknown_cli_tlabel]
            master_alleles_df.at[corr_tlabel_idices_in_master[0], 'Synonyms'] = ','.join(unknown_labels)
            continue

        corr_master_synonyms = corr_master_synonyms.split(',')
        corr_master_synonyms.append(unknown_tlabel)
        corr_master_synonyms.append(unknown_cli_tlabel)
        master_alleles_df.at[corr_tlabel_idices_in_master[0], 'Synonyms'] = ','.join(corr_master_synonyms)


    # Extra alleles that need to be added
    unseen_tlabels = [
        ('Gogo-B0101', 'Gogo-B*01:01', 'MRO:0000964'), 
        ('Mamu-A01', 'Mamu-A1*001:01', 'MRO:0001358'), 
        ('Mamu-A02', 'Mamu-A1*002:01', 'MRO:0001359'), 
        ('Mamu-A03', 'Mamu-A1*003:01', 'MRO:0038603'), # Specifically 'Mamu-A03' not found, but by looking at pattern, probably is 'Mamu-A1*003:01'.
        ] 
    for tlabel, cli_label, mroid in unseen_tlabels:
        corr_tlabel_idices = tm_df.index[tm_df['MRO ID'] == mroid].tolist()
        # find corresponding 'Tool Label' from the tools-mapping
        corr_tlabel = tm_df.at[corr_tlabel_idices[0], 'Tool Label']
        corr_iedblabel = tm_df.at[corr_tlabel_idices[0], 'IEDB Label']

        if tlabel == 'Gogo-B0101':
            correct_tools_label = 'Gogo-B*0101'
            syns = [corr_tlabel, corr_iedblabel, tlabel, cli_label]

        if tlabel == 'Mamu-A01':
            correct_tools_label = 'Mamu-A1*00101'
            syns = [corr_tlabel, corr_iedblabel, tlabel, cli_label]

        if tlabel == 'Mamu-A02':
            correct_tools_label = 'Mamu-A1*00201'
            syns = [corr_tlabel, corr_iedblabel, tlabel, cli_label]

        if tlabel == 'Mamu-A03':
            correct_tools_label = 'Mamu-A1*00301'
            syns = [corr_tlabel, corr_iedblabel, tlabel, cli_label]


        syns = ','.join([tlabel, cli_label])
        unseen_row = pd.DataFrame({
            'Tool Label': [correct_tools_label],
            'IEDB Label': [corr_iedblabel],
            'Synonyms': [syns],
        })

        master_alleles_df = pd.concat([master_alleles_df, unseen_row], ignore_index=True)


    MASTER_ALLELES_FILE=DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    master_alleles_df.to_csv(MASTER_ALLELES_FILE, sep='\t', index=False)

    print('----')
    print(unseen_tlabels)

def check_existence():
    MASTER_ALLELES_FILE=DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    AVAILABLE_CLI_ALLELES_FILE=DATA_DIR + '/available_standalone_unique_alleles.tsv'
    master_alleles_df = pd.read_csv(MASTER_ALLELES_FILE, sep='\t', index_col=False)
    valid_cli_alleles_df = pd.read_csv(AVAILABLE_CLI_ALLELES_FILE, sep='\t', index_col=False)
    master_synonyms = []
    unseen_alleles = []

    for idx, row in master_alleles_df.iterrows():
        syns = row['Synonyms']

        if pd.isna(syns): continue
        syns = syns.split(',')
        for s in syns: master_synonyms.append(s)

    for idx, row in valid_cli_alleles_df.iterrows():
        allele = row['Original Tool Label']
        print(allele)

        if allele not in master_alleles_df['Tool Label'].values:
            unseen_alleles.append(allele)

    print(unseen_alleles)
    print(len(unseen_alleles))

    filtered_unseen_alleles = []
    for unseen_allele in unseen_alleles:
        if unseen_allele.startswith('BoLA-NC'):
            # Shouldn't be included anyways, so let's skip
            continue

        if unseen_allele not in master_synonyms:
            filtered_unseen_alleles.append(unseen_allele)

    print(filtered_unseen_alleles)
    print(len(filtered_unseen_alleles))

    
    # The leftovers can be matched by difflib
    for yet_unseen_allele in filtered_unseen_alleles: 
        matched_allele = difflib.get_close_matches(yet_unseen_allele, master_alleles_df['Tool Label'].values, n=1)[0]
        print(yet_unseen_allele, matched_allele)
        
        # Confident that these all will have values and no nans
        indices = master_alleles_df.index[master_alleles_df['Tool Label'] == matched_allele].tolist()
        synonyms = master_alleles_df.at[indices[0], 'Synonyms']
        synonyms = synonyms.split(',')
        synonyms.append(yet_unseen_allele)
        synonyms = ','.join(synonyms)
        master_alleles_df.at[indices[0], 'Synonyms'] = synonyms

    MASTER_ALLELES_FILE=DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    master_alleles_df.to_csv(MASTER_ALLELES_FILE, sep='\t', index=False)


def final_run_curl():
    MASTER_ALLELES_FILE=DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    master_alleles_df = pd.read_csv(MASTER_ALLELES_FILE, sep='\t', index_col=False)

    invalid_alleles = []
    for idx, row in master_alleles_df.iterrows():
        toolname = row['Tool Label']

        # Define the curl command
        curl_command = [
            'curl',
            '--data', f'method=netmhcpan_el&sequence_text=SLYNTVATLYCVHQRIDV&allele={toolname}&length=9',
            'http://localhost/tools_api/mhci/'
        ]

        # Run the curl command
        result = subprocess.run(curl_command, capture_output=True, text=True)

        # Print the output
        print(result.stdout)
        
        if 'Invalid allele name' in result.stdout:
            invalid_alleles.append(toolname)

        print(f'{idx}/{len(master_alleles_df)}')

    print("------------")
    print(invalid_alleles)

def check_mro_ids_for_ones_with_no_iedb_label():
    MASTER_ALLELES_FILE=DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    master_alleles_df = pd.read_csv(MASTER_ALLELES_FILE, sep='\t', index_col=False)

    df_with_nan = master_alleles_df[master_alleles_df['IEDB Label'].isna()]

    # labels_with_unknown_mroid = df_with_nan['Tool Label'].to_list()

    # for label in labels_with_unknown_mroid:
    #     target_df = original_tm_df[original_tm_df['term']==label].iloc[0]
    #     print(label, target_df['MRO ID'])
    
    OUTPUT_FILE=DATA_DIR + '/alleles_with_no_mroid.tsv'
    df_with_nan.to_csv(OUTPUT_FILE, sep='\t', index=False)

def remove_missing_mroid():
    MASTER_ALLELES_FILE=DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    master_alleles_df = pd.read_csv(MASTER_ALLELES_FILE, sep='\t', index_col=False)

    df_without_nan = master_alleles_df[(master_alleles_df['IEDB Label'].notna())]
    
    MASTER_ALLELES_FILE=DATA_DIR + '/master_netmhcpan_4_1_alleles.tsv'
    df_without_nan.to_csv(MASTER_ALLELES_FILE, sep='\t', index=False)


if __name__ == '__main__':
    # compare_allele_files()
    # count_list()
    # create_netmhcpan_allele_file()

    # compare_allelenames()

    # compare_allele_files2()
    # double_check_invalid_alleles()

    # extra_analysis()

    # test_alleles()

    # update_unknown_mhc()

    # update_tools_mapping_file()

    # LATEST ONE: 9/4
    # test1()

    '''
    These steps will create the truly finalized unknown allele file
    that was checked from the 'model_list.txt' file.
    '''
    # analyze_model_list_file()
    # double_check_unknown_model_list_alleles()
    # triple_check_unknown_model_list_alleles()

    '''
    Create valid alleles file by taking the model_list.txt and removing
    all the unknown alleles (9 alleles) from the 'unknown_mhc_alleles_from_model_list_FINAL.txt'
    '''
    # create_valid_alleles_from_model_list_file()

    '''
    Compile all the unique alleles from 'allelenames', 'MHC_pseudo.dat', 'MHC_allele_names.txt',
    and 'model_list.txt'
    '''
    # compile_all_alleles()

    # handle_rest_of_the_alleles()

    # test1()

    '''
    Let's check if all of these valid alleles 'from model_list.txt' file
    exist in the 'tools-mapping.tsv' 
    '''
    # check_existence_of_valid_alleles_from_model_list()

    '''
    Then let's see if there are extra netmhcpan alleles from 'tools-mapping.tsv'
    and see if these extra alleles need to be removed.
    '''
    # Not working for curl commands.
    

    # 9/7 progress
    # Combine 'verified_standalone_alleles.tsv' into the 'master'
    # add_verfied_standalone_alleles()

    #9/7
    # Retrieve all the MRO ID for the 72 unknown alleles.
    # identify_mroid_for_unknown_alleles()
    # Check which alleles from tools-mapping file has the MRO ID, and add this as a new column
    # find_corr_labels_using_mroid()
    # Check to make sure all the supported unique alleles exist in the master sheet
    # check_existence()

    # Final check on the tool label. Make sure the tool label works.
    # final_run_curl()

    # Save those labels with no mroids to a separate file.
    # check_mro_ids_for_ones_with_no_iedb_label()

    # Remove missing mroids from master file
    remove_missing_mroid()
